package com.xdja.pki.gmssl.sdf.yunhsm.utils;

import com.dj.SJJ1805.HsmSJJ1805APIIns;
import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.dongjin.DongJinSdfSDKUtils;
import com.xdja.pki.gmssl.dongjin.pool.HsmConnectionProviderImpl;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDKUtils;
import com.xdja.pki.gmssl.x509.utils.bean.HsmInfoEntry;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmExceptionEnum;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.file.Files;
import java.security.Security;

/**
 * @description: 东进密码机工具类
 * @author: feng zhen
 * @date: 2021/7/3 15:20
 **/
public class GMSSLDongJinHsmUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLDongJinHsmUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 需要安装python2.7 并 确保 PYTHON_PATH 下有对应的python文件
     */
    private static final String CONFIG_PATH = File.separator + "conf";
    private static final String CONFIG_TEST_PATH = File.separator + "confTest";
    private static final String HSM_CONFIG_PATH = File.separator + "conf" + File.separator + "hsm";
    private static String CONFIG_INI_NAME = "DJHsmAPI.ini";


    /**
     * 配置密码机IP，Port 测试密码机连通性并存储 （下一步） ITSCA初始化服务调用
     *
     * @param tomcatPath    tomcat服务地址
     * @param ip            密码机服务ip地址
     * @param port          密码机服务端口
     * @param isOtherTomcat 是否为初始化其他Tomcat
     * @return 测试状态
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String tomcatPath, String ip, int port, boolean isOtherTomcat) {
        try {
            String confPath = tomcatPath + HSM_CONFIG_PATH + CONFIG_PATH;
            YunHsmExceptionEnum yunHsmExceptionEnum = testConnect(ip, port);
            if (yunHsmExceptionEnum != YunHsmExceptionEnum.NORMAL) {
                return yunHsmExceptionEnum;
            }
            GMSSLIniFileUtils.createDongJinHsmIni(confPath, ip, port);
            if (isOtherTomcat) {
                //为自己配置一份东进密码机的配置文件
                String thisConfigPath = getConfPath(false);
                GMSSLIniFileUtils.createDongJinHsmIni(thisConfigPath, ip, port);
            }
            GMSSLPkiCryptoInit.getDJHsmInstance();
            HsmConnectionProviderImpl.getInstance().reopen();
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("配置密码机测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }


    /**
     * 配置密码机IP，Port 测试密码机连通性并存储 （下一步）
     *
     * @param ip   密码机服务ip地址
     * @param port 密码机服务端口
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String ip, int port) {
        String tomcatPath = getTomcatPath();
        return initYunHsmConfigAndTestConnect(tomcatPath, ip, port, false);
    }

    /**
     * 配置密码机IP，Port 测试密码机连通性
     *
     * @param ip   密码机服务ip地址
     * @param port 密码机服务端口
     */
    public static YunHsmExceptionEnum testConnect(String ip, int port) {
        try {
            HsmSJJ1805APIIns singleConInstance = HsmSJJ1805APIIns.getSingleConInstance(ip, port);
            if (!singleConInstance.isconnectedHsm()) {
                logger.error("测试连通性失败");
                return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
            } else {
                singleConInstance.releaseSingleConInstance();
            }
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }


    /**
     * 调用现有的配置文件测试连通性
     */
    public static YunHsmExceptionEnum testConnect() {
        try {
            HsmInfoEntry configInfo = getConfigInfo();
            return testConnect(configInfo.getIp(), configInfo.getPort());
        } catch (Exception e) {
            logger.error("测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }

    /**
     * 获取东进密码机配置文件信息
     *
     * @return 密码机信息
     */
    public static HsmInfoEntry getConfigInfo() {
        String iniPath = getConfPath(false) + File.separator + CONFIG_INI_NAME;
        String ip = GMSSLIniFileUtils.getProfileValue(iniPath, "EncryptorGroup_1_1", "SrvIPAddr");
        String port = GMSSLIniFileUtils.getProfileValue(iniPath, "EncryptorGroup_1_1", "SrvPort");
        HsmInfoEntry hsmInfoEntry = new HsmInfoEntry();
        hsmInfoEntry.setIp(ip);
        hsmInfoEntry.setPort(Integer.valueOf(port));
        return hsmInfoEntry;
    }


    /**
     * 获取第三方密码机配置文件路径
     *
     * @param isTest 是否为test路径
     * @return
     */
    public static String getConfPath(boolean isTest) {
        String path;
        String confPath = getBaseConfPath();
        if (isTest) {
            path = confPath + CONFIG_TEST_PATH;
        } else {
            path = confPath + CONFIG_PATH;
        }
        return path;
    }

    /**
     * 获取本服务的Tomcat路径
     *
     * @return Tomcat路径
     */
    private static String getTomcatPath() {
        String tomcatPath = System.getProperty("catalina.home");
        if (null == tomcatPath || tomcatPath.contains("Temp")) {
            tomcatPath = "/home/xdja/dongjin";
        }
        return tomcatPath;
    }


    /**
     * 获取密码机配置文件Base路径
     *
     * @return 密码机配置文件Base路径
     */
    private static String getBaseConfPath() {
        return getTomcatPath() + HSM_CONFIG_PATH;
    }

}
