package com.xdja.pki.gmssl.sdf.yunhsm.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDKUtils;
import com.xdja.pki.gmssl.sdf.yunhsm.pool.HsmConnectionProviderImpl;
import com.xdja.pki.gmssl.x509.utils.bean.HsmInfoEntry;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmExceptionEnum;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.file.Files;
import java.security.Security;

/**
 * @description: 密码机操作相关类
 * @author: feng zhen
 * @date: 2021/6/18 15:12
 **/
public class GMSSLHsmUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLHsmUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 需要安装python2.7 并 确保 PYTHON_PATH 下有对应的python文件
     */
    private static final String PYTHON_PATH = "/usr/local/xdsdfjni/parseconf.py";
    private static final String CONFIG_PATH = File.separator + "conf";
    private static final String CONFIG_TEST_PATH = File.separator + "confTest";
    private static final String CONFIG_BAK_PATH = File.separator + "confBak";
    private static final String HSM_CONFIG_PATH = File.separator + "conf" + File.separator + "hsm";
    private static String JSON_PATH = File.separator + "etc" + File.separator + "xdsdfjni.json";

    /**
     * 配置密码机IP，Port 测试密码机连通性并存储 （下一步）
     *
     * @param ip   密码机服务ip地址
     * @param port 密码机服务端口
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String ip, int port, String hsmModel) {
        try {
            GMSSLPkiCryptoInit.getHsmInstance();
            String confPath = getBaseConfPath();
            File file = new File(confPath);
            if (!file.exists()) {
                Files.createDirectories(file.toPath());
            }
            //先测试连接
            YunHsmExceptionEnum yunHsmExceptionEnum = testConnect(ip, port, hsmModel);
            if (yunHsmExceptionEnum != YunHsmExceptionEnum.NORMAL) {
                return yunHsmExceptionEnum;
            }
            //备份配置文件
            bakConfigFile(confPath);
            try {
                //先添加路径
                addPathByPython(confPath, false);
                //更改参数
                changeConfigByPython(confPath, ip, port, hsmModel, false);
                //测试连通性
                YunhsmSdfSDKUtils.testConnection(confPath + CONFIG_PATH + JSON_PATH, true);
                //重新开启链接
                HsmConnectionProviderImpl.getInstance().reopen();
                //返回结果
                return YunHsmExceptionEnum.NORMAL;
            } catch (Exception e) {
                logger.error("使用新的配置文件测试连通性失败，", e);
                //不通的话 重新返回原来的文件
                resetHsm(confPath);
            }
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        } catch (Exception e) {
            logger.error("配置密码机测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }

    /**
     * 配置密码机IP，Port 测试密码机连通性
     *
     * @param ip   密码机服务ip地址
     * @param port 密码机服务端口
     */
    public static YunHsmExceptionEnum testConnect(String ip, int port, String hsmModel) {
        try {
            GMSSLPkiCryptoInit.getHsmInstance();
            String confPath = getBaseConfPath();
            //删除配置文件目录
            GMSSLFileUtils.deleteDirectory(confPath + File.separator + CONFIG_TEST_PATH);
            //放置配置文件路径 Test目录
            addPathByPython(confPath, true);
            //修改参数
            changeConfigByPython(confPath, ip, port, hsmModel, true);
            //测试连通性
            YunhsmSdfSDKUtils.testConnection(confPath + CONFIG_TEST_PATH + JSON_PATH, true);
            //返回结果
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }

    /**
     * 调用现有的配置文件测试连通性
     */
    public static YunHsmExceptionEnum testConnect() {
        try {
            GMSSLPkiCryptoInit.getHsmInstance();
            String confPath = getBaseConfPath();
            YunhsmSdfSDKUtils.testConnection(confPath + CONFIG_PATH + JSON_PATH, true);
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("测试连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }

    }


    /**
     * 获取第三方密码机信息
     *
     * @return 设备信息
     */
    public static HsmInfoEntry getConfigInfo() throws Exception {
        String sh = "python2.7 " + PYTHON_PATH + " 4 " + getConfPath(false);
        logger.debug("get env sh info is : {}", sh);
        String pythonPrint = runShAndGetPrint(sh);
        String[] s = pythonPrint.split(" ");
        HsmInfoEntry hsmInfoEntry = new HsmInfoEntry();
        hsmInfoEntry.setIp(s[1]);
        hsmInfoEntry.setPort(Integer.valueOf(s[2]));
        return hsmInfoEntry;
    }


    /**
     * 获取第三方密码机配置文件路径
     *
     * @param isTest 是否为test路径
     * @return
     */
    public static String getConfPath(boolean isTest) {
        String path;
        String confPath = getBaseConfPath();
        if (isTest) {
            path = confPath + CONFIG_TEST_PATH;
        } else {
            path = confPath + CONFIG_PATH;
        }
        return path;
    }

    /**
     * 获取密码机配置文件Base路径
     *
     * @return 密码机配置文件Base路径
     */
    private static String getBaseConfPath() {
        String tomcatPath = System.getProperty("catalina.home");
        if (null == tomcatPath || tomcatPath.contains("Temp")) {
            tomcatPath = "/home/xdja/aliyun";
        }
        return tomcatPath + HSM_CONFIG_PATH;
    }

    /**
     * 测试连通性失败，恢复配置文
     *
     * @param confPath 配置文件路径
     * @throws Exception
     */
    private static void resetHsm(String confPath) throws Exception {
        File file = new File(confPath + CONFIG_PATH);
        if (file.isDirectory() && file.exists()) {
            GMSSLFileUtils.deleteDirectory(confPath + CONFIG_PATH);
            GMSSLFileUtils.copyDir(confPath + CONFIG_BAK_PATH, confPath + CONFIG_PATH);
        }
    }

    /**
     * 备份配置文件
     *
     * @param confPath 配置文件路径
     * @throws Exception
     */
    private static void bakConfigFile(String confPath) throws Exception {
        File file = new File(confPath + CONFIG_PATH);
        if (file.isDirectory() && file.exists()) {
            //备份
            GMSSLFileUtils.copyDir(confPath + CONFIG_PATH, confPath + CONFIG_BAK_PATH);
        }
    }

    /**
     * 修改配置文件路径 通过密码机配置文件
     * type = 3时，为修改环境接口，执行命令：
     * python2.7 parseconf.py 3 path hsmtype ip port
     * 其中，path为形如/usr/local/xdsdfjni/的目录，hsmtype为形如SJJ1528的密码机类型。
     *
     * @param confPath 配置文件路径
     * @param ip       ip
     * @param port     端口
     */
    private static void changeConfigByPython(String confPath, String ip, int port, String hsmModel, boolean isTestConf) {
        String path;
        if (isTestConf) {
            path = confPath + File.separator + CONFIG_TEST_PATH;
        } else {
            path = confPath + File.separator + CONFIG_PATH;
        }
        String sh = "python2.7  " + PYTHON_PATH + " 3 " + path + " " + hsmModel + " " + ip + " " + port;
        logger.debug("change env sh info is : {}", sh);
        run(sh);
    }

    /**
     * 添加Path路径 ini路径等 python脚本固定
     * tyep = 1时，为新增环境接口，执行命令：
     * python2.7 parseconf.py 1 add_path
     * 其中，add_path为形如/usr/local/xdsdfjni/的目录
     *
     * @param confPath   配置文件路径
     * @param isTestConf
     */
    private static void addPathByPython(String confPath, boolean isTestConf) throws IOException {
        String path;
        //配置文件末尾必须加/
        if (isTestConf) {
            path = confPath + File.separator + CONFIG_TEST_PATH + File.separator;
        } else {
            path = confPath + File.separator + CONFIG_PATH + File.separator;
        }
        File file = new File(path);
        if (!file.exists()) {
            Files.createDirectories(file.toPath());
        }
        String sh = "python2.7  " + PYTHON_PATH + " 1 " + path;
        logger.debug("add env sh info is : {}", sh);
        run(sh);
    }


    /**
     * 执行脚本命令
     *
     * @param sh 脚本命令
     */
    private static boolean run(String sh) {
        Runtime rt = Runtime.getRuntime();
        Process ps = null;
        try {
            ps = rt.exec(sh);
            InputStream is = ps.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            ps.waitFor();
            is.close();
            reader.close();
            ps.destroy();
            logger.debug("=================完成=================");
            return true;
        } catch (Exception e) {
            logger.error(sh + " run is error ", e);
            return false;
        }
    }

    /**
     * 执行脚本并获取打印内容
     *
     * @param sh 脚本命令
     * @return 脚本打印的数据
     * @throws Exception
     */
    private static String runShAndGetPrint(String sh) throws Exception {
        Runtime rt = Runtime.getRuntime();
        Process ps = null;
        try {
            ps = rt.exec(sh);
            InputStream is = ps.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            ps.waitFor();
            LineNumberReader input = new LineNumberReader(reader);
            String line;
            String str = "";
            while ((line = input.readLine()) != null) {
                str = str + line;
            }
            is.close();
            reader.close();
            ps.destroy();
            logger.debug("=================完成=================");
            return str;
        } catch (Exception e) {
            logger.error(sh + " run is error ", e);
            throw new Exception(e);
        }
    }

}
