package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2EncryptUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.crmf.EncryptedValue;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.security.*;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;

public class GMSSLCMPUtils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLCMPUtils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 通过 BC 构造 EncryptedValue 结构体
     *
     * @param certificate 证书
     * @return
     * @throws CertificateEncodingException
     * @throws IOException
     * @throws CryptoException
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws InvalidAlgorithmParameterException
     */
    public static EncryptedValue generateEncryptedValueByBC(X509Certificate certificate) throws CertificateEncodingException, IOException, CryptoException, NoSuchPaddingException, NoSuchAlgorithmException, IllegalBlockSizeException, BadPaddingException, NoSuchProviderException, InvalidKeyException, InvalidAlgorithmParameterException {
//        EncryptedValueBuilder build = new EncryptedValueBuilder(
//                new JceAsymmetricKeyWrapper(certificate.getPublicKey()).setProvider(BouncyCastleProvider.PROVIDER_NAME),
//                //will throw CRMFException
//                new JceCRMFEncryptorBuilder(GMObjectIdentifiers.sms4_cbc).setProvider(BouncyCastleProvider.PROVIDER_NAME).build()
//        );
//        //will throw CertificateEncodingException, IOException
//        X509CertificateHolder x509CertificateHolder = new X509CertificateHolder(certificate.getEncoded());
//        EncryptedValue value = build.build(x509CertificateHolder);
//        return value;

        //参考 org.bouncycastle.cert.crmf.EncryptedValueBuilder
        //私钥时为私钥算法，证书时为空
        AlgorithmIdentifier intendedAlg = null;
        //用于加密值的对称算法 SM4
        AlgorithmIdentifier symmAlg = new AlgorithmIdentifier(GMObjectIdentifiers.sms4_cbc);
        byte[] key = new SecureRandom().generateSeed(16);
        //will throw SdfSDKException IOException
        byte[] encSymmKeyText = generateEncSymmKeyByBC((ECPublicKey) certificate.getPublicKey(), key);
        DERBitString encSymmKey = new DERBitString(encSymmKeyText);
        AlgorithmIdentifier keyAlg = new AlgorithmIdentifier(GMObjectIdentifiers.sm2encrypt_with_sm3);
        //私钥时为空，证书时为空
        ASN1OctetString valueHint = null;
        //will throw CertificateEncodingException
        byte[] plainText = certificate.getEncoded();
        //will throw SdfSDKException
        byte[] encValueText = generateEncValueByBC(key, plainText);
//        GMSSLByteArrayUtils.printHexBinary(logger, "encValueText", encValueText);
        DERBitString encValue = new DERBitString(encValueText);
        return new EncryptedValue(intendedAlg, symmAlg, encSymmKey, keyAlg, valueHint, encValue);
    }

    /**
     * 生成加密的会话密钥 并输出
     *
     * @param ecPublicKey 加密公钥
     * @return 加密的会话密钥
     * @throws SdfSDKException SDF 访问异常
     * @throws IOException     IO 异常
     */
    public static byte[] generateEncSymmKeyByBC(ECPublicKey ecPublicKey, byte[] key) throws IOException, CryptoException {
        return GMSSLSM2EncryptUtils.encryptASN1ByBC(ecPublicKey, key);
    }

    /**
     * 使用加密的会话密钥加密数据 并输出
     *
     * @param keyBytes  会话密钥
     * @param plainText 明文
     * @return 密文
     * @throws SdfSDKException SDF 访问异常
     */
    public static byte[] generateEncValueByBC(
            byte[] keyBytes,
            byte[] plainText
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException, InvalidAlgorithmParameterException {
        Cipher cipher = Cipher.getInstance("SM4/CBC/PKCS7Padding", BouncyCastleProvider.PROVIDER_NAME);
        Key key = new SecretKeySpec(keyBytes, "SM4");
        byte[] iv = new byte[16];
        cipher.init(Cipher.ENCRYPT_MODE, key, new IvParameterSpec(iv));
        byte[] encrypt = cipher.doFinal(plainText);
        return encrypt;
    }

    /**
     * 通过密码机
     * 生成 CMP 协议中需要的 EncryptedValue 结构，使用 sms4_cbc 作为对称加密算法 使用 sm2encrypt_with_sm3 作为非对称加密算法
     *
     * @param certificate 要加密的证书
     * @return GM/T 0014-2012 P37 B.4 证书申请协议 EncryptedValue ASN.1 结构体
     * @throws SdfSDKException              SDF 访问异常
     * @throws IOException                  IO 异常
     * @throws CertificateEncodingException 证书转码异常
     */
    public static EncryptedValue generateEncryptedValueByYunhsm(X509Certificate certificate) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return generateEncryptedValueByBC(certificate);
        }
        return generateEncryptedValueBySdf(SdfCryptoType.YUNHSM, certificate);
    }

    /**
     * 通过 PCIE 卡
     * 生成 CMP 协议中需要的 EncryptedValue 结构，使用 sms4_cbc 作为对称加密算法 使用 sm2encrypt_with_sm3 作为非对称加密算法
     *
     * @param certificate 要加密的证书
     * @return GM/T 0014-2012 P37 B.4 证书申请协议 EncryptedValue ASN.1 结构体
     * @throws SdfSDKException              SDF 访问异常
     * @throws IOException                  IO 异常
     * @throws CertificateEncodingException 证书转码异常
     */
    public static EncryptedValue generateEncryptedValueByPcie(X509Certificate certificate) throws Exception {
        return generateEncryptedValueBySdf(SdfCryptoType.PCIE, certificate);
    }

    /**
     * 生成 CMP 协议中需要的 EncryptedValue 结构，使用 sms4_cbc 作为对称加密算法 使用 sm2encrypt_with_sm3 作为非对称加密算法
     *
     * @param sdfCryptoType SDF 密码算法类型 密码机、PCIE
     * @param certificate   要加密的证书
     * @return GM/T 0014-2012 P37 B.4 证书申请协议 EncryptedValue ASN.1 结构体
     * @throws SdfSDKException              SDF 访问异常
     * @throws IOException                  IO 异常
     * @throws CertificateEncodingException 证书转码异常
     */
    public static EncryptedValue generateEncryptedValueBySdf(SdfCryptoType sdfCryptoType, X509Certificate certificate) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return generateEncryptedValueByBC(certificate);
        }
        //参考 org.bouncycastle.cert.crmf.EncryptedValueBuilder
        //私钥时为私钥算法，证书时为空
        AlgorithmIdentifier intendedAlg = null;
        //用于加密值的对称算法 SM4
        AlgorithmIdentifier symmAlg = new AlgorithmIdentifier(GMObjectIdentifiers.sms4_cbc);

        //will throw CertificateEncodingException
        byte[] plainText = certificate.getEncoded();

        //will throw SdfSDKException IOException
        SdfSymmetricCipher sdfSymmetric = new SdfSymmetricCipher(sdfCryptoType);
        CipherParameters param = new SdfSymmetricKeyParameters(
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                SdfAlgIdSymmetric.SGD_SM4_CBC,
                (ECPublicKey) certificate.getPublicKey()
        );
        sdfSymmetric.init(true, param);
        byte[] encValueText = sdfSymmetric.doFinal(plainText);
        sdfSymmetric.release();

        DERBitString encSymmKey = new DERBitString(sdfSymmetric.getKey());
        AlgorithmIdentifier keyAlg = new AlgorithmIdentifier(GMObjectIdentifiers.sm2encrypt_with_sm3);
        //私钥时为空，证书时为空
        ASN1OctetString valueHint = null;
//        GMSSLByteArrayUtils.printHexBinary(logger, "encValueText", encValueText);
        DERBitString encValue = new DERBitString(encValueText);
        return new EncryptedValue(intendedAlg, symmAlg, encSymmKey, keyAlg, valueHint, encValue);
    }

    public static X509Certificate decodeEncryptedValueByBC(PrivateKey privateKey, byte[] encryptedValueText) throws CertificateException, NoSuchPaddingException, BadPaddingException, NoSuchAlgorithmException, IOException, InvalidCipherTextException, IllegalBlockSizeException, NoSuchProviderException, InvalidKeyException, InvalidAlgorithmParameterException {
        EncryptedValue value = EncryptedValue.getInstance(encryptedValueText);
//        EncryptedValueParser parser = new EncryptedValueParser(value, null);
//        ValueDecryptorGenerator decGen = new JceAsymmetricValueDecryptorGenerator(privateKey).setProvider(BouncyCastleProvider.PROVIDER_NAME);

//        X509CertificateHolder holder = parser.readCertificateHolder(decGen);
//        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
        EncryptedValue encryptedValue = EncryptedValue.getInstance(encryptedValueText);
        //must be GMObjectIdentifiers.sms4_cbc
        AlgorithmIdentifier symmAlg = encryptedValue.getSymmAlg();
        if (!symmAlg.getAlgorithm().getId().equals(GMObjectIdentifiers.sms4_cbc.getId())) {
            throw new IOException("unSupport algorithm identifier " + symmAlg.getAlgorithm());
        }
        //GMObjectIdentifiers.sm2encrypt_with_sm3
        AlgorithmIdentifier keyAlg = encryptedValue.getKeyAlg();
        if (!keyAlg.getAlgorithm().getId().equals(GMObjectIdentifiers.sm2encrypt_with_sm3.getId())) {
            throw new IOException("unSupport algorithm identifier " + keyAlg.getAlgorithm());
        }
        byte[] encSymmKeyText = encryptedValue.getEncSymmKey().getOctets();
        //will throw IOException
        byte[] cipherText = encryptedValue.getEncValue().getOctets();
//        GMSSLByteArrayUtils.printHexBinary(logger, "cipherText", cipherText);
        byte[] plainText = decodeEncValueByBC(privateKey, encSymmKeyText, cipherText);
//        GMSSLByteArrayUtils.printHexBinary(logger, "plainText", plainText);
        return GMSSLX509Utils.readCertificateFromCerByte(plainText);
    }

    public static byte[] decodeEncValueByBC(
            PrivateKey privateKey,
            byte[] encSymmKeyText,
            byte[] cipherText
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, IOException, InvalidCipherTextException, BadPaddingException, IllegalBlockSizeException, InvalidAlgorithmParameterException {
        Cipher cipher = Cipher.getInstance("SM4/CBC/PKCS7Padding", BouncyCastleProvider.PROVIDER_NAME);
        byte[] keyBytes = GMSSLSM2EncryptUtils.decryptASN1ByBC(privateKey, encSymmKeyText);
//        GMSSLByteArrayUtils.printHexBinary(logger, "decrypt key", keyBytes);
        Key key = new SecretKeySpec(keyBytes, "SM4");
        byte[] iv = new byte[16];
        cipher.init(Cipher.DECRYPT_MODE, key, new IvParameterSpec(iv));
        return cipher.doFinal(cipherText);
    }

    public static X509Certificate decodeEncryptedValueByYunhsm(int privateKeyIndex, String privateKeyPassword, byte[] encryptedValueText) throws Exception {
        return decodeEncryptedValue(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, encryptedValueText);
    }

    public static X509Certificate decodeEncryptedValueByPcie(
            int privateKeyIndex, String privateKeyPassword,
            byte[] encryptedValueText
    ) throws Exception {
        return decodeEncryptedValue(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, encryptedValueText);
    }

    public static X509Certificate decodeEncryptedValue(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] encryptedValueText)
            throws Exception {

        EncryptedValue encryptedValue = EncryptedValue.getInstance(encryptedValueText);
        //must be GMObjectIdentifiers.sms4_cbc
        AlgorithmIdentifier symmAlg = encryptedValue.getSymmAlg();
        if (!symmAlg.getAlgorithm().getId().equals(GMObjectIdentifiers.sms4_cbc.getId())) {
            throw new SdfSDKException("unSupport algorithm identifier " + symmAlg.getAlgorithm());
        }
        //GMObjectIdentifiers.sm2encrypt_with_sm3
        AlgorithmIdentifier keyAlg = encryptedValue.getKeyAlg();
        if (!keyAlg.getAlgorithm().getId().equals(GMObjectIdentifiers.sm2encrypt_with_sm3.getId())) {
            throw new SdfSDKException("unSupport algorithm identifier " + keyAlg.getAlgorithm());
        }
        byte[] encSymmKeyText = encryptedValue.getEncSymmKey().getOctets();
        //will throw IOException
        byte[] cipherText = encryptedValue.getEncValue().getOctets();
//        GMSSLByteArrayUtils.printHexBinary(logger, "cipherText", cipherText);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        byte[] plainText = decodeEncValueBySdf(sdfCryptoType, sdfPrivateKey, encSymmKeyText, cipherText);
//        GMSSLByteArrayUtils.printHexBinary(logger, "plainText", plainText);
        return GMSSLX509Utils.readCertificateFromCerByte(plainText);
    }

    public static byte[] decodeEncValueBySdf(
            SdfCryptoType sdfCryptoType,
            SdfPrivateKey sdfPrivateKey,
            byte[] encSymmKeyText,
            byte[] cipherText
    ) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), true);
            return decodeEncValueByBC(bean.getPrivateKey(), encSymmKeyText, cipherText);
        }
        SdfSymmetricCipher sdfSymmetric = new SdfSymmetricCipher(sdfCryptoType);
        CipherParameters param = new SdfSymmetricKeyParameters(SdfSymmetricKeyParameters.KeyCipherType.ECC_CIPHER, SdfSymmetricKeyParameters.PaddingType.PKCS7Padding, SdfAlgIdSymmetric.SGD_SM4_CBC, sdfPrivateKey, encSymmKeyText);
        sdfSymmetric.init(false, param);
        byte[] plainText = sdfSymmetric.doFinal(cipherText);
        sdfSymmetric.release();
        return plainText;
    }
}
