package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import org.bouncycastle.crypto.BlockCipher;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.params.KeyParameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SdfSM1Engine implements BlockCipher {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final int BLOCK_SIZE = 16;

    private SdfSDK sdfSDK;

    private long[] phKeyHandle;
    private boolean forEncryption;

    public SdfSM1Engine() throws SdfSDKException {
        this(SdfCryptoType.YUNHSUM);
    }

    public SdfSM1Engine(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSM1Engine(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
    }

    @Override
    public void init(boolean forEncryption, CipherParameters params) throws IllegalArgumentException {
        byte[] key;
        if (params instanceof KeyParameter) {
            key = ((KeyParameter) params).getKey();
        }else {
            throw new IllegalArgumentException("invalid parameter passed to SM4 init - " + params.getClass().getName());
        }

        if (key.length != 16) {
            throw new IllegalArgumentException("SM4 requires a 128 bit key");
        }

        //导入明文会话密钥
        try {
            this.phKeyHandle = sdfSDK.importKey(key);
            this.forEncryption = forEncryption;
        } catch (SdfSDKException e) {
            logger.error("init import key", e);
            throw new IllegalArgumentException("SDF SM4 init - import key error");
        }
    }

    @Override
    public String getAlgorithmName() {
        return "SM1";
    }

    @Override
    public int getBlockSize() {
        return BLOCK_SIZE;
    }

    @Override
    public int processBlock(byte[] in, int inOff, byte[] out, int outOff) throws DataLengthException, IllegalStateException {
        GMSSLByteArrayUtils.printHexBinary(logger, "processBlock in", in);
        try {
            byte[] data = new byte[BLOCK_SIZE];
            System.arraycopy(in, inOff, data, 0, BLOCK_SIZE);
            GMSSLByteArrayUtils.printHexBinary(logger, "processBlock data", data);
            //这时 初始化向量为 00000000000000000000000000000000 16个 0
            byte[] iv = new byte[16];
            GMSSLByteArrayUtils.printHexBinary(logger, "processBlock iv", iv);
            if (this.forEncryption) {
                byte[] pucEncData = sdfSDK.encrypt(phKeyHandle, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);
                System.arraycopy(pucEncData, 0, out, outOff, BLOCK_SIZE);
            } else {
                byte[] d = sdfSDK.decrypt(phKeyHandle, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);
                System.arraycopy(d, 0, out, outOff, BLOCK_SIZE);
            }
        } catch (SdfSDKException e) {
            logger.error("processBlock", e);
            throw new DataLengthException(e.getMessage());
        }
        GMSSLByteArrayUtils.printHexBinary(logger, "processBlock out", out);

        return BLOCK_SIZE;
    }

    @Override
    public void reset() {

    }

    public void release() throws SdfSDKException {
        if (sdfSDK != null && phKeyHandle != null) {
            sdfSDK.destroyKey(phKeyHandle);
            sdfSDK.release();
        }
    }
}
