package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdHash;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import org.bouncycastle.crypto.ExtendedDigest;
import org.bouncycastle.util.Memoable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SdfSM3Digest implements ExtendedDigest, Memoable {

    private Logger logger = LoggerFactory.getLogger(this.getClass().getName() + "[" + this.getClass().hashCode()+ "]");

    private static final int DIGEST_LENGTH = 32;   // bytes
    private static final int BYTE_LENGTH = 64;   // bytes

    private SdfSDK sdfSDK;
    private byte[] pucID;
    private SdfECCPublicKey publicKey;

    private byte[] in = new byte[0];

    private volatile boolean isInit = false;

    public SdfSM3Digest() throws SdfSDKException {
        this(SdfCryptoType.YUNHSUM);
    }

    public SdfSM3Digest(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSM3Digest(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
        this.init();
    }

    public SdfSM3Digest(SdfSDK sdfSDK, byte[] pucID, SdfECCPublicKey publicKey) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
        this.init(pucID, publicKey);
    }

    public SdfSM3Digest(SdfCryptoType sdfCryptoType, byte[] pucID, SdfECCPublicKey publicKey) throws SdfSDKException {
        reset();
        this.sdfSDK = sdfCryptoType.getSdfSDK();
        this.sdfSDK.init();
        this.init(pucID, publicKey);
    }

    public SdfSM3Digest(SdfSM3Digest t) throws SdfSDKException {
        copyIn(t);
    }

    private void init() throws SdfSDKException {
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3);
        this.isInit = true;
    }

    private void init(byte[] pucID, SdfECCPublicKey publicKey) throws SdfSDKException {
        this.pucID = pucID;
        this.publicKey = publicKey;
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3, pucID, publicKey);
        this.isInit = true;
    }

    private void checkInit() throws SdfSDKException {
        if (!this.isInit) {
            //hash init
            if (this.publicKey != null && this.pucID != null) {
                this.init(this.pucID, this.publicKey);
            } else {
                this.init();
            }
        }
    }

    private void copyIn(SdfSM3Digest t) throws SdfSDKException {
        //sdf sdk copy
        this.sdfSDK = SdfCryptoType.cloneSdfSDK(t.sdfSDK);
        this.sdfSDK.init();
        this.publicKey = t.publicKey;
        this.pucID = t.pucID;
        this.checkInit();
        if (t.in != null){
            this.update(t.in, 0, t.in.length);
        }
    }

    @Override
    public String getAlgorithmName() {
        return "SM3";
    }

    @Override
    public int getDigestSize() {
        return DIGEST_LENGTH;
    }

    @Override
    public void update(byte in) {
        byte[] data = new byte[]{in};
        this.update(data, 0, data.length);
    }

    @Override
    public void update(byte[] in, int inOff, int len) {
        byte[] data = new byte[len];
        System.arraycopy(in, inOff, data, 0, len);
        byte[] out = new byte[this.in.length + len];
        System.arraycopy(this.in, 0, out, 0, this.in.length);
        System.arraycopy(data, 0, out, this.in.length, len);
        this.in = out;
        try {
            this.checkInit();
            sdfSDK.hashUpdate(data);
        } catch (SdfSDKException e) {
            logger.error("update " + in.length, e);
        }
    }

    @Override
    public int doFinal(byte[] out, int outOff) {
        try {
            byte[] res = sdfSDK.hashFinal();
            System.arraycopy(res, 0, out, outOff, res.length);
        } catch (SdfSDKException e) {
            logger.error("doFinal", e);
        } finally {
            this.isInit = false;
        }
        return DIGEST_LENGTH;
    }

    /**
     * reset the chaining variables
     */
    @Override
    public void reset() {

    }

    @Override
    public int getByteLength() {
        return BYTE_LENGTH;
    }


    public void release() throws SdfSDKException {
        if (sdfSDK != null) {
            sdfSDK.release();
        }
    }

    @Override
    public Memoable copy() {
        try {
            return new SdfSM3Digest(this);
        } catch (SdfSDKException e) {
            logger.error("Memoable copy", e);
            return null;
        }
    }

    @Override
    public void reset(Memoable other) {
        SdfSM3Digest d = (SdfSM3Digest) other;
        try {
            copyIn(d);
        } catch (SdfSDKException e) {
            logger.error("reset copy in error", e);
        }
    }
}
