package com.xdja.pki.gmssl.crypto.sdf.pcie.demo;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.pcie.PcieSdfSDK;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class PCryptoDemo {

    private static Logger logger = LoggerFactory.getLogger(PCryptoDemo.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static void main(String[] args) throws Exception {

        if (args.length == 0 || args[0] == null) {
            System.out.println("Please insert method, Example: all. random. sm2sign. sm2enc. sm4enc. sm3hash. sm3hmac. sm4mac");
            return;
        }

        System.out.println("You want to test: " + args[0]);

        PcieSdfSDK sdfSDK = new PcieSdfSDK();

        switch (args[0]) {
            case "all":
                System.out.println("*************** sm2SignDemo ***************");
                sm2SignDemo(sdfSDK);
                System.out.println("*************** sm2EncDemo ***************");
                sm2EncDemo(sdfSDK);
                System.out.println("*************** sm4EncDemo ***************");
                sm4EncDemo(sdfSDK);
                System.out.println("*************** sm3HashDemo ***************");
                sm3HashDemo(sdfSDK);
                System.out.println("*************** sm3HmacDemo ***************");
                sm3HmacDemo(sdfSDK);
                System.out.println("*************** sm4MacDemo ***************");
                sm4MacDemo(sdfSDK);
                break;
            case "random":
                random(sdfSDK);
                break;
            case "sm2sign":
                sm2SignDemo(sdfSDK);
                break;
            case "sm2enc":
                sm2EncDemo(sdfSDK);
                break;
            case "sm4enc":
                sm4EncDemo(sdfSDK);
                break;
            case "sm3hash":
                sm3HashDemo(sdfSDK);
                break;
            case "sm3hmac":
                sm3HmacDemo(sdfSDK);
                break;
            case "sm4mac":
                sm4MacDemo(sdfSDK);
                break;
            default:
                System.out.println("不支持的命令！");
                break;
        }
    }

    private static void random(PcieSdfSDK sdfSDK) throws Exception {
        SdfRandom pRandom = new SdfRandom(sdfSDK);

        byte[] out = pRandom.generateRandom(28);
        GMSSLByteArrayUtils.printHexBinary(logger,"random 28 ", out);

        byte[] bs = new byte[32];
        pRandom.nextBytes(bs);
        GMSSLByteArrayUtils.printHexBinary(logger,"random 32 ", bs);
    }

    /**
     * 导入SM2公私钥对 并进行 签名验签
     **/
    private static void sm2SignDemo(PcieSdfSDK sdfSDK) throws Exception {
        char[] pw = "63fbd873".toCharArray();
        String signP12 = "cert/admin14_sign_63fbd873.p12";

        //设备访问密码
        byte[] pwd = "xdja1234".getBytes();
        int sm2Index = 1;// sm2 签名、加密公私钥 可以用一个容器
        byte[] data = "12345678123456781234567812345678".getBytes();

//        ECPublicKey ecPublicKey = (ECPublicKey) PcieSdfSDKUtils.getPublicKeyFromP12(signP12, pw);
        ECPublicKey ecPublicKey = null;
        SdfECKeyParameters key = new SdfECKeyParameters(new SdfPrivateKey(sm2Index, pwd));

        SdfSM2Signer psm2Signer = new SdfSM2Signer(sdfSDK);
        psm2Signer.init(true, key);
        psm2Signer.update(data, 0, data.length);
        byte[] sign = psm2Signer.generateSignature();
        GMSSLByteArrayUtils.printHexBinary(logger,"psm2Signer.generateSignature()", sign);

        SdfSM2Signer psm2Verifier = new SdfSM2Signer(sdfSDK);
        psm2Verifier.init(false, key);
        psm2Verifier.update(data, 0, data.length);
        boolean isSign = psm2Verifier.verifySignature(sign);
        System.out.println("sm2 is sign verify " + isSign);
    }

    /**
     * 导入SM2公私钥对 并进行 加密解密操作
     **/
    private static void sm2EncDemo(PcieSdfSDK sdfSDK) throws Exception {
        char[] pw = "63fbd873".toCharArray();
        String encP12 = "cert/admin14_enc_63fbd873.p12";

        //设备访问密码
        byte[] pwd = "xdja1234".getBytes();
        int sm2Index = 1;// sm2 签名、加密公私钥 可以用一个容器
        byte[] data = "1234567812345678123456781234567811".getBytes();

        System.out.println();
        System.out.println("start sm2 encrypt asn.1");
        System.out.println();

//        ECPublicKey ecPublicKey = (ECPublicKey) PcieSdfSDKUtils.getPublicKeyFromP12(encP12, pw);
        ECPublicKey ecPublicKey = null;
        SdfSM2Engine encryptEngine = new SdfSM2Engine(sdfSDK);
        SdfECKeyParameters encrytKey = new SdfECKeyParameters(ecPublicKey);
        encryptEngine.init(true, encrytKey);
        byte[] cipherASN1 = encryptEngine.encryptASN1(data);
        GMSSLByteArrayUtils.printHexBinary(logger,"encryptEngine.encryptASN1", cipherASN1);

        System.out.println();
        System.out.println("start sm2 encrypt asn.1");
        System.out.println();

        SdfSM2Engine decryptEngine = new SdfSM2Engine(sdfSDK);
        SdfECKeyParameters decryptKey = new SdfECKeyParameters(new SdfPrivateKey(sm2Index, pwd));
        decryptEngine.init(false, decryptKey);
        byte[] dASN1 = decryptEngine.decryptASN1(cipherASN1);
        GMSSLByteArrayUtils.printHexBinary(logger,"decryptEngine.decryptASN1", dASN1);
        GMSSLByteArrayUtils.printHexBinary(logger,"data", data);

        System.out.println();
        System.out.println("start sm2 encrypt not asn.1");
        System.out.println();

        byte[] cipher = encryptEngine.encrypt(data);
        GMSSLByteArrayUtils.printHexBinary(logger,"encryptEngine.encrypt", cipher);

        System.out.println();
        System.out.println("start sm2 decrypt not asn.1");
        System.out.println();

        byte[] d = decryptEngine.decrypt(cipher);
        GMSSLByteArrayUtils.printHexBinary(logger,"decryptEngine.decrypt", d);
        GMSSLByteArrayUtils.printHexBinary(logger,"data", data);
    }

    /**
     * SM4 加密解密 导入会话密钥 销毁会话密钥
     **/
    public static void sm4EncDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        //SM4
        byte[] key = "1234567812345678".getBytes();
        byte[] data = "12345678123456781234567812345678".getBytes();
        System.out.println(data.length);

        //CBC模式加密	SDF_Encrypt
        KeyParameter keyParameter = new KeyParameter(key, 0, 16);
        SdfSM4Engine encrypt = new SdfSM4Engine();
        encrypt.init(true, keyParameter);
        byte[] eOut = new byte[data.length];
        encrypt.processBlock(data, 0, eOut, 0);
        GMSSLByteArrayUtils.printHexBinary(logger,"encrypt.processBlock", eOut);
        //销毁会话密钥	SDF_DestroyKey
        encrypt.reset();

        SdfSM4Engine decrypt = new SdfSM4Engine();
        decrypt.init(false, keyParameter);
        byte[] dOut = new byte[data.length];
        decrypt.processBlock(data, 0, dOut, 0);
        GMSSLByteArrayUtils.printHexBinary(logger,"decrypt.processBlock", dOut);
        //销毁会话密钥	SDF_DestroyKey
        decrypt.reset();
    }

    public static void sm3HashDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        byte[] data = "12345678123456781234567812345678".getBytes();
        SdfSM3Digest digest = new SdfSM3Digest();
        digest.update(data, 0, data.length);
        byte[] out = new byte[digest.getDigestSize()];
        digest.doFinal(out, 0);
        GMSSLByteArrayUtils.printHexBinary(logger,"digest.doFinal", out);
    }

    public static void sm3HmacDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
//        MAC计算	SDF_CalculateMAC
        byte[] data = "12345678123456781234567812345678".getBytes();
        byte[] key = "12345678123456781234567812345678".getBytes();
        byte[] mac = sdfSDK.sm3Hmac(data, key);
        GMSSLByteArrayUtils.printHexBinary(logger,"sdfSDK.sm3Hmac", mac);
    }

    public static void sm4MacDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
//        MAC计算	SDF_CalculateMAC
        sdfSDK.calculateMac(1);
    }

}
