package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.Signer;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.operator.RuntimeOperatorException;

import java.io.OutputStream;
import java.security.SignatureException;

public class SdfContentOutputStream extends OutputStream {

    private SdfSigner signer;

    SdfContentOutputStream(SdfCryptoType sdfCryptoType, AlgorithmIdentifier algorithmIdentifier, boolean forSigning, AsymmetricKeyParameter asymmetricKeyParameter) {
        try {
            if (SdfSHAType.checkSupportType(algorithmIdentifier.getAlgorithm())) {
                this.signer = new SdfRSASigner(sdfCryptoType, algorithmIdentifier);
            } else if (GMObjectIdentifiers.sm2sign_with_sm3.getId().equals(algorithmIdentifier.getAlgorithm().getId())) {
                this.signer = new SdfSM2Signer(sdfCryptoType);
            } else {
                throw new SdfSDKException("UnSupport algorithmIdentifier: " + algorithmIdentifier.getAlgorithm().getId());
            }
            this.signer.init(forSigning, asymmetricKeyParameter);
        } catch (SdfSDKException e) {
            throw new RuntimeOperatorException("exception obtaining signature init : " + e.getMessage(), e);
        }
    }

    @Override
    public void write(byte[] bytes, int off, int len) {
        signer.update(bytes, off, len);
    }

    @Override
    public void write(byte[] bytes) {
        signer.update(bytes, 0, bytes.length);
    }

    @Override
    public void write(int b) {
        signer.update((byte) b);
    }

    byte[] getSignature() throws SignatureException {
        try {
            byte[] signature = signer.generateSignature();
            signer.release();
            return signature;
        } catch (CryptoException | SdfSDKException e) {
            throw new SignatureException("exception obtaining signature: " + e.getMessage(), e);
        }
    }

    boolean verify(byte[] expected) throws SignatureException {
        try {
            boolean verify = signer.verifySignature(expected);
            signer.release();
            return verify;
        } catch (SdfSDKException e) {
            throw new SignatureException("exception obtaining signature: " + e.getMessage(), e);
        }
    }
}