package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.asn1.crypto.ASN1SM2Cipher;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCCipher;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.PublicKey;

public class SdfSM2Engine {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private SdfECKeyParameters ecKey;
    private boolean forEncryption;
    private SdfSDK sdfSDK;

    public SdfSM2Engine() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfSM2Engine(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSM2Engine(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
    }

    public void init(boolean forEncryption, CipherParameters param) {
        this.forEncryption = forEncryption;
        this.ecKey = (SdfECKeyParameters) param;
    }

    public PublicKey exportPublicKey() throws Exception {
        SdfECCPublicKey sdfECCPublicKey = this.sdfSDK.exportEncPublicKeyEcc(this.ecKey.getSm2Index());
        return GMSSLX509Utils.convertSM2PublicKey(sdfECCPublicKey.getX(), sdfECCPublicKey.getY());
    }

    public byte[] encryptASN1(byte[] data) {
        //SM2加密
        try {
            SdfECCPublicKey sdfECCPublicKey = this.ecKey.getSDFECCPublicKey();
            SdfECCCipher eccCipher = this.sdfSDK.externalEncryptECC(sdfECCPublicKey, data);
            ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(eccCipher.getX(), eccCipher.getY(), eccCipher.getM(), eccCipher.getC());
            byte[] out = asn1SM2Cipher.toASN1Primitive().getEncoded();
            GMSSLByteArrayUtils.printHexBinary(logger, "encrypt", out);
            return out;
        } catch (SdfSDKException | IOException e) {
            logger.error("encrypt asn1", e);
            return new byte[0];
        }
    }

    public byte[] decryptASN1(byte[] cipher) {
        ASN1SM2Cipher sm2CipherASN1 = ASN1SM2Cipher.getInstance(cipher);
        try {
            assert sm2CipherASN1 != null;
            SdfECCCipher eccCipher = new SdfECCCipher(
                    sm2CipherASN1.getxCoordinate().toByteArray(),
                    sm2CipherASN1.getyCoordinate().toByteArray(),
                    sm2CipherASN1.getHash(),
                    sm2CipherASN1.getCipherText().length,
                    sm2CipherASN1.getCipherText()
            );
            //SM2解密
            byte[] out = sdfSDK.internalDecryptECC(ecKey.getSm2Index(), ecKey.getPassword(), sm2CipherASN1.getCipherText().length, eccCipher);
//            GMSSLByteArrayUtils.printHexBinary(logger, "encrypt", out);
            return out;
        } catch (SdfSDKException e) {
            logger.error("decrypt asn1", e);
            return new byte[0];
        }

    }

    public byte[] processBlockASN1(byte[] in, int inOff, int inLen) throws InvalidCipherTextException {
        byte[] data = new byte[inLen];
        System.arraycopy(in, inOff, data, 0, inLen);
        if (forEncryption) {
            return encryptASN1(data);
        } else {
            return decryptASN1(data);
        }
    }

    public byte[] processBlock(byte[] in, int inOff, int inLen) throws InvalidCipherTextException {
        byte[] data = new byte[inLen];
        System.arraycopy(in, inOff, data, 0, inLen);
        if (forEncryption) {
            return encrypt(data);
        } else {
            return decrypt(data);
        }
    }

    public byte[] encrypt(byte[] data) {
        //SM2加密
        try {
            SdfECCPublicKey sdfECCPublicKey = ecKey.getSDFECCPublicKey();

            SdfECCCipher eccCipher = sdfSDK.externalEncryptECC(sdfECCPublicKey, data);

            ByteArrayOutputStream out = new ByteArrayOutputStream();

            byte[] x = eccCipher.getX();
            byte[] y = eccCipher.getY();
            byte[] PO = new byte[x.length + y.length + 1];
            PO[0] = 0x04;
            System.arraycopy(x, 0, PO, 1, x.length);
            System.arraycopy(y, 0, PO, x.length + 1, y.length);
            out.write(PO);
            out.write(eccCipher.getC()); //密文
            out.write(eccCipher.getM()); //hash

            byte[] cipher = out.toByteArray();
//            GMSSLByteArrayUtils.printHexBinary(logger, "encrypt", cipher);
            return cipher;
        } catch (SdfSDKException | IOException e) {
            logger.error("encrypt", e);
            return new byte[0];
        }
    }

    public byte[] decrypt(byte[] cipher) {
        GMSSLByteArrayUtils.printHexBinary(logger, "cipher", cipher);
        int curveLength = 32;
        int hashLength = 32;

        byte[] c1 = new byte[curveLength * 2 + 1];
        System.arraycopy(cipher, 0, c1, 0, c1.length);
        GMSSLByteArrayUtils.printHexBinary(logger, "c1", c1);

        byte[] x = new byte[curveLength];
        System.arraycopy(c1, 1, x, 0, x.length);
        byte[] y = new byte[curveLength];
        System.arraycopy(c1, x.length + 1, y, 0, y.length);

        byte[] c = new byte[cipher.length - hashLength - c1.length];
        System.arraycopy(cipher, c1.length, c, 0, c.length);

        byte[] m = new byte[hashLength];
        System.arraycopy(cipher, cipher.length - hashLength, m, 0, hashLength);

        SdfECCCipher eccCipher = new SdfECCCipher(x, y, m, c.length, c);

        try {
            //SM2解密
            return sdfSDK.internalDecryptECC(ecKey.getSm2Index(), ecKey.getPassword(), c.length, eccCipher);
        } catch (SdfSDKException e) {
            logger.error("decrypt", e);
            return new byte[0];
        }

    }

    public void release() throws SdfSDKException {
        if (sdfSDK != null) {
            sdfSDK.release();
        }
    }
}
