package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfECCSignature;
import org.bouncycastle.asn1.*;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.Signer;
import org.bouncycastle.math.ec.ECConstants;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;

public class SdfSM2Signer extends SdfSigner {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private SdfSM3Digest digest;

    private SdfECKeyParameters ecKey;

    public SdfSM2Signer() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfSM2Signer(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSM2Signer(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
    }

    @Override
    public void init(boolean forSigning, CipherParameters param) {
        this.ecKey = (SdfECKeyParameters) param;
        byte[] userID = GMSSLByteArrayUtils.hexDecode("31323334353637383132333435363738"); // the default value
        init(forSigning, userID, param);
    }

    public void init(boolean forSigning, byte[] userID, CipherParameters param) {
        this.ecKey = (SdfECKeyParameters) param;
        try {
            SdfECCPublicKey sdfECCPublicKey;
            if (forSigning) {
                sdfECCPublicKey = this.sdfSDK.exportSignPublicKeyEcc(this.ecKey.getSm2Index());
            } else {
                sdfECCPublicKey = this.ecKey.getSDFECCPublicKey();
            }
            digest = new SdfSM3Digest(sdfSDK, userID, sdfECCPublicKey);
        } catch (SdfSDKException e) {
            logger.error("init", e);
        }
    }

    @Override
    public void update(byte b) {
        byte[] bytes = new byte[]{b};
        update(bytes, 0, bytes.length);
    }

    @Override
    public void update(byte[] in, int off, int len) {
//        while (len > 4096) {
//            byte[] m = new byte[4096];
//            System.arraycopy(in, 0, m, 0, m.length);
//            digest.update(m, 0, m.length);
//            len = len - 4096;
//            byte[] inNow = new byte[in.length - m.length];
//            System.arraycopy(in,  m.length, inNow, 0, in.length - m.length);
//            in = inNow;
//        }
        digest.update(in, off, len);
    }

    @Override
    public byte[] generateSignature() throws CryptoException, DataLengthException {
        byte[] out = new byte[digest.getDigestSize()];
        digest.doFinal(out, 0);
        try {
            //SM2签名
            SdfECCSignature sdfECCSignature = sdfSDK.internalSignECC(ecKey.getSm2Index(), ecKey.getPassword(), out);
            return derEncode(sdfECCSignature.getR(), sdfECCSignature.getS());
        } catch (SdfSDKException | IOException e) {
            GMSSLByteArrayUtils.printHexBinary(logger, "signature sm3 digest", out);
            logger.error("generateSignature", e);
            return new byte[0];
        }
    }

    @Override
    public boolean verifySignature(byte[] signature) {
        try {
            byte[] out = new byte[digest.getDigestSize()];
            digest.doFinal(out, 0);
//            GMSSLByteArrayUtils.printHexBinary(logger, "signature sm3 digest", out);
            BigInteger[] bigIntegers = derDecode(signature);
            byte[] r = GMSSLByteArrayUtils.filterByteArrayZeroInHead(bigIntegers[0].toByteArray());
//            GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature R", r);
            byte[] s = GMSSLByteArrayUtils.filterByteArrayZeroInHead(bigIntegers[1].toByteArray());
//            GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature S", s);
            SdfECCSignature sdfECCSignature = new SdfECCSignature(r, s);
            //SM2验签
            sdfSDK.externalVerifyECC(ecKey.getSDFECCPublicKey(), out, sdfECCSignature);
            return true;
        } catch (IOException | SdfSDKException e) {
            if (signature != null) {
                GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature signature", signature);
            }
            logger.error("verifySignature", e);
            return false;
        }
    }

    @Override
    public void reset() {

    }

    protected BigInteger[] derDecode(byte[] encoding) throws IOException {
        ASN1Sequence seq = ASN1Sequence.getInstance(ASN1Primitive.fromByteArray(encoding));
        if (seq.size() != 2) {
            return null;
        }

        BigInteger r = ASN1Integer.getInstance(seq.getObjectAt(0)).getValue();
        BigInteger s = ASN1Integer.getInstance(seq.getObjectAt(1)).getValue();

        byte[] expectedEncoding = derEncode(r, s);
        if (!Arrays.constantTimeAreEqual(expectedEncoding, encoding)) {
            return null;
        }

        return new BigInteger[]{r, s};
    }

    protected byte[] derEncode(byte[] r, byte[] s) throws IOException {
        return derEncode(
                BigIntegers.fromUnsignedByteArray(r),
                BigIntegers.fromUnsignedByteArray(s)
        );
    }

    protected byte[] derEncode(BigInteger r, BigInteger s) throws IOException {
//        GMSSLByteArrayUtils.printHexBinary(logger, "r", r.toByteArray());
//        GMSSLByteArrayUtils.printHexBinary(logger, "s", s.toByteArray());
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(new ASN1Integer(r));
        v.add(new ASN1Integer(s));
        return new DERSequence(v).getEncoded(ASN1Encoding.DER);
    }
}
