package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdHash;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import org.bouncycastle.crypto.ExtendedDigest;
import org.bouncycastle.util.Memoable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

public abstract class SdfBaseDigest implements ExtendedDigest, Memoable {

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    private byte[] in = new byte[0];
    private volatile boolean isInit = false;

    protected SdfSDK sdfSDK;

    public SdfBaseDigest() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfBaseDigest(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfBaseDigest(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
//        this.init();
    }

    protected abstract void init() throws SdfSDKException;

    private synchronized void checkInit() throws SdfSDKException {
        if (!this.isInit) {
            //hash init
            this.init();
            this.isInit = true;
        }
    }

    private synchronized void resetInit() {
        if (this.isInit) {
            this.in = new byte[0];
            this.isInit = false;
        }
    }

    protected void copyIn(SdfBaseDigest t) throws SdfSDKException {
        //sdf sdk copy
        this.sdfSDK = SdfCryptoType.cloneSdfSDK(t.sdfSDK);
        this.sdfSDK.init();
        this.checkInit();
        if (t.in != null) {
            this.update(t.in, 0, t.in.length);
        }
    }

    @Override
    public void update(byte in) {
        byte[] data = new byte[]{in};
        this.update(data, 0, data.length);
    }

    @Override
    public void update(byte[] in, int inOff, int len) {
        if (in.length == 0 || len == 0) {
            logger.info("update len=0, do nothing.");
        }
        byte[] data = new byte[len];
        System.arraycopy(in, inOff, data, 0, len);
        byte[] out = new byte[this.in.length + len];
        System.arraycopy(this.in, 0, out, 0, this.in.length);
        System.arraycopy(data, 0, out, this.in.length, len);
        this.in = out;

        try {
            this.checkInit();
            while (len > 4096) {
                byte[] m = new byte[4096];
                System.arraycopy(data, 0, m, 0, m.length);
                sdfSDK.hashUpdate(m);
                len = len - 4096;
                byte[] dataNow = new byte[data.length - m.length];
                System.arraycopy(data, m.length, dataNow, 0, data.length - m.length);
                data = dataNow;
            }
            sdfSDK.hashUpdate(data);
        } catch (SdfSDKException e) {
            logger.error("update error in.length={} len={}", data.length, len, e.getMessage());
        }
    }

    @Override
    public int doFinal(byte[] out, int outOff) {
        int length = getDigestSize();
        try {
            byte[] res = sdfSDK.hashFinal(length);
            System.arraycopy(res, 0, out, outOff, res.length);
        } catch (SdfSDKException e) {
            logger.error("doFinal", e);
        } finally {
            this.resetInit();
        }
        return length;
    }

    /**
     * reset the chaining variables
     */
    @Override
    public void reset() {
        try {
            byte[] resBuf = new byte[this.getDigestSize()];
            if (this.isInit) {
                this.doFinal(resBuf, 0);
            }
            this.resetInit();
            this.checkInit();
        } catch (SdfSDKException e) {
            logger.error("reset error", e);
        }
    }

    public void releaseConnection() {
        if (sdfSDK instanceof YunhsmSdfSDK) {
            ((YunhsmSdfSDK) sdfSDK).releaseConnection();
        }
        this.resetInit();
    }

    public void release() throws SdfSDKException {
        if (sdfSDK != null) {
            sdfSDK.release();
        }
    }

}
