package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfECCSignature;
import org.bouncycastle.asn1.*;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.ExtendedDigest;
import org.bouncycastle.crypto.params.ParametersWithID;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;

public abstract class SdfECBaseSigner extends SdfSigner {

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    protected ExtendedDigest digest;

    protected SdfECKeyParameters ecKey;

    protected String signAlgName;

    public SdfECBaseSigner() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfECBaseSigner(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfECBaseSigner(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.sdfSDK.init();
    }

    public SdfECBaseSigner(SdfSDK sdfSDK, String signAlgName) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        this.signAlgName = signAlgName;
        this.sdfSDK.init();
    }

    @Override
    public void init(boolean forSigning, CipherParameters param) {
        byte[] userID;
        if (param instanceof ParametersWithID) {
            this.ecKey = (SdfECKeyParameters) ((ParametersWithID) param).getParameters();
            userID = ((ParametersWithID) param).getID();
        } else {
            this.ecKey = (SdfECKeyParameters) param;
            // the default value.
            userID = Hex.decode("31323334353637383132333435363738");
        }
        try {

            if (null == signAlgName || signAlgName.toUpperCase().contains("SM3")) {
                SdfECCPublicKey sdfECCPublicKey;
                if (forSigning) {
                    sdfECCPublicKey = this.sdfSDK.exportSignPublicKeyEcc(this.ecKey.getEcIndex());
                } else {
                    sdfECCPublicKey = this.ecKey.getSDFECCPublicKey();
                }
                initDigest(userID, sdfECCPublicKey);
            } else {
                initDigest(signAlgName);
            }
        } catch (SdfSDKException e) {
            logger.error("init", e);
        }
    }

    public abstract void initDigest(byte[] userID, SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException;

    public abstract void initDigest(String signAlgName) throws SdfSDKException;

    @Override
    public void update(byte b) {
        byte[] bytes = new byte[]{b};
        update(bytes, 0, bytes.length);
    }

    @Override
    public void update(byte[] in, int off, int len) {
        digest.update(in, off, len);
    }

    @Override
    public byte[] generateSignature() throws CryptoException, DataLengthException {
        byte[] out = new byte[digest.getDigestSize()];
        digest.doFinal(out, 0);
        out = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(out, 32);
        try {
            //SM2签名
            SdfECCSignature sdfECCSignature = sdfSDK.internalSignECC(ecKey.getEcIndex(), ecKey.getPassword(), out);
            return derEncode(sdfECCSignature.getR(), sdfECCSignature.getS());
        } catch (SdfSDKException | IOException e) {
            GMSSLByteArrayUtils.printHexBinary(logger, "signature  digest", out);
            logger.error("generateSignature error index={} password={}", ecKey.getEcIndex(), ecKey.getPassword(), e);
            return new byte[0];
        }
    }

    @Override
    public boolean verifySignature(byte[] signature) {
        byte[] out = new byte[digest.getDigestSize()];
        try {
            digest.doFinal(out, 0);
            out = GMSSLByteArrayUtils.fillByteArrayWithZeroInHead(out, 32);
            BigInteger[] bigIntegers = derDecode(signature);
            byte[] r = GMSSLByteArrayUtils.filterByteArrayZeroInHead(bigIntegers[0].toByteArray());
            byte[] s = GMSSLByteArrayUtils.filterByteArrayZeroInHead(bigIntegers[1].toByteArray());
            SdfECCSignature sdfECCSignature = new SdfECCSignature(r, s);
            //SM2验签
            if (null == signAlgName) {
                sdfSDK.externalVerifyECC(ecKey.getSDFECCPublicKey(), out, sdfECCSignature);
            } else {
                sdfSDK.externalVerifyECC(ecKey.getSDFECCPublicKey(), out, sdfECCSignature, signAlgName);
            }
            return true;
        } catch (IOException | SdfSDKException e) {
            if (signature != null) {
                GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature sm3 digest", out, true);
                GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature x", ecKey.getSDFECCPublicKey().getX(), true);
                GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature y", ecKey.getSDFECCPublicKey().getY(), true);
                GMSSLByteArrayUtils.printHexBinary(logger, "verifySignature signature", signature, true);
            }
            logger.error("verifySignature", e);
            return false;
        }
    }

    @Override
    public void reset() {

    }

    protected BigInteger[] derDecode(byte[] encoding) throws IOException {
        ASN1Sequence seq = ASN1Sequence.getInstance(ASN1Primitive.fromByteArray(encoding));
        if (seq.size() != 2) {
            return null;
        }

        BigInteger r = ASN1Integer.getInstance(seq.getObjectAt(0)).getValue();
        BigInteger s = ASN1Integer.getInstance(seq.getObjectAt(1)).getValue();

        byte[] expectedEncoding = derEncode(r, s);
        if (!Arrays.constantTimeAreEqual(expectedEncoding, encoding)) {
            return null;
        }

        return new BigInteger[]{r, s};
    }

    protected byte[] derEncode(byte[] r, byte[] s) throws IOException {
        return derEncode(
                BigIntegers.fromUnsignedByteArray(r),
                BigIntegers.fromUnsignedByteArray(s)
        );
    }

    protected byte[] derEncode(BigInteger r, BigInteger s) throws IOException {
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(new ASN1Integer(r));
        v.add(new ASN1Integer(s));
        return new DERSequence(v).getEncoded(ASN1Encoding.DER);
    }
}
