package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.params.ParametersWithIV;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.interfaces.ECPublicKey;

public class SdfSymmetricCipher {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final int BLOCK_SIZE = 16;

    private SdfSDK sdfSDK;

    private long[] phKeyHandle;
    private byte[] key;
    private boolean forEncryption;

    private ParametersWithIV parametersWithIV;
    private SdfSymmetricKeyParameters sdfBlockCipherKeyParameters;

    public SdfSymmetricCipher() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfSymmetricCipher(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSymmetricCipher(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
    }

    public byte[] getKey() {
        return key;
    }

    public void init(boolean forEncryption, CipherParameters params) throws IllegalArgumentException {
        try {
            this.sdfSDK.init();
        } catch (SdfSDKException e) {
            throw new IllegalArgumentException("sdf SDK init error", e);
        }
        if (params instanceof SdfSymmetricKeyParameters) {
            //这时 初始化向量为 00000000000000000000000000000000 16个 0
            this.sdfBlockCipherKeyParameters = (SdfSymmetricKeyParameters) params;
            this.parametersWithIV = new ParametersWithIV(params, new byte[16]);
        } else if (params instanceof ParametersWithIV) {
            this.parametersWithIV = (ParametersWithIV) params;
            this.sdfBlockCipherKeyParameters = (SdfSymmetricKeyParameters) ((ParametersWithIV) params).getParameters();
        } else {
            throw new IllegalArgumentException("invalid parameter passed to SdfSymmetricCipher init - " + params.getClass().getName());
        }

        this.forEncryption = forEncryption;

        if (this.sdfBlockCipherKeyParameters.getKeyCipherType() == SdfSymmetricKeyParameters.KeyCipherType.NONE) {
            try {
                SdfSymmetricKey sdfSymmetricKey = new SdfSymmetricKey(this.sdfSDK);
                SdfSymmetricKey.KeyHandle keyHandle = sdfSymmetricKey.generateKeyWithEpkEccKeyHandle((ECPublicKey) this.sdfBlockCipherKeyParameters.getPublicKey());
                this.key = keyHandle.getCipherKey();
                this.phKeyHandle = keyHandle.getHandle();
            } catch (SdfSDKException e) {
                throw new IllegalArgumentException("SDF SM4 init - generate key error");
            }
            return;
        }
        this.key = this.sdfBlockCipherKeyParameters.getKey();

        try {
            switch (this.sdfBlockCipherKeyParameters.getKeyCipherType()) {
                case ECC_CIPHER:
                    //导入密文会话密钥
                    SdfSymmetricKey eccSymmetricKey = new SdfSymmetricKey(this.sdfSDK);
                    this.phKeyHandle = eccSymmetricKey.importKeyWithIskEcc(this.sdfBlockCipherKeyParameters.getSdfPrivateKey(), key);
                    break;
                case PLAIN:
                    //导入明文会话密钥
                    if (key.length != 16) {
                        throw new IllegalArgumentException("SM4 requires a 128 bit key");
                    }
                    this.phKeyHandle = sdfSDK.importKey(key);
                    break;
                case RSA_CIPHER:
                    SdfSymmetricKey rsaSymmetricKey = new SdfSymmetricKey(this.sdfSDK);
                    this.phKeyHandle = rsaSymmetricKey.importKeyWithIskRsa(this.sdfBlockCipherKeyParameters.getSdfPrivateKey(), key);
                    break;
                case KEK:
                    SdfSymmetricKey kekSymmetricKey = new SdfSymmetricKey(this.sdfSDK);
                    this.phKeyHandle = kekSymmetricKey.importKeyWithKek(this.sdfBlockCipherKeyParameters.getKeyEncryptKeyType(), this.sdfBlockCipherKeyParameters.getPrivateKeyIndex(), key);
                    break;
                default:
                    throw new IllegalArgumentException("SDF SM4 init - unknown key cipher type " + this.sdfBlockCipherKeyParameters.getKeyCipherType());
            }
        } catch (SdfSDKException e) {
            logger.error("init import key", e);
            throw new IllegalArgumentException("SDF SM4 init - import key error");
        }
    }

    public String getAlgorithmName() {
        return this.sdfBlockCipherKeyParameters.getSdfAlgIdBlockCipher().getName();
    }

    public int getBlockSize() {
        return BLOCK_SIZE;
    }

    public int getEncryptionLength(int inLength, SdfSymmetricKeyParameters.PaddingType paddingType) {
        if (paddingType == SdfSymmetricKeyParameters.PaddingType.NoPadding) {
            return inLength;
        } else {
            int paddingLength = BLOCK_SIZE - (inLength % BLOCK_SIZE);
            return inLength + paddingLength;
        }
    }

    public byte[] doFinal(byte[] data) throws DataLengthException, IllegalStateException {
//        GMSSLByteArrayUtils.printHexBinary(logger, "processBlock in", data);
        logger.info("padding type " + this.sdfBlockCipherKeyParameters.getPaddingType());

        try {
            byte[] iv = parametersWithIV.getIV();
//            GMSSLByteArrayUtils.printHexBinary(logger, "processBlock iv", iv);
            if (this.forEncryption) {
                byte[] plainText;
                if (this.sdfBlockCipherKeyParameters.getPaddingType() == SdfSymmetricKeyParameters.PaddingType.NoPadding) {
                    plainText = new byte[data.length];
                    System.arraycopy(data, 0, plainText, 0, data.length);
                } else {
                    int paddingLength = BLOCK_SIZE - (data.length % BLOCK_SIZE);
                    plainText = new byte[data.length + paddingLength];
                    System.arraycopy(data, 0, plainText, 0, data.length);
                    byte padding;
                    if (this.sdfBlockCipherKeyParameters.getPaddingType() == SdfSymmetricKeyParameters.PaddingType.SSL3Padding) {
                        padding = (byte) (paddingLength - 1);
                    } else {
                        padding = (byte) (paddingLength);
                    }
                    int off = data.length;
                    for (int i = 0; i < paddingLength; i++) {
                        plainText[off++] = padding;
                    }
                }
                byte[] pucEncData = sdfSDK.encrypt(phKeyHandle, this.sdfBlockCipherKeyParameters.getSdfAlgIdBlockCipher(), iv, plainText);
//                GMSSLByteArrayUtils.printHexBinary(logger, "processBlock pucEncData", pucEncData);
                return pucEncData;
            } else {
                byte[] plain = sdfSDK.decrypt(phKeyHandle, this.sdfBlockCipherKeyParameters.getSdfAlgIdBlockCipher(), iv, data);
                byte lastByte = plain[plain.length - 1];
                byte[] plainText;
                switch (this.sdfBlockCipherKeyParameters.getPaddingType()) {
                    case PKCS5Padding:
                    case PKCS7Padding:
                        plainText = new byte[data.length - (lastByte & 0xff)];
                        break;
                    case SSL3Padding:
                        plainText = new byte[data.length - ((lastByte & 0xff) + 1)];
                        break;
                    default:
                        plainText = new byte[data.length];
                        break;
                }
                System.arraycopy(plain, 0, plainText, 0, plainText.length);
//                GMSSLByteArrayUtils.printHexBinary(logger, "processBlock plainText", plainText);
                return plainText;
            }
        } catch (SdfSDKException e) {
            logger.error("processBlock", e);
            throw new IllegalStateException(e.getMessage());
        }
    }

    public void reset() {

    }

    public void release() throws SdfSDKException {
        if (sdfSDK != null && phKeyHandle != null) {
            sdfSDK.destroyKey(phKeyHandle);
            sdfSDK.release();
        }
    }
}
