package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.asn1.crypto.ASN1SM2Cipher;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Random;

public class SdfSymmetricKey {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    public class KeyHandle{
        private byte[] cipherKey;
        private long[] handle;

        public KeyHandle(byte[] cipherKey, long[] handle) {
            this.cipherKey = cipherKey;
            this.handle = handle;
        }

        public byte[] getCipherKey() {
            return cipherKey;
        }

        public long[] getHandle() {
            return handle;
        }
    }

    private SdfSDK sdfSDK;

    public SdfSymmetricKey() throws SdfSDKException {
        this(SdfCryptoType.YUNHSM);
    }

    public SdfSymmetricKey(SdfCryptoType sdfCryptoType) throws SdfSDKException {
        this(sdfCryptoType.getSdfSDK());
    }

    public SdfSymmetricKey(SdfSDK sdfSDK) throws SdfSDKException {
        this.sdfSDK = sdfSDK;
        sdfSDK.init();
    }

    /********************************
     * ECC
     ********************************/

    /**
     * 生成ECC公钥加密的 会话密钥，返回加密会话密钥
     * @param publicKey 加密会话密钥的ECC公钥
     * @return 加密的会话密钥 ASN1结构体
     * @throws SdfSDKException 异常
     */
    public byte[] generateKeyWithEpkEcc(ECPublicKey publicKey) throws SdfSDKException {
        SdfECCPublicKey sdfECCPublicKey = SdfECCPublicKey.getInstance(publicKey);
        SdfECCCipher eccCipher = this.sdfSDK.generateKeyWithEpkEcc(sdfECCPublicKey);
        ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(eccCipher.getX(), eccCipher.getY(), eccCipher.getM(), eccCipher.getC());
        byte[] out;
        try {
            out = asn1SM2Cipher.toASN1Primitive().getEncoded();
        } catch (IOException e) {
            throw new SdfSDKException("asn1 sm2 cipher encode error", e);
        }
//        GMSSLByteArrayUtils.printHexBinary(logger, "encrypt", out);
        return out;
    }

    /**
     * 生成ECC公钥加密的 会话密钥，返回handle
     * @param publicKey 加密的ECC公钥
     * @return 加密的会话密钥 handle
     * @throws SdfSDKException 异常
     */
    public long[] generateKeyWithEpkEccHandle(ECPublicKey publicKey) throws SdfSDKException {
        SdfECCPublicKey sdfECCPublicKey = SdfECCPublicKey.getInstance(publicKey);
        return this.sdfSDK.generateKeyWithEpkEccHandle(sdfECCPublicKey);
    }

    /**
     * 生成ECC公钥加密的 会话密钥，返回加密会话密钥及handle
     * @param publicKey 加密会话密钥的ECC公钥
     * @return 加密的会话密钥 ASN1结构体
     * @throws SdfSDKException 异常
     */
    public KeyHandle generateKeyWithEpkEccKeyHandle(ECPublicKey publicKey) throws SdfSDKException {
        SdfECCPublicKey sdfECCPublicKey = SdfECCPublicKey.getInstance(publicKey);
        SdfSymmetricKeyHandle keyHandle = this.sdfSDK.generateKeyWithEpkEccKeyHandle(sdfECCPublicKey);
        SdfECCCipher eccCipher = keyHandle.getCipherKey();
        ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(eccCipher.getX(), eccCipher.getY(), eccCipher.getM(), eccCipher.getC());
        byte[] out;
        try {
            out = asn1SM2Cipher.toASN1Primitive().getEncoded();
        } catch (IOException e) {
            throw new SdfSDKException("asn1 sm2 cipher encode error", e);
        }
//        GMSSLByteArrayUtils.printHexBinary(logger, "encrypt", out);
        return new KeyHandle(out, keyHandle.getHandle());
    }

    /**
     * 导入ECC公钥加密的 会话密钥
     * @param index 加密ECC公钥对应的私钥 索引
     * @param password 加密ECC公钥对应的私钥 索引
     * @param cipher 加密的会话密钥 handle
     * @return 会话密钥 handle
     * @throws SdfSDKException 异常
     */
    public long[] importKeyWithIskEcc(int index, String password, byte[] cipher) throws SdfSDKException {
        ASN1SM2Cipher sm2CipherASN1 = ASN1SM2Cipher.getInstance(cipher);
        assert sm2CipherASN1 != null;
        SdfECCCipher sdfECCCipher = new SdfECCCipher(
                sm2CipherASN1.getxCoordinate().toByteArray(),
                sm2CipherASN1.getyCoordinate().toByteArray(),
                sm2CipherASN1.getHash(),
                sm2CipherASN1.getCipherText().length,
                sm2CipherASN1.getCipherText()
        );
        //SM2解密
        long[] handle = sdfSDK.importKeyWithIskEcc(index, password.getBytes(), sdfECCCipher);
        return handle;
    }

    public long[] importKeyWithIskEcc(SdfPrivateKey sdfPrivateKey, byte[] cipher) throws SdfSDKException {
        return importKeyWithIskEcc(sdfPrivateKey.getIndex(), sdfPrivateKey.getStringPassword(), cipher);
    }

    /********************************
     * RSA
     ********************************/

    /**
     * 生成RSA公钥加密的 会话密钥，返回加密会话密钥
     * @param publicKey 加密会话密钥的RSA公钥
     * @return 密文会话密钥
     * @throws SdfSDKException 异常
     */
    public byte[] generateKeyWithEpkRsa(RSAPublicKey publicKey) throws SdfSDKException {
        SdfRSAPublicKey sdfRSAPublicKey = SdfRSAPublicKey.getInstance(publicKey);
        return this.sdfSDK.generateKeyWithEpkRsa(sdfRSAPublicKey);
    }

    /**
     * 生成RSA公钥加密的 会话密钥，返回handle
     * @param publicKey 加密的RSA公钥
     * @return 加密的会话密钥
     * @throws SdfSDKException 异常
     */
    public long[] generateKeyWithEpkRsaHandle(RSAPublicKey publicKey) throws SdfSDKException {
        SdfRSAPublicKey sdfRSAPublicKey = SdfRSAPublicKey.getInstance(publicKey);
        return this.sdfSDK.generateKeyWithEpkRsaHandle(sdfRSAPublicKey);
    }

    /**
     * 生成RSA公钥加密的 会话密钥，返回加密会话密钥及handle
     * @param publicKey 加密会话密钥的RSA公钥
     * @return 加密会话密钥 ASN1结构体
     * @throws SdfSDKException 异常
     */
    public KeyHandle generateKeyWithEpkRsaKeyHandle(RSAPublicKey publicKey) throws SdfSDKException {
        SdfRSAPublicKey sdfRSAPublicKey = SdfRSAPublicKey.getInstance(publicKey);
        SdfSymmetricKeyHandle keyHandle = this.sdfSDK.generateKeyWithEpkRsaKeyHandle(sdfRSAPublicKey);
        return new KeyHandle(keyHandle.getRsaCipherKey(), keyHandle.getHandle());
    }

    /**
     * 导入RSA公钥加密的 会话密钥
     * @param index 加密RSA公钥对应的私钥 索引
     * @param password 加密RSA公钥对应的私钥 索引
     * @param cipher 加密会话密钥
     * @return 会话密钥 handle
     * @throws SdfSDKException
     */
    public long[] importKeyWithIskRsa(int index, String password, byte[] cipher) throws SdfSDKException {
//        ASN1SM2Cipher sm2CipherASN1 = ASN1SM2Cipher.getInstance(cipher);
//        assert sm2CipherASN1 != null;
        return sdfSDK.importKeyWithIskRsa(index, password.getBytes(), cipher);
    }

    public long[] importKeyWithIskRsa(SdfPrivateKey sdfPrivateKey, byte[] cipher) throws SdfSDKException {
        return importKeyWithIskRsa(sdfPrivateKey.getIndex(), sdfPrivateKey.getStringPassword(), cipher);
    }


    /********************************
     * 密钥加密密钥
     ********************************/

    /**
     * 导入 会话密钥 并用 密钥加密密钥 解密
     *
     * @param keyIndex    算法标识 加密模式使用ECB模式
     * @param keyIndex 密钥加密密钥索引值
     * @param cipher     加密会话密钥
     * @return 会话密钥句柄
     */
    public long[] importKeyWithKek(SdfAlgIdSymmetric sdfAlgIdSymmetric, int keyIndex, byte[] cipher) throws SdfSDKException {
        return this.sdfSDK.importKeyWithKek(sdfAlgIdSymmetric, keyIndex, cipher);
    }

    /********************************
     * 明文
     ********************************/
    public long[] importKey(byte[] key) throws SdfSDKException {
        return this.sdfSDK.importKey(key);
    }

    public void destroyKey(long[] handle) throws SdfSDKException {
        this.sdfSDK.destroyKey(handle);
    }

    public void release() throws SdfSDKException {
        this.sdfSDK.release();
    }
}