package com.xdja.pki.gmssl.crypto.sdf;

import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import org.bouncycastle.crypto.CipherParameters;

import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;

public class SdfSymmetricKeyParameters implements CipherParameters {

    public enum KeyCipherType {
        NONE,
        PLAIN,
        ECC_CIPHER,
        RSA_CIPHER,
        KEK
    }

    /*******************************
     * NoPadding 无填充
     *
     * PKCS5Padding 明确定义 block 的大小为 8 字节，
     * PKCS7Padding RFC 5652 P27 填充的为填充字节序列的长度 blocksize 为 1-255 字节。
     *     PKCS＃5填充与PKCS＃7填充相同，只是它仅为使用64位（8字节）块大小的块密码定义。在实践中，这两者可以互换使用。
     *          例如： block 为 8，数据长度为 9，
     *              数据： FF FF FF FF FF FF FF FF FF
     *              PKCS7 填充： FF FF FF FF FF FF FF FF FF 07 07 07 07 07 07 07
     *
     * SSL3Padding RFC 5246 P23
     *          例如： block 为 8，数据长度为 9，
     *              数据： FF FF FF FF FF FF FF FF FF
     *              PKCS7 填充： FF FF FF FF FF FF FF FF FF 06 06 06 06 06 06 06
     *      GenericBlockCipher实例的大小必须是块密码块长度的倍数。
     *              sizeof(content) + sizeof(MAC) % block_length = 0,
     *      填充必须是（block_length - 1）个字节长，因为 padding_length 的存在。
     *      这使填充方案与 PKCS5Padding 相似（但不完全），其中填充长度在填充中编码（范围从 1 到 block_length ）。
     *      使用SSL方案，sizeof（填充）以始终存在的 padding_length 编码，因此范围从 0 到 block_length - 1 。
     *          GMSSL 中 padding_length 在前 padding 在后
     *
     * The padding scheme defined in the SSL Protocol Version 3.0, November 18, 1996, section 5.2.3.2 (CBC block cipher):
     * block-ciphered struct {
     *      opaque content[SSLCompressed.length];
     *      opaque MAC[CipherSpec.hash_size];
     *      uint8 padding[GenericBlockCipher.padding_length];
     *      uint8 padding_length;
     * } GenericBlockCipher;
     ********************************/
    public enum PaddingType {
        NoPadding,
        PKCS5Padding,
        PKCS7Padding,
        SSL3Padding
    }

    private byte[] key;
    private KeyCipherType keyCipherType;
    private final PaddingType paddingType;
    private SdfAlgIdSymmetric keyEncryptKeyType;
    private SdfAlgIdSymmetric sdfAlgIdBlockCipher;

    private SdfPrivateKey sdfPrivateKey;
    private PublicKey publicKey;

    /**
     * ECC 加密的 会话密钥参数 构造方法
     *
     * @param sdfAlgIdSymmetric 算法 ID
     * @param sdfPrivateKey     非对称加密 的 私钥 索引及访问控制密码
     * @param cipherKey         ECC 加密的会话密钥
     */
    public SdfSymmetricKeyParameters(KeyCipherType keyCipherType, PaddingType paddingType, SdfAlgIdSymmetric sdfAlgIdSymmetric, SdfPrivateKey sdfPrivateKey, byte[] cipherKey) {
        this(null, keyCipherType, paddingType, sdfAlgIdSymmetric, sdfPrivateKey, null, cipherKey, 0, cipherKey.length);
    }


    /**
     * ECC 加密的 会话密钥参数 构造方法
     *
     * @param sdfAlgIdSymmetric 算法 ID
     * @param index             密钥加密的索引
     * @param cipherKey         ECC 加密的会话密钥
     */
    public SdfSymmetricKeyParameters(SdfAlgIdSymmetric keyEncryptKeyType, PaddingType paddingType, SdfAlgIdSymmetric sdfAlgIdSymmetric, int index, byte[] cipherKey) {
        this(keyEncryptKeyType, KeyCipherType.KEK, paddingType, sdfAlgIdSymmetric, new SdfPrivateKey(index), null, cipherKey, 0, cipherKey.length);
    }


    /**
     * 明文 会话密钥参数 构造方法
     *
     * @param paddingType       待加密数据 填充模式
     * @param sdfAlgIdSymmetric 算法 ID
     * @param key               明文会话密钥
     */
    public SdfSymmetricKeyParameters(PaddingType paddingType, SdfAlgIdSymmetric sdfAlgIdSymmetric, byte[] key) {
        this(null, KeyCipherType.PLAIN, paddingType, sdfAlgIdSymmetric, null, null, key, 0, key.length);
    }

    /**
     * 通过 handle 进行计算 构造方法
     *
     * @param paddingType       待加密数据 填充模式
     * @param sdfAlgIdSymmetric 算法 ID
     */
    public SdfSymmetricKeyParameters(PaddingType paddingType, SdfAlgIdSymmetric sdfAlgIdSymmetric, ECPublicKey publicKey) {
        this(null, KeyCipherType.NONE, paddingType, sdfAlgIdSymmetric, null, publicKey, null, 0, 0);
    }

    /**
     * 明文 会话密钥参数 构造方法
     *
     * @param keyCipherType     加密的会话密钥的 加密算法类型
     * @param paddingType       待加密数据 填充模式
     * @param sdfAlgIdSymmetric 算法 ID
     * @param sdfPrivateKey     非对称加密 的 私钥 索引及访问控制密码 明文会话密钥 时 为 null
     * @param key               明文会话密钥 或 加密的会话密钥
     */
    public SdfSymmetricKeyParameters(
            SdfAlgIdSymmetric keyEncryptKeyType,
            KeyCipherType keyCipherType,
            PaddingType paddingType,
            SdfAlgIdSymmetric sdfAlgIdSymmetric,
            SdfPrivateKey sdfPrivateKey,
            PublicKey publicKey,
            byte[] key, int keyOff, int keyLen
    ) {
        if (keyCipherType == KeyCipherType.ECC_CIPHER && sdfPrivateKey == null) {
            throw new IllegalArgumentException("key cipher type is ecc, private key is not null");
        }
        this.keyEncryptKeyType = keyEncryptKeyType;
        this.keyCipherType = keyCipherType;
        this.paddingType = paddingType;
        this.sdfAlgIdBlockCipher = sdfAlgIdSymmetric;
        this.sdfPrivateKey = sdfPrivateKey;
        this.publicKey = publicKey;
        if (keyLen > 0) {
            this.key = new byte[keyLen];
            System.arraycopy(key, keyOff, this.key, 0, keyLen);
        }
    }

    public SdfAlgIdSymmetric getKeyEncryptKeyType() {
        return keyEncryptKeyType;
    }

    public int getPrivateKeyIndex() {
        return this.sdfPrivateKey.getIndex();
    }

    public PublicKey getPublicKey() {
        return publicKey;
    }

    public byte[] getKey() {
        return key;
    }

    public KeyCipherType getKeyCipherType() {
        return keyCipherType;
    }

    public PaddingType getPaddingType() {
        return paddingType;
    }

    public SdfAlgIdSymmetric getSdfAlgIdBlockCipher() {
        return sdfAlgIdBlockCipher;
    }

    public SdfPrivateKey getSdfPrivateKey() {
        return sdfPrivateKey;
    }
}