package com.xdja.pki.gmssl.sdf.pcie;

import com.xdja.pcie.sdf.jni.SdfApiDemo;
import com.xdja.pcie.sdf.jni.bean.*;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.AbstractSdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.*;
import com.xdja.pcie.sdf.jni.SdfApi;
import com.xdja.pcie.sdf.jni.alg.AlgId;
import com.xdja.pki.gmssl.sdf.bean.SdfApiCode;

import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.PublicKey;

public class PcieSdfSDK extends AbstractSdfSDK {

    private SdfApi sdfApi = new SdfApi();

    @Override
    public void init() throws SdfSDKException {
        //打开设备
        int ret = sdfApi.openDevice(dev);
        checkRet("openDevice", ret);
        //打开 session
        ret = sdfApi.openSession(dev[0], ses);
        checkRet("openSession", ret);
    }

    @Override
    public void release() throws SdfSDKException {
        int ret = sdfApi.closeSession(ses[0]);
        checkRet("closeSession", ret);
        ret = sdfApi.closeDevice(dev[0]);
        checkRet("closeDevice", ret);
        this.dev = new long[]{0};
        this.ses = new long[]{0};
    }

    @Override
    public byte[] generateRandom(int uiLength) throws SdfSDKException {
        byte[] pucRandom = new byte[uiLength];
        int ret = sdfApi.generateRandom(ses[0], uiLength, pucRandom);
        checkRet("generateRandom", ret);
        return pucRandom;
    }

    @Override
    public SdfECCKeyPair generateKeyPairEcc() throws SdfSDKException {
        //ECC签名公钥  ECC加密公钥
        EccPublicKey pucPublicKey = new EccPublicKey();
        EccPrivateKey pucPrivateKey = new EccPrivateKey();
        int ret = sdfApi.generateKeyPairEcc(ses[0], AlgId.SGD_SM2, GMSSLX509Utils.ECC_KEY_BITS, pucPublicKey, pucPrivateKey);
        checkRet("generateKeyPairEcc", ret);
        byte[] k = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucPrivateKey.getK());
        return new SdfECCKeyPair(
                SdfECCPublicKey.getInstanceFilterFoot(pucPublicKey.getX(), pucPublicKey.getY()),
                SdfECCPrivateKey.getInstanceFilterFoot(pucPrivateKey.getK())
        );
    }

    @Override
    public boolean checkPrivateKeyAccessRight(int index, byte[] password) throws SdfSDKException {
        boolean result = false;
        int ret = sdfApi.getPrivateKeyAccessRight(ses[0], index, password, password.length);
        if (ret == SdfApiCode.SDR_OK) {
            sdfApi.releasePrivateKeyAccessRight(ses[0], index);
            result = true;
        }
        checkRet("checkPrivateKeyAccessRight", ret);
        return result;
    }

    public void getPrivateKeyAccessRight(int index, byte[] password) throws SdfSDKException {
        int ret = sdfApi.getPrivateKeyAccessRight(ses[0], index, password, password.length);
        checkRet("getPrivateKeyAccessRight", ret);
    }

    public void releasePrivateKeyAccessRight(int index) throws SdfSDKException {
        int ret = sdfApi.releasePrivateKeyAccessRight(ses[0], index);
        checkRet("releasePrivateKeyAccessRight", ret);
    }

    public PublicKey importSM2SignKey(KeyStore keyStore, char[] pw, int index) throws SdfSDKException {
        return importSM2Key(keyStore, pw, index, true);
    }

    public PublicKey importSM2EnvKey(KeyStore keyStore, char[] pw, int index) throws SdfSDKException {
        return importSM2Key(keyStore, pw, index, false);
    }

    private PublicKey importSM2Key(KeyStore keyStore, char[] pw, int index, boolean isSign) throws SdfSDKException {
        try {
            String alias = keyStore.aliases().nextElement();
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(alias, pw);
            PublicKey publicKey = keyStore.getCertificate(alias).getPublicKey();
            importSM2PrivateKey(privateKey, index, isSign);
            importSM2PublicKey(publicKey, index, isSign);
            return publicKey;
        } catch (Exception e) {
            throw new SdfSDKException(e);
        }
    }

    public void importSM2PublicKey(PublicKey key, int index, boolean isSign) throws SdfSDKException {
        String method = isSign ? "sm2ImportSignpub" : "sm2ImportEncpub";
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, key);

        int ret;
        if (isSign) {
            ret = sdfApi.sm2ImportSignpub(ses[0], index, pucPublicKey);
        } else {
            ret = sdfApi.sm2ImportEncpub(ses[0], index, pucPublicKey);
        }
        checkRet(method, ret);

        EccPublicKey ecPublicKey_exp = new EccPublicKey();
        if (isSign) {
            ret = sdfApi.exportSignPublicKeyEcc(ses[0], index, ecPublicKey_exp);
        } else {
            ret = sdfApi.exportEncPublicKeyEcc(ses[0], index, ecPublicKey_exp);
        }
        checkRet("exportSignPublicKeyEcc", ret);
    }

    public void importSM2PrivateKey(PrivateKey key, int index, boolean isSign) throws SdfSDKException {
        String method = isSign ? "sm2ImportSignpri" : "sm2ImportEncpri";
        EccPrivateKey pucPrivateKey = PcieSdfSDKUtils.generateEccPrivateKey(logger, key);
        int ret;
        if (isSign) {
            ret = sdfApi.sm2ImportSignpri(ses[0], index, pucPrivateKey, 0);
        } else {
            ret = sdfApi.sm2ImportEncpri(ses[0], index, pucPrivateKey, 0);
        }
        checkRet(method, ret);
    }

    @Override
    public SdfECCSignature internalSignECC(int index, byte[] password, byte[] data) throws SdfSDKException {
        EccSignature pucSignature = new EccSignature();
        int ret = sdfApi.internalSignEcc(ses[0], index, data, data.length, pucSignature);
        if (ret == SdfApiCode.SDR_PARDENY) {
            this.release();
            this.init();
            this.getPrivateKeyAccessRight(index, password);
            ret = sdfApi.internalSignEcc(ses[0], index, data, data.length, pucSignature);
            checkRet("internalSignECC next", ret);
        } else {
            checkRet("internalSignECC", ret);
        }
//        GMSSLByteArrayUtils.printHexBinary(logger, "sign r", pucSignature.getR());
//        GMSSLByteArrayUtils.printHexBinary(logger, "sign s", pucSignature.getS());
        byte[] r = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucSignature.getR());
        byte[] s = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucSignature.getS());
        return new SdfECCSignature(r, s);
    }

    @Override
    public SdfECCPublicKey exportSignPublicKeyEcc(int index) throws SdfSDKException {
        EccPublicKey pucPublicKey = new EccPublicKey();
        int ret = sdfApi.exportSignPublicKeyEcc(ses[0], index, pucPublicKey);
        checkRet("exportSignPublicKeyEcc", ret);
//        GMSSLByteArrayUtils.printHexBinary(logger, "exportSignPublicKeyEcc x", pucPublicKey.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "exportSignPublicKeyEcc y", pucPublicKey.getY());
        return SdfECCPublicKey.getInstanceFilterFoot(pucPublicKey.getX(), pucPublicKey.getY());
    }

    @Override
    public void externalVerifyECC(SdfECCPublicKey publicKey, byte[] data, SdfECCSignature sdfECCSignature) throws SdfSDKException {
        byte[] r = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCSignature.getR());
        byte[] s = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCSignature.getS());
//        GMSSLByteArrayUtils.printHexBinary(logger, "verify r", r);
//        GMSSLByteArrayUtils.printHexBinary(logger, "verify s", s);
        EccSignature pucSignature = new EccSignature(r, s);
        byte[] x = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(publicKey.getX());
        byte[] y = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(publicKey.getY());
//        GMSSLByteArrayUtils.printHexBinary(logger, "verify pub x", x);
//        GMSSLByteArrayUtils.printHexBinary(logger, "verify pub y", y);
        logger.info("data length " + data.length);
        EccPublicKey pucPublicKey = new EccPublicKey(GMSSLX509Utils.ECC_KEY_BITS, x, y);
        int ret = sdfApi.externalVerifyEcc(ses[0], AlgId.SGD_SM2, pucPublicKey, data, data.length, pucSignature);
        checkRet("externalVerifyECC", ret);
    }

    @Override
    public SdfECCPublicKey exportEncPublicKeyEcc(int index) throws SdfSDKException {
        EccPublicKey pucPublicKey = new EccPublicKey();
        int ret = sdfApi.exportEncPublicKeyEcc(ses[0], index, pucPublicKey);
        checkRet("exportEncPublicKeyEcc", ret);
        return SdfECCPublicKey.getInstanceFilterFoot(pucPublicKey.getX(), pucPublicKey.getY());
    }

    @Override
    public SdfECCCipher externalEncryptECC(SdfECCPublicKey sdfECCPublicKey, byte[] data) throws SdfSDKException {
        EccCipher pucEncData = new EccCipher();
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, sdfECCPublicKey);
        int ret = sdfApi.externalEncryptEcc(ses[0], AlgId.SGD_SM2, pucPublicKey, data, data.length, pucEncData);
        checkRet("externalEncryptECC", ret);
//        GMSSLByteArrayUtils.printHexBinary(logger, "ecc cipher x", pucEncData.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "ecc cipher y", pucEncData.getY());
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucEncData.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucEncData.getY());
        return new SdfECCCipher(x, y, pucEncData.getM(), pucEncData.getL(), pucEncData.getC());
    }

    @Override
    public byte[] internalDecryptECC(int index, byte[] password, int len, SdfECCCipher sdfECCCipher) throws SdfSDKException {
        byte[] x = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCCipher.getX());
        byte[] y = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCCipher.getY());
        EccCipher pucEncData = new EccCipher(x, y, sdfECCCipher.getM(), sdfECCCipher.getL(), sdfECCCipher.getC());
//        GMSSLByteArrayUtils.printHexBinary(logger, "ecc cipher x", pucEncData.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "ecc cipher y", pucEncData.getY());
        byte[] pucData = new byte[len];
        int[] puiDataLength = {len};
        int ret = sdfApi.internalDecryptEcc(ses[0], index, AlgId.SGD_SM2, pucEncData, pucData, puiDataLength);
        if (ret == SdfApiCode.SDR_PARDENY) {
            this.release();
            this.init();
            this.getPrivateKeyAccessRight(index, password);
            ret = sdfApi.internalDecryptEcc(ses[0], index, AlgId.SGD_SM2, pucEncData, pucData, puiDataLength);
            checkRet("internalDecryptEcc next", ret);
        } else {
            checkRet("internalDecryptEcc", ret);
        }
        return pucData;
    }

    @Override
    public SdfECCCipher generateKeyWithEpkEcc(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException {
        SdfSymmetricKeyHandle keyHandle = generateKeyWithEpkEccKeyHandle(sdfECCPublicKey);
        return keyHandle.getCipherKey();
    }

    @Override
    public long[] generateKeyWithEpkEccHandle(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException {
        SdfSymmetricKeyHandle keyHandle = generateKeyWithEpkEccKeyHandle(sdfECCPublicKey);
        return keyHandle.getHandle();
    }

    @Override
    public SdfSymmetricKeyHandle generateKeyWithEpkEccKeyHandle(SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException {
        long[] phKeyHandle = {0};
        int uiKeyBits_SES = 128;
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, sdfECCPublicKey);
        EccCipher pucKey = new EccCipher();
        int ret = sdfApi.generateKeyWithEpkEcc(ses[0], uiKeyBits_SES, AlgId.SGD_SM2, pucPublicKey, pucKey, phKeyHandle);
        checkRet("generateKeyWithEpkEcc", ret);
//        GMSSLByteArrayUtils.printHexBinary(logger, "enc data x", pucKey.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "enc data y", pucKey.getY());
        byte[] x = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucKey.getX());
        byte[] y = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(pucKey.getY());
        SdfECCCipher cipher = new SdfECCCipher(x, y, pucKey.getM(), pucKey.getL(), pucKey.getC());
        return new SdfSymmetricKeyHandle(cipher, phKeyHandle);
    }

    @Override
    public long[] importKeyWithIskEcc(int uiIskIndex, byte[] password, SdfECCCipher sdfECCCipher) throws SdfSDKException {
        long[] phKeyHandle = {0};
        byte[] x = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCCipher.getX());
        byte[] y = GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(sdfECCCipher.getY());
        EccCipher pucKey = new EccCipher(x, y, sdfECCCipher.getM(), sdfECCCipher.getL(), sdfECCCipher.getC());
        int ret = sdfApi.importKeyWithIskEcc(ses[0], uiIskIndex, pucKey, phKeyHandle);
        if (ret == SdfApiCode.SDR_PARDENY) {
            this.release();
            this.init();
            this.getPrivateKeyAccessRight(uiIskIndex, password);
            ret = sdfApi.importKeyWithIskEcc(ses[0], uiIskIndex, pucKey, phKeyHandle);
            checkRet("internalSignECC next", ret);
        } else {
            checkRet("internalSignECC", ret);
        }
        checkRet("importKeyWithIskEcc", ret);
        return phKeyHandle;
    }

    @Override
    public long[] importKey(byte[] pucKey) throws SdfSDKException {
        long[] phKeyHandle = {0};
        int ret = sdfApi.importKey(ses[0], pucKey, pucKey.length, phKeyHandle);
        checkRet("importKey", ret);
        return phKeyHandle;
    }

    @Override
    public byte[] encrypt(long[] phKeyHandle, SdfAlgIdSymmetric sdfAlgIdBlockCipher, byte[] iv, byte[] pucData) throws SdfSDKException {
        byte[] pucEncData = new byte[pucData.length];
        int[] puiEncDataLength = {pucData.length};
        byte[] pucIv = null;
        if (iv != null) {
            pucIv = new byte[iv.length];
            System.arraycopy(iv, 0, pucIv, 0, iv.length);
        }
        int ret = sdfApi.encrypt(ses[0], phKeyHandle[0], sdfAlgIdBlockCipher.getId(), pucIv, pucData, pucData.length, pucEncData, puiEncDataLength);
        checkRet("encrypt", ret);
        return pucEncData;
    }

    @Override
    public byte[] decrypt(long[] phKeyHandle, SdfAlgIdSymmetric sdfAlgIdBlockCipher, byte[] iv, byte[] pucEncData) throws SdfSDKException {
        byte[] pucData = new byte[pucEncData.length];
        int[] puiDataLength = {pucEncData.length};
        byte[] pucIv = null;
        if (iv != null) {
            pucIv = new byte[iv.length];
            System.arraycopy(iv, 0, pucIv, 0, iv.length);
        }
        int ret = sdfApi.decrypt(ses[0], phKeyHandle[0], sdfAlgIdBlockCipher.getId(), pucIv, pucEncData, pucEncData.length, pucData, puiDataLength);
        checkRet("decrypt", ret);
        return pucData;
    }

    @Override
    public void destroyKey(long[] phKeyHandle) throws SdfSDKException {
        int ret = sdfApi.destroyKey(ses[0], phKeyHandle[0]);
        checkRet("destroyKey", ret);
    }

    @Override
    public void hashInit(SdfAlgIdHash sdfAlgIdHash) throws SdfSDKException {
        int ret = sdfApi.hashInit(ses[0], sdfAlgIdHash.getId(), null, null, 0);
        checkRet("hashInit", ret);
    }

    @Override
    public void hashInit(SdfAlgIdHash sdfAlgIdHash, byte[] pucID, SdfECCPublicKey sdfECCPublicKey) throws SdfSDKException {
        EccPublicKey pucPublicKey = PcieSdfSDKUtils.generateEccPublicKey(logger, sdfECCPublicKey);
        int ret = sdfApi.hashInit(ses[0], sdfAlgIdHash.getId(), pucPublicKey, pucID, pucID.length);
        checkRet("hashInit", ret);
    }

    @Override
    public void hashUpdate(byte[] data) throws SdfSDKException {
        int ret = sdfApi.hashUpdate(ses[0], data, data.length);
        checkRet("hashUpdate", ret);
    }

    @Override
    public byte[] hashFinal(int digestLen) throws SdfSDKException {
        byte[] pucHash = new byte[digestLen];
        int[] pucHashLen = {pucHash.length};
        int ret = sdfApi.hashFinal(ses[0], pucHash, pucHashLen);
        checkRet("hashFinal", ret);
        return pucHash;
    }

    @Override
    public void calculateMac(int uiKeyIndex) throws SdfSDKException {
        int ret;
        int uiKeyBits_SES = 128;

        long[] phKeyHandle = {0};
        EccCipher pucKey = new EccCipher();

        //对称算法运算类函数缓冲区指针定义
        byte[] pucIV = new byte[1024];
        byte[] pucEncDate = new byte[1024];
        int[] pucEncDateLen = {1024};

        //要加密的数据 和数据长度
        String pucDate_t = "12345678123456781234567812345678";
        byte[] pucDate = pucDate_t.getBytes();

        ret = sdfApi.generateKeyWithIpkEcc(ses[0], uiKeyIndex, uiKeyBits_SES, pucKey, phKeyHandle);
        checkRet("generateKeyWithIpkEcc", ret);
        ret = sdfApi.calculateMac(ses[0], phKeyHandle[0], AlgId.SGD_SM4_MAC, pucIV, pucDate, pucDate.length, pucEncDate, pucEncDateLen);
        checkRet("calculateMac", ret);
    }

    @Override
    public byte[] sm3Hmac(byte[] dataIn, byte[] key) throws SdfSDKException {
        int ret;
        byte[] mac = new byte[32];
        ret = sdfApi.sm3Hmac(ses[0], dataIn, dataIn.length, key, key.length, mac);
        checkRet("sm3HmacDemo", ret);
        return mac;
    }

    public void getDeviceInfo() throws SdfSDKException {
        DeviceInfo devinfo = new DeviceInfo();
        int ret = sdfApi.getDeviceInfo(ses[0], devinfo);
        checkRet("getDeviceInfo", ret);
        System.out.println(devinfo.toString());
    }

    @Override
    public SdfRSAPublicKey exportSignPublicKeyRsa(int index) throws SdfSDKException {
        return null;
    }

    @Override
    public SdfRSAPublicKey exportEncPublicKeyRsa(int index) throws SdfSDKException {
        return null;
    }

    @Override
    public SdfRsaKeyPair generateKeyPairRsa(int bits) throws SdfSDKException {
        return null;
    }

    @Override
    public byte[] externalPublicKeyOperationRsa(SdfRSAPublicKey sdfRSAPublicKey, byte[] data) throws SdfSDKException
    {
        return new byte[0];
    }

    @Override
    public byte[] internalPrivateKeyOperationRsa(int index, byte[] password, byte[] data) throws SdfSDKException {
        return new byte[0];
    }
}
