package com.xdja.pki.gmssl.sdf.pcie;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.*;
import org.bouncycastle.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;

import static com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils.hexEncode;

public class PcieSdfSDKDemo {
    private static Logger logger = LoggerFactory.getLogger(PcieSdfSDKDemo.class.getName());
    //设备访问密码
    private static byte[] pwd = "xdja1234".getBytes();
    private static int sm2Index = 2;// sm2 签名、加密公私钥 可以用一个容器

    public static void main(String[] args) throws Exception {

        if (args.length == 0 || args[0] == null) {
            System.out.println("Please insert method, Example: all. getinfo. exportenc. exportsign. random, sm1enc. gensm2key.  sm2sign. sm2enc. sm4enc. sm3hash. sm3hmac. sm4mac,sm2signTest,sm2encTest");
            return;
        }

        System.out.println("You want to test: " + args[0]);
        PcieSdfSDK sdfSDK = new PcieSdfSDK();

        switch (args[0]) {
            case "all":
                System.out.println("*************** getDeviceInfo ***************");
                getDeviceInfo(sdfSDK);
                System.out.println("*************** random ***************");
                random(sdfSDK);
                System.out.println("*************** generateKeyPairEcc ***************");
                sdfSDK.generateKeyPairEcc();
                System.out.println("*************** exportsign ***************");
                exportSignKey(sdfSDK, args);
                System.out.println("*************** exportenc ***************");
                exportEncKey(sdfSDK, args);
                System.out.println("*************** sm2SignDemo ***************");
                sm2SignDemo(sdfSDK);
                System.out.println("*************** sm2EncDemo ***************");
                sm2EncDemo(sdfSDK);
                System.out.println("*************** sm4EncDemo ***************");
                sm4EncDemo(sdfSDK);
                System.out.println("*************** sm3HashDemoWithPublicKey ***************");
                sm3HashDemoWithPublicKey(sdfSDK);
                System.out.println("*************** sm3HashDemo ***************");
                sm3HashDemo(sdfSDK);
                System.out.println("*************** sm3HmacDemo ***************");
                sm3HmacDemo(sdfSDK);
                System.out.println("*************** sm4MacDemo ***************");
                sm4MacDemo(sdfSDK);
                break;
            case "getinfo":
                //测试生成 随机数
                getDeviceInfo(sdfSDK);
                break;
            case "random":
                //测试生成 随机数
                random(sdfSDK);
                break;
            case "gensm2key":
                //生成SM2公私钥对
                generateKeyPairEcc(sdfSDK);
                break;
            case "exportsign":
                exportSignKey(sdfSDK, args);
                break;
            case "exportenc":
                exportEncKey(sdfSDK, args);
                break;
            case "sm1enc":
                sm1EncDemo(sdfSDK);
                break;
            case "sm2sign":
                sm2SignDemo(sdfSDK);
                break;
            case "sm2enc":
                sm2EncDemo(sdfSDK);
                break;
            case "sm4enc":
                sm4EncDemo(sdfSDK);
                break;
            case "sm3hash":
                System.out.println("*************** sm3HashDemoWithPublicKey ***************");
                sm3HashDemoWithPublicKey(sdfSDK);
                System.out.println("*************** sm3HashDemo ***************");
                sm3HashDemo(sdfSDK);
                break;
            case "sm3hmac":
                sm3HmacDemo(sdfSDK);
                break;
            case "sm4mac":
                sm4MacDemo(sdfSDK);
                break;
            case "sm2signTest":
                sm2SignDemoTest(sdfSDK);
                break;
            case "sm2encTest":
                sm2EncDemoTest(sdfSDK);
                break;
            default:
                System.out.println("不支持的命令！");
                break;
        }
    }

    /**
     * 获取设备信息
     **/
    private static void getDeviceInfo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        logger.info("*************** getDeviceInfo ***************");
        sdfSDK.init();
        sdfSDK.getDeviceInfo();
        sdfSDK.release();
    }

    /**
     * 导出sm2签名公钥
     **/
    private static void exportSignKey(PcieSdfSDK sdfSDK, String[] args) throws SdfSDKException {
        logger.info("*************** exportSM2Key ***************");
        sdfSDK.init();
        int index = sm2Index;
        try {
            index = Integer.valueOf(args[1]);
        } catch (Exception e) {
            //do nothing
        }
        System.out.println(index);
        SdfECCPublicKey signPublicKeyEcc = sdfSDK.exportSignPublicKeyEcc(index);
        GMSSLByteArrayUtils.printHexBinary(logger, "signPublicKeyEcc getX()", signPublicKeyEcc.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "signPublicKeyEcc getY()", signPublicKeyEcc.getY());

        sdfSDK.release();
    }

    /**
     * 导出sm2加密公钥
     **/
    private static void exportEncKey(PcieSdfSDK sdfSDK, String[] args) throws SdfSDKException {
        logger.info("*************** exportSM2Key ***************");
        sdfSDK.init();
        int index = sm2Index;
        try {
            index = Integer.valueOf(args[1]);
        } catch (Exception e) {
            //do nothing
        }

        System.out.println(index);

        SdfECCPublicKey eccPublicKey = sdfSDK.exportEncPublicKeyEcc(index);
        GMSSLByteArrayUtils.printHexBinary(logger, "eccPublicKey getX()", eccPublicKey.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "eccPublicKey getY()", eccPublicKey.getY());

        sdfSDK.release();
    }


    /**
     * 生成密钥对
     **/
    private static void generateKeyPairEcc(PcieSdfSDK sdfSDK) throws SdfSDKException {
        logger.info("*************** generateKeyPairEccDemo ***************");
        sdfSDK.init();
        SdfECCKeyPair sdfECCKeyPair = sdfSDK.generateKeyPairEcc();
        SdfECCPublicKey sdfECCPublicKey = sdfECCKeyPair.getSdfECCPublicKey();

        GMSSLByteArrayUtils.printHexBinary(logger, "SdfECCPublicKey getX  ", sdfECCPublicKey.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "SdfECCPublicKey getY  ", sdfECCPublicKey.getY());
        sdfSDK.release();
    }


    private static void random(PcieSdfSDK sdfSDK) throws SdfSDKException {
        logger.info("*************** randomDemo ***************");
        sdfSDK.init();
        int len = 28;
        byte[] out = sdfSDK.generateRandom(len);
        GMSSLByteArrayUtils.printHexBinary(logger, "generateRandom out", out);
        sdfSDK.release();
    }

    /**
     * SM1 加密解密操作
     **/
    public static void sm1EncDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        byte[] key = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] data = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] cipher = GMSSLByteArrayUtils.hexDecode("9203a010b50c4e3ca2897308a9d48c2e");
        logger.info(" sm1 CBC encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM1_CBC, key, data, cipher);
        logger.info(" sm1 ECB encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM1_ECB, key, data, cipher);
    }

    /**
     * SM2公私钥对 进行 签名验签
     **/
    private static void sm2SignDemoTest(PcieSdfSDK sdfSDK) throws SdfSDKException, NoSuchProviderException, KeyStoreException, InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException, CertificateException {
        logger.info("*************** sm2SignDemoTest ***************");
        //设备访问密码
        sdfSDK.init();

        byte[] data = "12345678123456781234567812345678".getBytes();
        KeyPair keyPair = GMSSLX509Utils.generateSM2KeyPair();

        //获取私钥访问权限
        sdfSDK.getPrivateKeyAccessRight(sm2Index, pwd);
        //容器内进行签名
        SdfECCSignature sdfECCSignature = sdfSDK.internalSignECC(sm2Index, pwd, data);
        GMSSLByteArrayUtils.printHexBinary(logger, "sdf sign r", sdfECCSignature.getR());
        GMSSLByteArrayUtils.printHexBinary(logger, "sdf sign s", sdfECCSignature.getS());
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportSignPublicKeyEcc(sm2Index);
        //导入外部公钥进行验签
        sdfSDK.externalVerifyECC(sdfECCPublicKey, data, sdfECCSignature);
        logger.info("sm2Sign is success");
        //释放私钥访问权限
        sdfSDK.releasePrivateKeyAccessRight(sm2Index);

        sdfSDK.release();
    }

    /**
     * 导入SM2公私钥对 并进行 签名验签
     **/
    private static void sm2SignDemo(PcieSdfSDK sdfSDK) throws SdfSDKException, NoSuchProviderException, KeyStoreException, InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException, CertificateException {
        logger.info("*************** sm2SignDemo ***************");
        //设备访问密码
        sdfSDK.init();

        byte[] data = "12345678123456781234567812345678".getBytes();
        KeyPair keyPair = GMSSLX509Utils.generateSM2KeyPair();

        sdfSDK.importSM2PublicKey(keyPair.getPublic(), sm2Index, true);
        //导入sm2私钥
        sdfSDK.importSM2PrivateKey(keyPair.getPrivate(), sm2Index, true);
        //获取私钥访问权限
        sdfSDK.getPrivateKeyAccessRight(sm2Index, pwd);
        //容器内进行签名
        SdfECCSignature sdfECCSignature = sdfSDK.internalSignECC(sm2Index, pwd, data);
        GMSSLByteArrayUtils.printHexBinary(logger, "sdf sign r", sdfECCSignature.getR());
        GMSSLByteArrayUtils.printHexBinary(logger, "sdf sign s", sdfECCSignature.getS());
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportSignPublicKeyEcc(sm2Index);
        //导入外部公钥进行验签
        sdfSDK.externalVerifyECC(sdfECCPublicKey, data, sdfECCSignature);
        logger.info("sm2Sign is success");
        //释放私钥访问权限
        sdfSDK.releasePrivateKeyAccessRight(sm2Index);

        sdfSDK.release();
    }

    /**
     * SM2公私钥对 进行 加密解密操作
     **/
    private static void sm2EncDemoTest(PcieSdfSDK sdfSDK) throws Exception {
        logger.info("*************** sm2EncDemoTest ***************");

        sdfSDK.init();
        //原始数据
        byte[] data = "12345678123456781234567812345678".getBytes();

        KeyPair keyPair = GMSSLX509Utils.generateSM2KeyPair();

        //导出获取ECCpublicKey
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportEncPublicKeyEcc(sm2Index);
        //导入外部公钥进行sm2加密
        SdfECCCipher sdfECCCipher = sdfSDK.externalEncryptECC(sdfECCPublicKey, data);
        //获取私钥访问权限
        sdfSDK.getPrivateKeyAccessRight(sm2Index, pwd);
        //解密
        byte[] plaintext = sdfSDK.internalDecryptECC(sm2Index, pwd,sdfECCCipher.getC().length, sdfECCCipher);
        //释放私钥访问权限
        sdfSDK.releasePrivateKeyAccessRight(sm2Index);
        sdfSDK.release();

        if (Arrays.equals(plaintext, data)) {
            logger.info("decrypt data is equals with plain data");
        } else {
            logger.error("decrypt data is not equals with plain data");
        }
    }

    /**
     * 导入SM2公私钥对 并进行 加密解密操作
     **/
    private static void sm2EncDemo(PcieSdfSDK sdfSDK) throws Exception {
        logger.info("*************** sm2EncDemo ***************");

        sdfSDK.init();
        //原始数据
        byte[] data = "12345678123456781234567812345678".getBytes();

        KeyPair keyPair = GMSSLX509Utils.generateSM2KeyPair();
        //导入公钥
        sdfSDK.importSM2PublicKey(keyPair.getPublic(), sm2Index, false);
        //导入解密私钥
        sdfSDK.importSM2PrivateKey(keyPair.getPrivate(), sm2Index, false);

        //导出获取ECCpublicKey
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportEncPublicKeyEcc(sm2Index);
        //导入外部公钥进行sm2加密
        SdfECCCipher sdfECCCipher = sdfSDK.externalEncryptECC(sdfECCPublicKey, data);
        //获取私钥访问权限
        sdfSDK.getPrivateKeyAccessRight(sm2Index, pwd);
        //解密
        byte[] plaintext = sdfSDK.internalDecryptECC(sm2Index, pwd,sdfECCCipher.getC().length, sdfECCCipher);
        //释放私钥访问权限
        sdfSDK.releasePrivateKeyAccessRight(sm2Index);
        sdfSDK.release();

        if (Arrays.equals(plaintext, data)) {
            logger.info("decrypt data is equals with plain data");
        } else {
            logger.error("decrypt data is not equals with plain data");
        }
    }

    /**
     * SM4 加密解密
     **/
    public static void sm4EncDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        byte[] key = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] data = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] cipher = GMSSLByteArrayUtils.hexDecode("681edf34d206965e86b3e94f536e4246");
        logger.info(" sm4 CBC encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM4_CBC, key, data, cipher);
        logger.info(" sm4 ECB encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM4_ECB, key, data, cipher);
    }


    /**
     * SM3 HASH摘要算法 withNOPublicKey
     **/
    public static void sm3HashDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        sdfSDK.init();
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3);
        sdfSDK.hashUpdate("111".getBytes());
        byte[] out = sdfSDK.hashFinal(32);
        sdfSDK.release();
        GMSSLByteArrayUtils.printHexBinary(logger, "out", out);
    }

    /**
     * SM3 HASH摘要算法 withPublicKey SM2签名算法预处理过程数据 并测试输出是否符合规范
     **/
    public static void sm3HashDemoWithPublicKey(PcieSdfSDK sdfSDK) throws SdfSDKException {

        byte[] xa = GMSSLByteArrayUtils.hexDecode("09F9DF31 1E5421A1 50DD7D16 1E4BC5C6 72179FAD 1833FC07 6BB08FF3 56F35020");
        byte[] ya = GMSSLByteArrayUtils.hexDecode("CCEA490C E26775A5 2DC6EA71 8CC1AA60 0AED05FB F35E084A 6632F607 2DA9AD13");

        byte[] ID_A = Strings.toByteArray("1234567812345678");
        byte[] msg = Strings.toByteArray("message digest");

        byte[] expecteds = GMSSLByteArrayUtils.hexDecode("F0B43E94 BA45ACCA ACE692ED 534382EB 17E6AB5A 19CE7B31 F4486FDF C0D28640");

        GMSSLByteArrayUtils.printHexBinary(logger, "xa", xa);
        GMSSLByteArrayUtils.printHexBinary(logger, "ya", ya);
        GMSSLByteArrayUtils.printHexBinary(logger, "ID_A", ID_A);
        GMSSLByteArrayUtils.printHexBinary(logger, "message digest", msg);

        sdfSDK.init();
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3, ID_A, SdfECCPublicKey.getInstance(new BigInteger(xa), new BigInteger(ya)));
        sdfSDK.hashUpdate(msg);
        byte[] actuals = sdfSDK.hashFinal(32);
        sdfSDK.release();

        GMSSLByteArrayUtils.printHexBinary(logger, "expecteds", expecteds);
        GMSSLByteArrayUtils.printHexBinary(logger, "actuals", actuals);
        if (Arrays.equals(expecteds, actuals)) {
            logger.info("expecteds data is equals with actuals data");
        } else {
            logger.error("expecteds data is not equals with actuals data");
        }
    }

    public static void sm3HmacDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
//        MAC计算	SDF_CalculateMAC
        sdfSDK.init();
        byte[] data = "12345678123456781234567812345678".getBytes();
        byte[] key = "12345678123456781234567812345678".getBytes();
        byte[] mac = sdfSDK.sm3Hmac(data, key);
        GMSSLByteArrayUtils.printHexBinary(logger, "sdfSDK.sm3Hmac", mac);
        sdfSDK.release();
    }

    public static void sm4MacDemo(PcieSdfSDK sdfSDK) throws SdfSDKException {
        sdfSDK.init();
//        MAC计算	SDF_CalculateMAC
        sdfSDK.calculateMac(1);
        sdfSDK.release();
    }

    private static void symmetricEncDemo(PcieSdfSDK sdfSDK, SdfAlgIdSymmetric symmetric, byte[] key, byte[] data, byte[] cipher) throws SdfSDKException {

        sdfSDK.init();

        //句柄  导入key获取
        long[] handle = sdfSDK.importKey(key);
        //初始化向量
        byte[] iv = GMSSLByteArrayUtils.hexDecode("00000000000000000000000000000000");
        //加密数据
        byte[] c = sdfSDK.encrypt(handle, symmetric, iv, data);
        GMSSLByteArrayUtils.printHexBinary(logger, "encrypt c", c);

        byte[] p = sdfSDK.decrypt(handle, symmetric, iv, c);
        GMSSLByteArrayUtils.printHexBinary(logger, "decrypt p", p);

        if (Arrays.equals(data, p)) {
            logger.info("decrypt data is equals with plain data");
        } else {
            logger.error("decrypt data is not equals with plain data");
        }

        if (Arrays.equals(cipher, c)) {
            logger.info("encrypt data is equals with expect cipher data");
        } else {
            logger.error("encrypt data is not equals with expect cipher data");
        }
        sdfSDK.release();
    }
}
