package com.xdja.pcie;

import com.xdja.pcie.base.*;
import com.xdja.pki.gmssl.jni.NativeUtils;


import java.io.IOException;

/**
 * 密码设备应用接口(依据 国密标准GM/T 0018-2012)
 *
 * @author wyf
 */
public class SDFAPI {

    static {
//       System.loadLibrary("sdf_pcie");
        try {
            //NativeUtils.loadLibraryFromJar("/armlibxdjapcieJNI.so");
            NativeUtils.loadLibraryFromJar("/libxdjapcieJNI.so");
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 打开设备
     * <p/>
     * 描述：打开密码设备。
     *
     * @param phDeviceHandle [out]设备句柄，获取方式：phDeviceHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int openDevice(long[] phDeviceHandle);

    /**
     * 关闭设备
     * <p/>
     * 描述：关闭密码设备。
     *
     * @param hDeviceHandle [in]已打开的设备句柄
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int closeDevice(long hDeviceHandle);

    /**
     * 创建会话
     * <p/>
     * 描述：创建与密码设备的会话。
     *
     * @param hDeviceHandle   [in]已打开的设备句柄
     * @param phSessionHandle [out]返回与密码设备建立的新会话句柄，获取方式：phSessionHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int openSession(long hDeviceHandle, long[] phSessionHandle);

    /**
     * 关闭会话
     * <p>
     * 描述：关闭与密码设备已建立的会话，并释放相关资源。
     *
     * @param hSessionHandle [in]与密码设备已建立的会话句柄
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int closeSession(long hSessionHandle);

    /**
     * 获取设备信息
     * <p/>
     * 描述：获取密码设备能力描述。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pstDeviceInfo  [out]设备能力描述信息，内容及格式件设备信息定义
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int getDeviceInfo(long hSessionHandle, DeviceInfo pstDeviceInfo);

    /**
     * 产生随机数
     * <p/>
     * 描述：获取指定长度的随机数。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param hSessionHandle [in]欲获取的随机数长度
     * @param pucRandom      [out]随机数
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateRandom(long hSessionHandle, int uiLength, byte[] pucRandom);

    /**
     * 获取私钥使用权限
     * <p/>
     * 描述：获取密码设备内部存储的指定索引私钥的使用权。
     * <p/>
     * 备注：本标准涉及密码设备存储的密钥对索引值的起始索引为1，最大为n，密码设备的实际存储容量决定n的值。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储私钥的索引值
     * @param pucPassword    [in]使用私钥权限的标识码
     * @param uiPwdLength    [in]私钥访问控制码长度
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int getPrivateKeyAccessRight(long hSessionHandle, int uiKeyIndex, byte[] pucPassword, int uiPwdLength);

    /**
     * 释放私钥使用权限
     * <p/>
     * 描述：释放密码设备存储的指定索引私钥的使用授权。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储私钥的索引值
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int releasePrivateKeyAccessRight(long hSessionHandle, int uiKeyIndex);

    /**
     * 导出RSA签名公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的签名公钥。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储私钥的索引值
     * @param pucPublicKey   [out]RSA公钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exportSignPublicKeyRSA(long hSessionHandle, int uiKeyIndex, RSAPublicKey pucPublicKey);

    /**
     * 导出RSA加密公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的加密公钥。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储私钥的索引值
     * @param pucPublicKey   [out]RSA公钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exportEncPublicKeyRSA(long hSessionHandle, int uiKeyIndex, RSAPublicKey pucPublicKey);

    /**
     * 产生RSA密钥对并输出
     * <p/>
     * 描述：请求密码设备产生指定模长的RSA密钥对。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyBits      [in]指定密钥模长
     * @param pucPublicKey   [out]RSA公钥对象
     * @param pucPrivateKey  [out]RSA私钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyPairRSA(long hSessionHandle, int uiKeyBits, RSAPublicKey pucPublicKey, RSAPrivateKey pucPrivateKey);

    /**
     * 生成会话密钥并用内部RSA公钥加密输出
     * <p/>
     * 描述：生成会话密钥并用指定索引的内部加密公钥加密输出，同时返回密钥句柄。
     * <p/>
     * 备注：公钥加密数据时填充方式按照PKCS#1 v1.5 的要求进行。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIpkIndex     [in]密码设备内部存储公钥的索引值
     * @param uiKeyBits      [in]指定产生的会话密钥长度
     * @param pucKey         [out]会话密钥密文
     * @param puiKeyLength   [out]密钥密文长度，获取方式：puiKeyLength[0]
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithIPKRSA(long hSessionHandle, int uiIpkIndex, int uiKeyBits, byte[] pucKey, int[] puiKeyLength, long[] phKeyHandle);

    /**
     * 生成会话密钥并用外部RSA公钥加密输出
     * <p/>
     * 描述：生成会话密钥并用外部公钥加密输出，同时返回密钥句柄。
     * <p/>
     * 备注：公钥加密数据时填充方式按照PKCS#1 v1.5 的要求进行。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyBits      [in]指定产生的会话密钥长度
     * @param pucPublicKey   [in]输入的外部RSA公钥对象
     * @param pucKey         [out]会话密钥密文
     * @param puiKeyLength   [out]会话密钥密文长度，获取方式：puiKeyLength[0]
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithEPKRSA(long hSessionHandle, int uiKeyBits, RSAPublicKey pucPublicKey, byte[] pucKey, int[] puiKeyLength, long[] phKeyHandle);

    /**
     * 导入会话密钥并使用内部RSA私钥解密
     * <p/>
     * 描述：导入会话密钥并用内部私钥解密，同时返回密钥句柄。
     * <p/>
     * 备注：填充方式与公钥加密时相同。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIskIndex     [in]密码设备内部存储加密私钥的索引值，对应于加密时的公钥
     * @param pucKey         [in]会话密钥密文
     * @param puiKeyLength   [in]会话密钥密文长度
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int importKeyWithISKRSA(long hSessionHandle, int uiIskIndex, byte[] pucKey, int[] puiKeyLength, long[] phKeyHandle);

    /**
     * 基于RSA算法的数字信封转换
     * <p/>
     * 描述：将由内部加密公钥加密的会话密钥转换为由外部指定的公钥加密，可用于数字信封转换。
     *
     * @param hSessionHandle  [in]与设备建立的会话句柄
     * @param uiKeyIndex      [in]密码设备存储的内部RSA密钥对索引值
     * @param pucPublickey    [in]外部RSA公钥对象
     * @param pucDeInput      [in]会话密钥密文
     * @param uiDeInputLength [in]会话密钥密文长度
     * @param pucDeOutput     [out]会话密钥密文(由外部RSA公钥加密)
     * @param puiDeLength     [out]会话密钥密文长度
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exchangeDigitEnvelopeBaseOnRSA(long hSessionHandle, int uiKeyIndex, RSAPublicKey pucPublickey, byte[] pucDeInput, int uiDeInputLength, byte[] pucDeOutput, int[] puiDeLength);

    /**
     * 导出ECC签名公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的签名公钥。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储的ECC密钥对索引值
     * @param pucPublicKey   [out]ECC公钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exportSignPublicKeyECC(long hSessionHandle, int uiKeyIndex, ECCPublicKey pucPublicKey);

    /**
     * 导出ECC加密公钥
     * <p/>
     * 描述：导出密码设备内部存储的指定索引位置的加密公钥。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储的ECC密钥对索引值
     * @param pucPublicKey   [out]ECC公钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exportEncPublicKeyECC(long hSessionHandle, int uiKeyIndex, ECCPublicKey pucPublicKey);

    /**
     * 产生ECC密钥对并输出
     * <p/>
     * 描述：请求密码设备产生指定类型和模长的ECC密钥对。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]指定算法标识
     * @param uiKeyBits      [in]指定密钥长度
     * @param pucPublicKey   [out]ECC公钥对象
     * @param pucPrivateKey  [out]ECC私钥对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyPairECC(long hSessionHandle, int uiAlgId, int uiKeyBits, ECCPublicKey pucPublicKey, ECCPrivateKey pucPrivateKey);

    /**
     * 生成会话密钥并用内部ECC公钥加密输出
     * <p/>
     * 描述：生成会话密钥并用指定索引的内部ECC加密公钥加密输出，同时返回密钥句柄。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIpkIndex     [in]密码设备内部存储公钥的索引值
     * @param uiKeyBits      [in]指定产生的会话密钥长度
     * @param pucKey         [out]会话密钥密文
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithIPKECC(long hSessionHandle, int uiIpkIndex, int uiKeyBits, ECCCipher pucKey, long[] phKeyHandle);

    /**
     * 生成会话密钥并用外部ECC公钥加密输出
     * <p/>
     * 描述：生成会话密钥并用外部ECC加密公钥加密输出，同时返回密钥句柄。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyBits      [in]指定产生的会话密钥长度
     * @param uiAlgId        [in]外部ECC公钥的算法标识
     * @param pucPublicKey   [in]输入的外部ECC公钥对象
     * @param pucKey         [out]会话密钥密文
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithEPKECC(long hSessionHandle, int uiKeyBits, int uiAlgId, ECCPublicKey pucPublicKey, ECCCipher pucKey, long[] phKeyHandle);

    /**
     * 导入会话密钥并用内部ECC私钥解密
     * <p/>
     * 描述：导入会话密钥并用内部ECC加密私钥解密，同时返回密钥句柄。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIskIndex     [in]密码设备内部存储加密私钥的索引值，对应于加密时的公钥
     * @param pucKey         [in]会话密钥密文
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int importKeyWithISKECC(long hSessionHandle, int uiIskIndex, ECCCipher pucKey, long[] phKeyHandle);

    /**
     * 生成密钥协商参数并输出
     * <p/>
     * 描述：使用ECC密钥协商算法，为计算会话密钥而产生协商参数，同时返回指定索引位置的ECC公钥、临时ECC密钥对的公钥及协商句柄。
     * <p/>
     * 备注：为协商会话密钥，协商的发起方应首先调用本函数。
     *
     * @param hSessionHandle         [in]与设备建立的会话句柄
     * @param uiIskIndex             [in]密码设备内部存储加密私钥的索引值，该私钥用于参与密钥协商
     * @param uiKeyBits              [in]要求协商的密钥长度
     * @param pucSponsorId           [in]参与密钥协商的发起方Id值
     * @param uiSponsorIdLength      [in]发起方Id长度
     * @param pucSponsorPublicKey    [out]发起方ECC公钥结构
     * @param pucSponsorTmpPublicKey [out]发起方临时ECC公钥结构
     * @param phAgreementHandle      [out]协商句柄，用于计算协商密钥，获取方式：phAgreementHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateAgreementDataWithECC(long hSessionHandle, int uiIskIndex, int uiKeyBits, byte[] pucSponsorId, int uiSponsorIdLength, ECCPublicKey pucSponsorPublicKey, ECCPublicKey pucSponsorTmpPublicKey, long[] phAgreementHandle);

    /**
     * 计算会话密钥
     * <p/>
     * 描述：使用ECC密钥协商算法，使用自身协商句柄和响应方的协商参数计算会话密钥，同时返回会话密钥句柄。
     * <p/>
     * 备注：协商的发起方获得响应方的协商参数后调用本函数，计算会话密钥。使用SM2算法计算会话密钥的过程见 GM/T 0009。
     *
     * @param hSessionHandle          [in]与设备建立的会话句柄
     * @param pucResponseId           [in]外部输入的响应方Id值
     * @param uiResponseIdLength      [in]外部输入的响应方Id长度
     * @param pucResponsePublicKey    [in]外部输入的响应方ECC公钥对象
     * @param pucResponseTmpPublicKey [in]外部输入的响应方临时ECC公钥对象
     * @param hAgreementHandle        [in]协商句柄，用户计算协商密钥
     * @param phKeyHandle             [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithECC(long hSessionHandle, byte[] pucResponseId, int uiResponseIdLength, ECCPublicKey pucResponsePublicKey, ECCPublicKey pucResponseTmpPublicKey, long hAgreementHandle, long[] phKeyHandle);

    /**
     * 产生协商数据并计算会话密钥
     * <p/>
     * 描述：使用ECC密钥协商算法，产生协商参数并计算会话密钥，同时返回产生的协商参数和密钥句柄。
     * <p/>
     * 备注：本函数由响应方调用。使用SM2算法计算会话密钥的过程见GM/T 0009。
     *
     * @param hSessionHandle          [in]与设备建立的会话句柄
     * @param uiIskIndex              [in]密码设备内部存储加密私钥的索引值，该私钥用于参与密钥协商
     * @param uiKeyBits               [in]协商后要求输入的密钥长度
     * @param pucResponseId           [in]响应方Id值
     * @param uiResponseIdLength      [in]响应方Id长度
     * @param pucSponsorId            [in]发起方Id值
     * @param uiSponsorIdLength       [in]发起方Id长度
     * @param pucSponsorPublicKey     [in]外部输入的发起方ECC公钥对象
     * @param pucSponsorTmpPublicKey  [in]外部输入的发起方临时ECC公钥对象
     * @param pucResponsePublicKey    [out]响应方ECC公钥对象
     * @param pucResponseTmpPublicKey [out]响应方临时ECC公钥对象
     * @param phKeyHandle             [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateAgreementDataAndKeyWithECC(long hSessionHandle, int uiIskIndex, int uiKeyBits, byte[] pucResponseId, int uiResponseIdLength, byte[] pucSponsorId, int uiSponsorIdLength, ECCPublicKey pucSponsorPublicKey, ECCPublicKey pucSponsorTmpPublicKey, ECCPublicKey pucResponsePublicKey, ECCPublicKey pucResponseTmpPublicKey, long[] phKeyHandle);

    /**
     * 基于ECC算法的数字信封转换
     * <p/>
     * 描述：将由内部加密公钥加密的会话密钥转换为由外部指定的公钥加密，可用于数字信封转换。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyIndex     [in]密码设备存储的ECC密钥对索引
     * @param uiAlgId        [in]外部ECC公钥的算法标识
     * @param pucPublicKey   [in]外部ECC公钥对象
     * @param pucEncDataIn   [in]会话密钥密文
     * @param pucEncDataOut  [out]会话密钥密文(外部ECC公钥加密)
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int exchangeDigitEnvelopeBaseOnECC(long hSessionHandle, int uiKeyIndex, int uiAlgId, ECCPublicKey pucPublicKey, ECCCipher pucEncDataIn, ECCCipher pucEncDataOut);

    /**
     * 生成会话密钥并用密钥加密密钥加密输出
     * <p/>
     * 描述：生成会话密钥并用密钥加密密钥加密输出，同时返回密钥句柄。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiKeyBits      [in]指定产生的会话密钥长度
     * @param uiAlgId        [in]算法标识，指定对称加密算法
     * @param uiKekIndex     [in]密码设备内部存储秘钥加密密钥的索引值
     * @param pucKey         [out]会话秘钥密文
     * @param puiKeyLength   [out]会话密钥密文长度，获取方式：puiKeyLength[0]
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int generateKeyWithKEK(long hSessionHandle, int uiKeyBits, int uiAlgId, int uiKekIndex, byte[] pucKey, int[] puiKeyLength, long[] phKeyHandle);

    /**
     * 导入会话密钥并用密钥加密密钥解密
     * <p/>
     * 描述：导入会话密钥并用密钥加密密钥解密，同时返回会话密钥句柄。
     * <p/>
     * 备注：加密模式使用ECB模式。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定对称加密算法
     * @param uiKekIndex     [in]密码设备内部存储秘钥加密密钥的索引值
     * @param pucKey         [in]会话密钥密文
     * @param puiKeyLength   [in]会话密钥密文长度
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int importKeyWithKEK(long hSessionHandle, int uiAlgId, int uiKekIndex, byte[] pucKey, int puiKeyLength, long[] phKeyHandle);

    /**
     * 导入明文会话密钥
     * <p/>
     * 描述：导入明文会话密钥，同时返回会话密钥句柄。
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucKey         [in]会话密钥明文
     * @param puiKeyLength   [in]会话密钥明文长度
     * @param phKeyHandle    [out]会话密钥句柄，获取方式：phKeyHandle[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int ImportKey(long hSessionHandle, byte[] pucKey, int puiKeyLength, long[] phKeyHandle);

    /**
     * 销毁会话密钥
     * <p/>
     * 描述：销毁会话密钥，并释放为密钥句柄分配的内存等资源。
     * <p/>
     * 备注：在对称算法运算完成后，应调用本函数销毁会话密钥。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param hKeyHandle     [out]会话密钥句柄
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int destroyKey(long hSessionHandle, long hKeyHandle);

    /**
     * 外部公钥RSA运算
     * <p/>
     * 描述：指定使用外部公钥对数据进行运算。
     * <p/>
     * 备注：数据格式由应用层封装。
     *
     * @param hSessionHandle  [in]与设备建立的会话句柄
     * @param pucPublicKey    [in]外部RSA公钥对象
     * @param pucDataInput    [in]输入的数据
     * @param uiInputLength   [in]输入的数据长度
     * @param pucDataOutput   [out]输出的数据
     * @param puiOutputLength [out]输出的数据长度，获取方式：puiOutputLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int externalPublicKeyOperationRSA(long hSessionHandle, RSAPublicKey pucPublicKey, byte[] pucDataInput, int uiInputLength, byte[] pucDataOutput, int[] puiOutputLength);

    /**
     * 内部公钥RSA运算
     * <p/>
     * 描述：使用内部指定索引的公钥对数据进行运算。
     * <p/>
     * 备注：索引范围仅限于内部签名密钥对，数据格式由应用层封装。
     *
     * @param hSessionHandle  [in]与设备建立的会话句柄
     * @param uiKeyIndex      [in]密码设备内部存储公钥的索引值
     * @param pucDataInput    [in]输入的数据
     * @param uiInputLength   [in]输入的数据长度
     * @param pucDataOutput   [out]输出的数据
     * @param puiOutputLength [out]输出的数据长度，获取方式：puiOutputLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int internalPublicKeyOperationRSA(long hSessionHandle, int uiKeyIndex, byte[] pucDataInput, int uiInputLength, byte[] pucDataOutput, int[] puiOutputLength);

    /**
     * 内部私钥RSA运算
     * <p/>
     * 描述：使用内部指定索引的私钥对数据进行运算。
     * <p/>
     * 备注：索引范围仅限于内部签名密钥对，数据格式由应用层封装。
     *
     * @param hSessionHandle  [in]与设备建立的会话句柄
     * @param uiKeyIndex      [in]密码设备内部存储私钥的索引值
     * @param pucDataInput    [in]输入的数据
     * @param uiInputLength   [in]输入的数据长度
     * @param pucDataOutput   [out]输出的数据
     * @param puiOutputLength [out]输出的数据长度，获取方式：puiOutputLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int internalPrivateKeyOperationRSA(long hSessionHandle, int uiKeyIndex, byte[] pucDataInput, int uiInputLength, byte[] pucDataOutput, int[] puiOutputLength);

    /**
     * 外部密钥ECC签名
     * <p/>
     * 描述：使用外部ECC私钥对数据进行签名运算。
     * <p/>
     * 备注：输入数据为待签数据的杂凑值。当使用SM2算法时，该输入数据为待签数据经过SM2签名预处理的结果，预处理过程见 GM/T 0009。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定使用的ECC算法
     * @param pucPrivateKey  [in]外部ECC私钥结构
     * @param pucData        [in]输入的待签数据
     * @param uiDataLength   [in]输入的待签数据长度
     * @param pucSignature   [out]签名值数据
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int externalSignEcc(long hSessionHandle, int uiAlgId, ECCPrivateKey pucPrivateKey, byte[] pucData, int uiDataLength, ECCSignature pucSignature);

    /**
     * 外部密钥ECC运算
     * <p/>
     * 描述：使用外部ECC公钥对ECC签名值进行验证运算。
     * <p/>
     * 备注：输入数据为待签数据的杂凑值。当使用SM2算法时，该输入数据为待签数据经过SM2签名预处理的结果，预处理过程见 GM/T 0009。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定使用的ECC算法
     * @param pucPublicKey   [in]外部ECC公钥结构
     * @param pucDataInput   [in]输入的数据
     * @param uiInputLength  [in]输入的数据长度
     * @param pucSignature   [in]签名值数据对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int externalVerifyECC(long hSessionHandle, int uiAlgId, ECCPublicKey pucPublicKey, byte[] pucDataInput, int uiInputLength, ECCSignature pucSignature);

    /**
     * 内部密钥ECC签名
     * <p/>
     * 描述：使用内部ECC私钥对数据进行签名运算。
     * <p/>
     * 备注：输入数据为待签数据的杂凑值。当使用SM2算法时，该输入数据为待签数据经过SM2签名预处理的结果，预处理过程见 GM/T 0009。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIskIndex     [in]密码设备内部存储的ECC签名私钥的索引值
     * @param pucData        [in]输入的待签数据
     * @param uiDataLength   [in]输入的待签数据长度
     * @param pucSignature   [out]签名值数据
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int internalSignECC(long hSessionHandle, int uiIskIndex, byte[] pucData, int uiDataLength, ECCSignature pucSignature);

    /**
     * 内部密钥ECC验证
     * <p/>
     * 描述：使用内部ECC公钥对ECC签名值进行验证运算。
     * <p/>
     * 备注：输入数据为待签数据的杂凑值。当使用SM2算法时，该输入数据为待签数据经过SM2签名预处理的结果，预处理过程见 GM/T 0009。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiIskIndex     [in]密码设备内部存储的ECC签名公钥的索引值
     * @param pucData        [in]输入的待签数据
     * @param uiDataLength   [in]输入的待签数据长度
     * @param pucSignature   [in]签名值数据
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int internalVerifyECC(long hSessionHandle, int uiIskIndex, byte[] pucData, int uiDataLength, ECCSignature pucSignature);

    /**
     * 外部密钥ECC公钥加密
     * <p/>
     * 描述：使用外部ECC公钥对数据进行加密运算。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定使用的算法
     * @param pucPublicKey   [in]外部ECC公钥对象
     * @param pucData        [in]输入的待加密数据
     * @param uiDataLength   [in]输入的待加密数据长度
     * @param pucEncData     [out]数据密文对象
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int externalEncryptECC(long hSessionHandle, int uiAlgId, ECCPublicKey pucPublicKey, byte[] pucData, int uiDataLength, ECCCipher pucEncData);

    /**
     * 外部密钥ECC私钥解密
     * <p/>
     * 描述：使用外部ECC私钥对数据进行解密运算。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定使用的算法
     * @param pucPrivateKey  [in]外部ECC私钥结构
     * @param pucEncData     [in]数据密文对象
     * @param pucData        [out]输入的待加密数据
     * @param puiDataLength  [out]输入的待加密数据长度
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int externalDecryptEcc(long hSessionHandle, int uiAlgId, ECCPrivateKey pucPrivateKey, ECCCipher pucEncData, byte[] pucData, int[] puiDataLength);

    /**
     * 内部密钥ECC私钥加密
     * <p/>
     * 描述：使用内部ECC私钥对数据进行解密运算。
     *
     * @param hSessionHandle [in]	与设备建立的会话句柄
     * @param uiAlgId        [in]			算法标识，指定使用的ECC算法
     * @param uiKeyIndex     [in]	密码设备内部存储的ECC签名公钥的索引值
     * @param pucData        [in]			缓冲区指针，用于存放外部输入的数据
     * @param uiDataLength   [in]	输入数据长度
     * @param pucEncData     [out]		缓冲区指针，用于存放外部输出的密文
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_Internal_Encrypt_ECC(void *hSessionHandle,
     * unsigned int uiAlgID,
     * unsigned int uiKeyIndex,
     * unsigned char *pucData,
     * unsigned int uiDataLength,
     * ECCCipher *pucEncData);
     */
    public native int internalEncryptEcc(long hSessionHandle, int uiAlgId, int uiKeyIndex, byte[] pucData, int uiDataLength, ECCCipher pucEncData);

    /**
     * 内部密钥ECC私钥解密
     * <p/>
     * 描述：使用内部ECC私钥对数据进行解密运算。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]算法标识，指定使用的算法
     * @param uiKeyIndex     [in]内部ECC私钥索引
     * @param pucEncData     [in]数据密文对象
     * @param pucData        [out]输入的待加密数据
     * @param puiDataLength  [out]输入的待加密数据长度
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int internalDecryptECC(long hSessionHandle, int uiAlgId, int uiKeyIndex, ECCCipher pucEncData, byte[] pucData, int[] puiDataLength);

    /**
     * 对称加密
     * <p/>
     * 描述：使用指定的密钥句柄和IV对数据进行对称加密运算。
     * <p/>
     * 备注：此函数不对数据进行填充处理，输入的数据必须是指定算法分组长度的整倍数。
     *
     * @param hSessionHandle   [in]与设备建立的会话句柄
     * @param hKeyHandle       [in]指定的密钥句柄
     * @param uiAlgId          [in]算法标识，指定对称加密算法
     * @param pucIv            [in|out]输入和返回的IV数据
     * @param pucData          [in]输入的数据明文
     * @param uiDataLength     [in]输入的数据明文长度
     * @param pucEncData       [out]数据密文
     * @param puiEncDataLength [out]数据密文长度，获取方式：puiEncDataLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int encrypt(long hSessionHandle, long hKeyHandle, int uiAlgId, byte[] pucIv, byte[] pucData, int uiDataLength, byte[] pucEncData, int[] puiEncDataLength);

    /**
     * 对称解密
     * <p/>
     * 描述：使用指定的密钥句柄和IV对数据进行对称解密运算。
     * <p/>
     * 备注：此函数不对数据进行填充处理，输入的数据必须是指定算法分组长度的整倍数。
     *
     * @param hSessionHandle  [in]与设备建立的会话句柄
     * @param hKeyHandle      [in]指定的密钥句柄
     * @param uiAlgId         [in]算法标识，指定对称加密算法
     * @param pucIv           [in|out]输入和返回的IV数据
     * @param pucEncData      [in]输入的密文数据
     * @param uiEncDataLength [in]输入的密文数据长度
     * @param pucData         [out]明文数据
     * @param puiDataLength   [out]明文数据长度，获取方式：puiDataLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int decrypt(long hSessionHandle, long hKeyHandle, int uiAlgId, byte[] pucIv, byte[] pucEncData, int uiEncDataLength, byte[] pucData, int[] puiDataLength);

    /**
     * 计算MAC
     * <p/>
     * 描述：使用指定的密钥句柄和IV对数据进行MAC运算。
     * <p/>
     * 备注：此函数不对数据进行分包处理，多包数据MAC运算由IV控制最后的MAC值。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param hKeyHandle     [in]指定的密钥句柄
     * @param uiAlgId        [in]算法标识，指定MAC加密算法
     * @param pucIv          [in|out]输入和返回的IV数据
     * @param pucData        [in]输入的明文数据
     * @param uiDataLength   [in]输入的明文数据长度
     * @param pucMac         [out]MAC值
     * @param puiMacLength   [out]MAC值长度，获取方式：puiMacLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int calculateMac(long hSessionHandle, long hKeyHandle, int uiAlgId, byte[] pucIv, byte[] pucData, int uiDataLength, byte[] pucMac, int[] puiMacLength);

    /**
     * 杂凑运算初始化
     * <p/>
     * 描述：三步式数据杂凑运算第一步。
     * <p/>
     * 备注：uiIdLength非零且uiAlgId为SGD_SM3时，函数执行SM2的预处理1操作。计算过程见 GM/T 0009。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param uiAlgId        [in]指定杂凑算法标识
     * @param pucPublicKey   [in]签名者公钥，当uiAlgId为SGD_SM3时有效。
     * @param pucId          [in]签名者的Id值，当uiAlgId为SGD_SM3时有效。
     * @param uiIdLength     [in]签名者Id的长度，当uiAlgId为SGD_SM3时有效。
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int hashInit(long hSessionHandle, int uiAlgId, ECCPublicKey pucPublicKey, byte[] pucId, int uiIdLength);

    /**
     * 多包杂凑运算
     * <p/>
     * 描述：三步式数据杂凑运算第二步，对输入的明文进行杂凑运算。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucData        [in]输入的明文数据
     * @param uiDataLength   [in]输入的明文数据长度
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int hashUpdate(long hSessionHandle, byte[] pucData, int uiDataLength);

    /**
     * 多凑运算结束
     * <p/>
     * 描述：三步式数据杂凑运算第三步，杂凑运算结束返回杂凑数据并清除中间数据。
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucHash        [out]缓冲区指针，用于存放输出的杂凑数据
     * @param puiHashLength  [out]杂凑数据长度，获取方式：puiHashLength[0]
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int hashFinal(long hSessionHandle, byte[] pucHash, int[] puiHashLength);

    /**
     * 创建文件
     * <p/>
     * 描述：在密码设备内部创建用于存储用户数据的文件
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucFileName    [in]输入文件名，最大长度为128
     * @param uiNameLen      [in]文件名长度
     * @param uiFileSize     [in]文件所占存储空间的长度
     *                       <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int createFile(long hSessionHandle, byte[] pucFileName, int uiNameLen, int uiFileSize);

    /**
     * 读取文件
     * <p/>
     * 描述：读取在密码设备内部存储用户数据的文件的内容
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucFileName    [in]输入文件名，最大长度为128
     * @param uiNameLen      [in]文件名长度
     * @param uiOffset       [in]指定读取文件时的偏移值
     * @param puiFileLength  [in|out]入参指定读取文件的长度；出参返回实际读取文件内容的长度
     * @param pucBuffer      [in]缓冲区指针，用于存放读取的文件数据
     *                       <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int readFile(long hSessionHandle, byte[] pucFileName, int uiNameLen, int uiOffset, int[] puiFileLength, byte[] pucBuffer);

    /**
     * 写文件
     * <p/>
     * 描述：向密码设备内部存储用户数据的文件中写入内容
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucFileName    [in]输入文件名，最大长度为128
     * @param uiNameLen      [in]文件名长度
     * @param uiOffset       [in]指定写入文件时的偏移值
     * @param puiFileLength  [in]指定写入文件内容的长度
     * @param pucBuffer      [in]缓冲区指针，用于存放输入的写文件数据
     *                       <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int writeFile(long hSessionHandle, byte[] pucFileName, int uiNameLen, int uiOffset, int puiFileLength, byte[] pucBuffer);

    /**
     * 用户文件操作类函数
     * <p/>
     * 描述：删除指定文件名的密码设备内部存储用户数据的文件
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param pucFileName    [in]输入文件名，最大长度为128
     * @param uiNameLen      [in]文件名长度
     *                       <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     */
    public native int deleteFile(long hSessionHandle, byte[] pucFileName, int uiNameLen);


    /**
     * 导入SM2签名公钥
     * <p/>
     * 描述：导入SM2签名公钥，同时返回会话密钥句柄。
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index          [in]密码设备内部存储秘钥签名密钥的索引值
     * @param pucPublicKey   [in]签名公钥
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_sm2_importSignpub(void *hSessionHandle,int index,ECCrefPublicKey *sdf_pub);
     */
    public native int SM2ImportSignpub(long hSessionHandle, int index, ECCPublicKey pucPublicKey);

    /**
     * 导入SM2签名私钥
     * <p/>
     * 描述：导入SM2签名私钥，同时返回会话密钥句柄。
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index          [in]密码设备内部存储秘钥签名密钥的索引值
     * @param pucPrivateKey  [in]签名私钥
     * @param mode           [in] 算法标示
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_sm2_importSignpub(void *hSessionHandle,int index,ECCrefPublicKey *sdf_pub);
     */
    public native int SM2ImportSignpri(long hSessionHandle, int index, ECCPrivateKey pucPrivateKey, int mode);


    /**
     * 导入SM2加密公钥
     * <p/>
     * 描述：导入SM2加密公钥，同时返回会话密钥句柄。
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index          [in]密码设备内部存储秘钥加密密钥的索引值
     * @param pucPublicKey   [in]加密公钥
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_sm2_importEncpub(void *hSessionHandle,int index,ECCrefPublicKey *sdf_pub);
     */
    public native int SM2ImportEncpub(long hSessionHandle, int index, ECCPublicKey pucPublicKey);

    /**
     * 导入SM2加密私钥
     * <p/>
     * 描述：导入SM2加密私钥，同时返回会话密钥句柄。
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index          [in]密码设备内部存储秘钥加密密钥的索引值
     * @param pucPrivateKey  [in]加密私钥
     * @param mode           [in] 算法标示
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_sm2_importEncpub(void *hSessionHandle,int index,ECCrefPublicKey *sdf_pub);
     */
    public native int SM2ImportEncpri(long hSessionHandle, int index, ECCPrivateKey pucPrivateKey, int mode);

    /**
     * <p/>
     * 描述：
     * <p/>
     *
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param dataIn         [in]输入数据message
     * @param dataInLen      [in]输入数据长度
     * @param key            [in]输入key
     * @param keyLen         [in]输入key 长度
     * @param mac            [out]输出hmac 32位
     * @return 0：成功；
     * <br/>非0：失败，返回错误代码(错误码定义参见 {@link SdfApiCode})
     * <p>
     * XDJA_API int SDF_sm3_hmac(void *hSessionHandle, unsigned char *dataIn, int dataInLen, unsigned char *key, int keyLen, unsigned char *mac);
     */
    public native int SM3HMAC(long hSessionHandle, byte[] dataIn, int dataInLen, byte[] key, int keyLen, byte[] mac);



    /***
     * 导入RSA加密公钥
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index [in]密钥索引
     * @param rsaPublicKey [in] RSA公钥
     *
     * @return 0
     * XDJA_API int SDF_rsa_importEncpub(void *hSessionHandle,int index,RSArefPublicKey *sdf_pub);
     */
    public native int RSAImportEncpub(long hSessionHandle, int index, RSAPublicKey rsaPublicKey);

    /***
     * 导入RSA加密私钥
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index [in]密钥索引
     * @param rsaPrivateKey [in] RSA私钥
     * @param model [in]导入模式 默认为0
     * @return 0
     * XDJA_API int SDF_rsa_importEncpri(void *hSessionHandle,int index,RSArefPrivateKey *sdf_pri,int mode);
     */
    public native int RSAImportEncpri(long hSessionHandle, int index, RSAPrivateKey rsaPrivateKey, int model);


    /***
     * 导入RSA签名公钥
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index [in]密钥索引
     * @param rsaPublicKey [in] RSA公钥
     *
     * @return 0
     * XDJA_API int SDF_rsa_importSignpub(void *hSessionHandle,int index,RSArefPublicKey *sdf_pub);
     */
    public native int RSAImportSignpub(long hSessionHandle, int index, RSAPublicKey rsaPublicKey);


    /***
     * 导入RSA签名私钥
     * @param hSessionHandle [in]与设备建立的会话句柄
     * @param index [in]密钥索引
     * @param rsaPrivateKey [in] RSA私钥
     * @param model [in]导入模式 默认为0
     * @return 0
     * XDJA_API int SDF_rsa_importEncpri(void *hSessionHandle,int index,RSArefPrivateKey *sdf_pri,int mode);
     */
    public native int RSAImportSignpri(long hSessionHandle, int index, RSAPrivateKey rsaPrivateKey, int model);


    public native int loadSystemIC(long paramLong, byte[] paramArrayOfByte, int paramInt);

    public native int bindUserIC(long paramLong, byte[] paramArrayOfByte, int paramInt);

    public native int loadUserIC(long paramLong, byte[] paramArrayOfByte, int paramInt);
}
