package com.xdja.pki.gmssl.sdf.pcie;

import com.xdja.pcie.SDFAPI;
import com.xdja.pcie.base.*;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfApiCode;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPrivateKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.InputStream;
import java.security.*;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;

public class PcieSdfSDKUtils {

    private static Logger logger = LoggerFactory.getLogger(PcieSdfSDKUtils.class);

    /*************************************************************************************************
     *                                          JAVA to SDF                                          *
     *************************************************************************************************/
    public static ECCPublicKey generateEccPublicKey(Logger logger, PublicKey key) {
        ECPublicKey ecPublicKey = (ECPublicKey) key;

        return generateEccPublicKey(logger, GMSSLByteArrayUtils.asUnsignedByteArray32(ecPublicKey.getW().getAffineX()),
                GMSSLByteArrayUtils.asUnsignedByteArray32(ecPublicKey.getW().getAffineY()));
    }

    public static ECCPublicKey generateEccPublicKey(Logger logger, SdfECCPublicKey key) {
        logger.info(key.toString());
        return generateEccPublicKey(logger, key.getX(), key.getY());
    }

    public static ECCPublicKey generateEccPublicKey(Logger logger, byte[] x, byte[] y) {
        logger.info("generateEccPublicKey");

//        GMSSLByteArrayUtils.printHexBinary(logger, "byte[] x", x);
//        GMSSLByteArrayUtils.printHexBinary(logger, "byte[] y", y);

        ECCPublicKey pucPublicKey = new ECCPublicKey(
                GMSSLX509Utils.ECC_KEY_BITS,
                GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(x),
                GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(y)
        );

//        GMSSLByteArrayUtils.printHexBinary(logger, "EccPublicKey x", pucPublicKey.getX());
//        GMSSLByteArrayUtils.printHexBinary(logger, "EccPublicKey y", pucPublicKey.getY());

        return pucPublicKey;
    }

    public static ECCPrivateKey generateEccPrivateKey(Logger logger, PrivateKey key) {
        ECPrivateKey ecPrivateKey = (ECPrivateKey) key;
        return generateEccPrivateKey(logger, GMSSLByteArrayUtils.asUnsignedByteArray32(ecPrivateKey.getS()));
    }

    public static ECCPrivateKey generateEccPrivateKey(Logger logger, byte[] k) {
        logger.info("generateEccPrivateKey");
        ECCPrivateKey pucPrivateKey = new ECCPrivateKey(
                GMSSLX509Utils.ECC_KEY_BITS,
                GMSSLByteArrayUtils.fillByteArrayWithZeroInFoot(k)
        );
        GMSSLByteArrayUtils.printHexBinary(logger, "pucPrivateKey k", pucPrivateKey.getK());

        return pucPrivateKey;
    }

    public static void checkRet(String method, int ret, long dev, long ses) throws SdfSDKException {
        if (ret != 0) {
            logger.error(
                    "{} error! dev = {} session = {} ret = {} HEX={}:{}",
                    method,
                    dev,
                    ses,
                    ret,
                    Integer.toHexString(ret),
                    SdfApiCode.apiCodeToString(ret)
            );
            throw new SdfSDKException(method, ret);
        } else {
            logger.info(method + " success! dev = " + dev + " session = " + ses);
        }
    }

    //打开设备
    public static void openDevice(SDFAPI sdfApi, long[] dev) throws SdfSDKException {
        int ret = sdfApi.openDevice(dev);
        checkRet("openDevice", ret, dev[0], 0);
    }


    //打开 session
    public static void openSession(SDFAPI sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        int ret = sdfApi.openSession(dev[0], ses);
        checkRet("openSession", ret, dev[0], ses[0]);
    }

    public static void closeSession(SDFAPI sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        int ret = sdfApi.closeSession(ses[0]);
        checkRet("closeSession", ret, dev[0], ses[0]);
    }

    public static void closeDevice(SDFAPI sdfApi, long[] dev) throws SdfSDKException {
        int ret = sdfApi.closeDevice(dev[0]);
        checkRet("closeDevice", ret, dev[0], 0);
    }

    public static String getDeviceInfo(SDFAPI sdfApi, long[] dev, long[] ses) throws SdfSDKException {
        DeviceInfo deviceInfo = new DeviceInfo();
        int ret = sdfApi.getDeviceInfo(ses[0], deviceInfo);
        checkRet("getDeviceInfo", ret, dev[0], ses[0]);
        return deviceInfo.toString();
    }


    /*************************************************************************************************
     *                                          SDF to JAVA                                          *
     *************************************************************************************************/

    public static PublicKey getPublicKeyFromCer(String path) throws Exception {
        InputStream inputStream = ClassLoader.getSystemResourceAsStream(path);
        if (inputStream == null) {
            inputStream = new FileInputStream(path);
        }
        CertificateFactory factory = CertificateFactory.getInstance("X.509", "BC");
        X509Certificate certificate = (X509Certificate) factory.generateCertificate(inputStream);
        return certificate.getPublicKey();
    }

    /**
     * 从SDF RSA Public Key转换成 JNI的 RSAPublicKey
     * @param rsaPublic SDF RSA Public Key
     * @return RSAPublicKey
     */
    public static RSAPublicKey generateRsaPublicKey(SdfRSAPublicKey rsaPublic) {
        return new RSAPublicKey(rsaPublic.getBits(), rsaPublic.getM(), rsaPublic.getE());
    }
    /**
     * 从SDF RSA Private Key转换成 JNI的 RSAPrivateKey
     * @param privateKey SDF RSA Private Key
     * @return RSAPrivateKey
     */
    public static RSAPrivateKey generateRsaPrivateKey(SdfRSAPrivateKey privateKey) {
        return new RSAPrivateKey(privateKey.getBits(),
                privateKey.getM(),
                GMSSLByteArrayUtils.changeByteArrayLength(privateKey.getE(),RSAPublicKey.RSA_MAX_LEN),
                privateKey.getD(),
                privateKey.getPrime(),
                privateKey.getPexp(),
                privateKey.getCoef());
    }
}
