package com.xdja.pki.gmssl.sdf.pcie.pool;

import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import org.apache.commons.pool2.PooledObjectFactory;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

/**
 * @ClassName PcieConnectionProviderImpl
 * @Description TODO
 * @Date 2020/5/18 20:04
 * @Author FengZhen
 */
public class PcieConnectionProviderImpl implements PcieConnectionPrivoider {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private GenericObjectPool<PcieConnection> genericObjectPool;

    private volatile PciePooledObjectFactory factory;

    private PcieConnectionProviderImpl() {
        this.factory = new PciePooledObjectFactory();
        this.genericObjectPool = new GenericObjectPool<>(this.factory);
        initPool();
    }

    private void initPool() {
        //设置最大连接数量 默认是 8
        int maxTotal = 200;
        int maxIdle = 100;
        int minIdle = 30;
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        List<StackTraceElement> list = Arrays.asList(stackTrace);
        for (StackTraceElement element : list) {
            if (element.getClassName().startsWith("org.junit.")) {
                maxTotal = 8;
                maxIdle = 2;
                minIdle = 1;
            }
            if (element.getClassName().startsWith("com.xdja.pki.gmssl.main.")) {
                maxTotal = 1;
                maxIdle = 1;
                minIdle = 1;
            }
        }
        String property = System.getProperty("catalina.home");
        File hsmCfg = new File(property + "/conf/hsm.conf");
        logger.error("INFO init connection pool: config={} ", hsmCfg);
        if (hsmCfg.exists()) {
            try {
                byte[] bytes = GMSSLFileUtils.readFileToByte(hsmCfg.getAbsolutePath());
                String cfg = new String(bytes);
                cfg = cfg.replaceAll("\n", "")
                        .replaceAll("\r", "")
                        .replaceAll("\r\n", "");
                String[] cfgs = cfg.split("-");
                maxTotal = Integer.valueOf(cfgs[0]);
                maxIdle = Integer.valueOf(cfgs[1]);
                minIdle = Integer.valueOf(cfgs[2]);
            } catch (IOException e) {
                logger.error("read hsm.conf error! {}", e);
            }

        }
        logger.error("INFO init connection pool: setMaxTotal={} setMaxIdle={} setMinIdle={}", maxTotal, maxIdle, minIdle);
        //设置池在给定时间可以分配的对象数上限（签出给客户端，或空闲等待签出）。
        this.genericObjectPool.setMaxTotal(maxTotal);
        //设置池中“空闲”实例数的上限。
        this.genericObjectPool.setMaxIdle(maxIdle);
        //为池中要维护的最小空闲对象数设置目标。
        this.genericObjectPool.setMinIdle(minIdle);
        if (this.factory.isDeviceOpen()) {
            try {
                logger.error("INFO init connection pool: prepare hsm connection pool minIdle={}", minIdle);
                //尝试确保getMinIdle()空闲实例在池中可用。
                this.genericObjectPool.preparePool();
            } catch (Exception e) {
                logger.error("INFO init connection pool: prepare hsm connection pool error {}", e);
            }
        }
        logger.error("INFO init connection pool: getMaxTotal={} getMaxIdle={} " +
                        "getMinIdle={} getNumIdle={} getCreatedCount={}",
                genericObjectPool.getMaxTotal(), genericObjectPool.getMaxIdle(),
                genericObjectPool.getMinIdle(), genericObjectPool.getNumIdle(), genericObjectPool.getCreatedCount());
    }

    @Override
    public PcieConnection getConnection() {
        try {
            if (this.genericObjectPool.getNumWaiters() > 0) {
                StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
                String methodName = stackTrace[2].getMethodName();
                logger.info(
                        "get connection pool: {} getNumIdle={} getNumActive={} getNumWaiters={} " +
                                "getCreatedCount={} getBorrowedCount={} getReturnedCount={}",
                        methodName, genericObjectPool.getReturnedCount(), genericObjectPool.getNumActive(), genericObjectPool.getNumWaiters(),
                        genericObjectPool.getCreatedCount(), genericObjectPool.getBorrowedCount(), genericObjectPool.getReturnedCount()
                );
            }
            return this.genericObjectPool.borrowObject();
        } catch (Exception e) {
            logger.error("getConnection error", e);
            throw new RuntimeException("getConnection error", e);
        }
    }

    @Override
    public void releaseConnection(PcieConnection connection) {
        try {
            this.genericObjectPool.returnObject(connection);
            if (this.genericObjectPool.getNumWaiters() > 0) {
                StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
                String methodName = stackTrace[3].getMethodName();
                logger.info(
                        "release connection pool: {} getNumIdle={} getNumActive={} getNumWaiters={} " +
                                "getCreatedCount={} getBorrowedCount={} getReturnedCount={}",
                        methodName, genericObjectPool.getReturnedCount(), genericObjectPool.getNumActive(), genericObjectPool.getNumWaiters(),
                        genericObjectPool.getCreatedCount(), genericObjectPool.getBorrowedCount(), genericObjectPool.getReturnedCount()
                );
            }

        } catch (Exception e) {
            logger.error("releaseConnection error", e);
            throw new RuntimeException("releaseConnection error", e);
        }
    }

    public synchronized void clear() {
        this.genericObjectPool.clear();
        this.factory.closeDevice();
    }

    public synchronized void reopen() throws Exception {
        this.clear();
        this.factory.openDevice();
        this.genericObjectPool.preparePool();
    }

    private static class ConnectionProviderHolder {
        private static final PcieConnectionProviderImpl INSTANCE = new PcieConnectionProviderImpl();
    }

    public static synchronized PcieConnectionProviderImpl getInstance() {
        return PcieConnectionProviderImpl.ConnectionProviderHolder.INSTANCE;
    }
}
