package com.xdja.pki.gmssl.sdf.yunhsm;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;

public class HsmConnectionPool {

    private static Logger logger = LoggerFactory.getLogger(HsmConnectionPool.class);

    private static int POOL_MAX_LEN = 100;
    private static int POOL_LESS_LEN = POOL_MAX_LEN/5;
    private static int FREE_LONG_TIME = 1000;

    static {
        if (isJUnitTest()){
            POOL_MAX_LEN = 20;
            FREE_LONG_TIME = 1000;
        }
        HsmConnectionPool.getInstance();
    }

    private LinkedBlockingQueue<HsmConnection> freePool = new LinkedBlockingQueue<>(POOL_MAX_LEN);
    private LinkedBlockingQueue<HsmConnection> usedPool = new LinkedBlockingQueue<>(POOL_MAX_LEN);

    public static boolean isJUnitTest() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        List<StackTraceElement> list = Arrays.asList(stackTrace);
        for (StackTraceElement element : list) {
            if (element.getClassName().startsWith("org.junit.")) {
                return true;
            }
        }
        return false;
    }

    private HsmConnectionPool(){
        logger.info("yunhsm connection pool init");
        for (int i = 0; i < POOL_MAX_LEN; i++) {
            try {
                HsmConnection hsmConnection = new HsmConnection();
                hsmConnection.setId(i);
                hsmConnection.setStartTime(new Date().getTime());
                freePool.put(hsmConnection);
            } catch (Exception e) {
                logger.error("init connection error", e);
            }
        }
    }

    private static class HsmConnectionPoolHolder {
        private static final HsmConnectionPool instance = new HsmConnectionPool();
    }

    public static HsmConnectionPool getInstance(){
        return HsmConnectionPoolHolder.instance;
    }

    public synchronized HsmConnection getConnection() {
        if (this.freePool.size() <= POOL_LESS_LEN && this.usedPool.size() > 0){
            List<Integer> list = new ArrayList<>();
            for (HsmConnection conn: this.usedPool) {
                long now = new Date().getTime();
                if (now - conn.getStartTime() > FREE_LONG_TIME){
                    try {
                        HsmConnection take = this.usedPool.take();
                        list.add(take.getId());
                        this.freePool.put(take);
                    } catch (InterruptedException e) {
                        logger.error("move connection error", e);
                    }
                }
            }
//            logger.info("now move connection from used to free pool len {} {}", list.size(), list);
        }
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        String methodName = stackTrace[2].getMethodName();
//        logger.info("get now time pool size {}", freePool.size());
        try {
            HsmConnection take = freePool.take();
//            logger.debug(" {} get connection id {}", methodName, take.getId());
            take.setStartTime(new Date().getTime());
            usedPool.put(take);
            return take;
        } catch (InterruptedException e) {
            logger.error("get connection error {}", methodName, e);
            return null;
        }
    }

    public synchronized void releaseConnection(HsmConnection hsmConnection){
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        String methodName = stackTrace[2].getMethodName();
        try {
            for (HsmConnection conn: this.usedPool) {
                if (conn.getId() == hsmConnection.getId()){
                    this.usedPool.remove(conn);
                    this.freePool.put(hsmConnection);
                }
            }
//            logger.info("release now time pool size {} producer is {} connection id {}", freePool.size(), methodName, hsmConnection.getId());
            List<Integer> list = new ArrayList<>();
            for (HsmConnection conn: this.freePool) {
                list.add(conn.getId());
            }
//            logger.info("now time pool is len {} value {}", this.freePool.size(), list);
        } catch (InterruptedException e) {
            logger.error("release connection error{}", methodName, e);
        }
    }
}
