package com.xdja.pki.gmssl.sdf.yunhsm;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.*;
import org.bouncycastle.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.util.Arrays;

import static com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils.hexEncode;


public class YumhsmSdfDemo {
    private static Logger logger = LoggerFactory.getLogger(YumhsmSdfDemo.class.getName());

    public static void main(String[] args) throws Exception {
        YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();

        if (args.length == 0 || args[0] == null) {
            System.out.println("Please insert method, Example: all. random. gensm2key. getprikeyacc,sm1enc, exportenc. exportsign. sm2sign. sm2enc. sm4enc. sm3hash.  sm4mac");
            System.out.println("                    if use 'random' you can add length, Example: random 28, Default: 28");
            System.out.println("                    if use 'getprikeyacc' you can test getPrivateKeyAccess and releasePrivateKeyAccessRight");
            System.out.println("                    ......");
            return;
        }

        System.out.println("You want to test: " + args[0]);

        switch (args[0]) {
            case "all":
                System.out.println("*************** random ***************");
                random(sdfSDK, args);
                System.out.println("*************** generateKeyPairEcc ***************");
                generateKeyPairEcc(sdfSDK);
                System.out.println("*************** getPrivateKeyAccess ***************");
                getPrivateKeyAccessRight(sdfSDK);
                System.out.println("*************** sm1EncDemo ***************");
                sm1EncDemo(sdfSDK);
                System.out.println("*************** sm2SignDemo ***************");
                sm2SignDemo(sdfSDK);
                System.out.println("*************** sm2EncDemo ***************");
                sm2EncDemo(sdfSDK);
                System.out.println("*************** sm3HashDemo ***************");
                sm3HashDemo(sdfSDK);
                System.out.println("*************** sm3HashDemoWithPublicKey ***************");
                sm3HashDemoWithPublicKey(sdfSDK);
                System.out.println("*************** sm4EncDemo ***************");
                sm4EncDemo(sdfSDK);
                System.out.println("*************** sm4MacDemo ***************");
                sm4MacDemo(sdfSDK);
                break;
            case "random":
                //测试生成 随机数
                random(sdfSDK, args);
                break;
            case "getprikeyacc":
                getPrivateKeyAccessRight(sdfSDK);
                break;
            case "sm1enc":
                sm1EncDemo(sdfSDK);
                break;
            case "exportsign":
                exportSignKey(sdfSDK, args);
                break;
            case "exportenc":
                exportEncKey(sdfSDK, args);
                break;
            case "gensm2key":
                //生成SM2公私钥对
                generateKeyPairEcc(sdfSDK);
                break;
            case "sm2sign":
                sm2SignDemo(sdfSDK);
                break;
            case "sm2enc":
                sm2EncDemo(sdfSDK);
                break;
            case "sm3hash":
                System.out.println("*************** sm3HashDemoWithPublicKey ***************");
                sm3HashDemoWithPublicKey(sdfSDK);
                System.out.println("*************** sm3HashDemo ***************");
                sm3HashDemo(sdfSDK);
                break;
            case "sm4enc":
                sm4EncDemo(sdfSDK);
                break;
            case "sm4mac":
                sm4MacDemo(sdfSDK);
                break;
            default:
                System.out.println("不支持的命令！");
                break;
        }

    }

    private static void random(YunhsmSdfSDK sdfSDK, String[] args) throws SdfSDKException {
        sdfSDK.init();
        int len = 28;
        if (args.length == 2) {
            len = Integer.valueOf(args[1]);
            logger.info("random number length is ", +len);
        }
        byte[] out = sdfSDK.generateRandom(len);
        printHexBinary("generateRandom out", out);
        sdfSDK.release();
    }


    private static void generateKeyPairEcc(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
        sdfSDK.init();
        SdfECCKeyPair sdfECCKeyPair = sdfSDK.generateKeyPairEcc();
        byte[] x = sdfECCKeyPair.getSdfECCPublicKey().getX();
        byte[] y = sdfECCKeyPair.getSdfECCPublicKey().getY();
        printHexBinary("SdfECCPublicKey().getX()", x);
        printHexBinary("SdfECCPublicKey().getY()", y);
        sdfSDK.release();
    }
    /**
     * 导出sm2签名公钥
     **/
    private static void exportSignKey(YunhsmSdfSDK sdfSDK, String[] args) throws SdfSDKException {
        logger.info("*************** exportSM2Key ***************");
        sdfSDK.init();
        int index = 26;
        try {
            index = Integer.valueOf(args[1]);
        } catch (Exception e) {
            //do nothing
        }
        System.out.println(index);
        SdfECCPublicKey signPublicKeyEcc = sdfSDK.exportSignPublicKeyEcc(index);
        GMSSLByteArrayUtils.printHexBinary(logger, "signPublicKeyEcc getX()", signPublicKeyEcc.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "signPublicKeyEcc getY()", signPublicKeyEcc.getY());

        sdfSDK.release();
    }

    /**
     * 导出sm2加密公钥
     **/
    private static void exportEncKey(YunhsmSdfSDK sdfSDK, String[] args) throws SdfSDKException {
        logger.info("*************** exportSM2Key ***************");
        sdfSDK.init();
        int index = 26;
        try {
            index = Integer.valueOf(args[1]);
        } catch (Exception e) {
            //do nothing
        }

        System.out.println(index);

        SdfECCPublicKey eccPublicKey = sdfSDK.exportEncPublicKeyEcc(index);
        GMSSLByteArrayUtils.printHexBinary(logger, "eccPublicKey getX()", eccPublicKey.getX());
        GMSSLByteArrayUtils.printHexBinary(logger, "eccPublicKey getY()", eccPublicKey.getY());

        sdfSDK.release();
    }


    private static void getPrivateKeyAccessRight(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
        sdfSDK.init();
        logger.info("*************** checkPrivateKeyAccessRight ***************");
        sdfSDK.checkPrivateKeyAccessRight(20, "xdja1234".getBytes());
        sdfSDK.release();
    }

    /**
     * SM1 加密解密操作
     **/
    public static void sm1EncDemo(YunhsmSdfSDK sdfSDK) throws SdfSDKException {

        byte[] key = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] data = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] cipher = GMSSLByteArrayUtils.hexDecode("9203a010b50c4e3ca2897308a9d48c2e");
        logger.info(" sm1 CBC encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM1_CBC, key, data, cipher);
        logger.info(" sm1 ECB encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM1_ECB, key, data, cipher);
    }

    /**
     * SM2 签名验签
     **/
    private static void sm2SignDemo(YunhsmSdfSDK sdfSDK) throws Exception {
        //初始化
        sdfSDK.init();
        //导出签名公钥
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportSignPublicKeyEcc(20);
        //待签名数据
        byte[] in = "12345678123456781234567812345678".getBytes();
        printHexBinary("in", in);
        //利用私钥进行内部签名
        SdfECCSignature signature = sdfSDK.internalSignECC(20, "xdja1234".getBytes(), in);
        printHexBinary("signature r", signature.getR());
        printHexBinary("signature s", signature.getS());
        //使用外部公钥进行验签
        sdfSDK.externalVerifyECC(sdfECCPublicKey, in, signature);
        logger.info("Yunhsm Verify Success !");
        sdfSDK.release();
    }

    /**
     * SM2 加密解密操作
     **/
    private static void sm2EncDemo(YunhsmSdfSDK sdfSDK) throws SdfSDKException, IOException {
        //导出公钥
        sdfSDK.init();
        SdfECCPublicKey sdfECCPublicKey = sdfSDK.exportEncPublicKeyEcc(20);
        byte[] plain = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        printHexBinary("plain", plain);
        //使用外部公钥进行加密
        SdfECCCipher eccCipher = sdfSDK.externalEncryptECC(sdfECCPublicKey, plain);
        //SM2解密                                        密文长度
        byte[] decrypt = sdfSDK.internalDecryptECC(20, "xdja1234".getBytes(), eccCipher.getC().length, eccCipher);
        printHexBinary("decrypt", decrypt);
        if (Arrays.equals(plain, decrypt)) {
            logger.info("decrypt data is equals with plain data");
        } else {
            logger.error("decrypt data is not equals with plain data");
        }
        sdfSDK.release();
    }

    /**
     * SM3 HASH摘要算法 withNOPublicKey
     **/
    public static void sm3HashDemo(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
        sdfSDK.init();
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3);
        sdfSDK.hashUpdate("111".getBytes());
        byte[] out = sdfSDK.hashFinal();
        sdfSDK.release();
        printHexBinary("out", out);
    }

    /**
     * SM3 HASH摘要算法 withPublicKey SM2签名算法预处理过程数据 并测试输出是否符合规范
     **/
    public static void sm3HashDemoWithPublicKey(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
        BigInteger SM2_ECC_XA = new BigInteger("09F9DF311E5421A150DD7D161E4BC5C672179FAD1833FC076BB08FF356F35020", 16);
        BigInteger SM2_ECC_YA = new BigInteger("CCEA490CE26775A52DC6EA718CC1AA600AED05FBF35E084A6632F6072DA9AD13", 16);
        byte[] xa = GMSSLByteArrayUtils.hexDecode("09F9DF31 1E5421A1 50DD7D16 1E4BC5C6 72179FAD 1833FC07 6BB08FF3 56F35020");
        byte[] ya = GMSSLByteArrayUtils.hexDecode("CCEA490C E26775A5 2DC6EA71 8CC1AA60 0AED05FB F35E084A 6632F607 2DA9AD13");

        byte[] ID_A = Strings.toByteArray("1234567812345678");
        byte[] msg = Strings.toByteArray("message digest");

        byte[] expecteds = GMSSLByteArrayUtils.hexDecode("F0B43E94 BA45ACCA ACE692ED 534382EB 17E6AB5A 19CE7B31 F4486FDF C0D28640");

        printHexBinary("xa", xa);
        printHexBinary("ya", ya);
        printHexBinary("ID_A", ID_A);
        printHexBinary("message digest", msg);

        sdfSDK.init();
        sdfSDK.hashInit(SdfAlgIdHash.SGD_SM3, ID_A, SdfECCPublicKey.getInstance(new BigInteger(xa), new BigInteger(ya)));
        sdfSDK.hashUpdate(msg);
        byte[] actuals = sdfSDK.hashFinal();
        sdfSDK.release();

        printHexBinary("expecteds", expecteds);
        printHexBinary("actuals", actuals);
        if (Arrays.equals(expecteds, actuals)) {
            logger.info("expecteds data is equals with actuals data");
        } else {
            logger.error("expecteds data is not equals with actuals data");
        }
    }

    /**
     * SM4 加密解密
     **/
    public static void sm4EncDemo(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
        byte[] key = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] data = GMSSLByteArrayUtils.hexDecode("0123456789abcdeffedcba9876543210");
        byte[] cipher = GMSSLByteArrayUtils.hexDecode("681edf34d206965e86b3e94f536e4246");
        logger.info(" sm4 CBC encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM4_CBC, key, data, cipher);
        logger.info(" sm4 ECB encrypt demo ");
        symmetricEncDemo(sdfSDK, SdfAlgIdSymmetric.SGD_SM4_ECB, key, data, cipher);
    }

    public static void sm4MacDemo(YunhsmSdfSDK sdfSDK) throws SdfSDKException {
//        MAC计算	SDF_CalculateMAC
        sdfSDK.init();
        sdfSDK.calculateMac(1);
        sdfSDK.release();
    }

    private static void symmetricEncDemo(YunhsmSdfSDK sdfSDK, SdfAlgIdSymmetric symmetric, byte[] key, byte[] data, byte[] cipher) throws SdfSDKException {

        sdfSDK.init();

        //句柄  导入key获取
        long[] handle = sdfSDK.importKey(key);
        //初始化向量

        byte[] iv = GMSSLByteArrayUtils.hexDecode("00000000000000000000000000000000");
        //加密数据
        byte[] c = sdfSDK.encrypt(handle, symmetric, iv, data);
        printHexBinary("encrypt c", c);

        byte[] p = sdfSDK.decrypt(handle, symmetric, iv, c);
        printHexBinary("decrypt p", p);

        if (Arrays.equals(data, p)) {
            logger.info("decrypt data is equals with plain data");
        } else {
            logger.error("decrypt data is not equals with plain data");
        }

        if (Arrays.equals(cipher, c)) {
            logger.info("encrypt data is equals with expect cipher data");
        } else {
            logger.error("encrypt data is not equals with expect cipher data");
        }
        sdfSDK.release();
    }

    private static void printHexBinary(String message, byte[] data) {
        logger.info(message + " >> Hex len : " + data.length + " data : " + hexEncode(data));
    }


}
