package com.xdja.pki.gmssl.test;

import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCertUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLExtensionUtils;
import org.bouncycastle.asn1.x509.AuthorityKeyIdentifier;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.X509ExtensionUtils;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

public class GMSSLKeyStore {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    private KeyStore serverKeyStore;
    private char[] password;
    private KeyStore trustKeyStore;

    private PrivateKey rootKey;
    private X509Certificate rootCert;
    private PrivateKey signKey;
    private X509Certificate signCert;
    private PrivateKey encKey;
    private X509Certificate encCert;

    public GMSSLKeyStore(String password, PrivateKey rootKey, X509Certificate rootCert, PrivateKey signKey, X509Certificate signCert, PrivateKey encKey, X509Certificate encCert) throws NoSuchAlgorithmException, CertificateException, NoSuchProviderException, KeyStoreException, IOException {
        this.password = password.toCharArray();
        this.rootKey = rootKey;
        this.rootCert = rootCert;
        this.signKey = signKey;
        this.signCert = signCert;
        this.encKey = encKey;
        this.encCert = encCert;

        this.serverKeyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStore(password, rootCert, "sign", signKey, signCert, "enc", encKey, encCert);
        this.trustKeyStore = GMSSLKeyStoreUtils.generateGMSSLTrustStore(rootCert);
    }

    public KeyStore getServerKeyStore() {
        return serverKeyStore;
    }

    public char[] getPassword() {
        return password;
    }

    public KeyStore getTrustKeyStore() {
        return trustKeyStore;
    }

    public PrivateKey getRootKey() {
        return rootKey;
    }

    public X509Certificate getRootCert() {
        return rootCert;
    }

    public PrivateKey getSignKey() {
        return signKey;
    }

    public X509Certificate getSignCert() {
        return signCert;
    }

    public PrivateKey getEncKey() {
        return encKey;
    }

    public X509Certificate getEncCert() {
        return encCert;
    }

    public void save() throws Exception {
        File pathf = new File("." + File.separator + "target" + File.separator + "tomcat" + File.separator + "cert");
        if (!pathf.exists()) {
            pathf.mkdirs();
        }
        String path = pathf.getAbsolutePath() + File.separator;

        FileOutputStream trustOut = new FileOutputStream(path + "trust.keystore");
        this.getTrustKeyStore().store(trustOut, this.getPassword());
        trustOut.close();

        FileOutputStream serverOut = new FileOutputStream(path + "server.keystore");
        this.getServerKeyStore().store(serverOut, this.getPassword());
        serverOut.close();
    }


    public static GMSSLKeyStore generateGMSSLKeyStoreByBC() throws Exception {
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rootDN = "CN=ROOTCA,O=XDJA,C=CN";
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());
        KeyPair rootKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        X509Certificate rootCert = GMSSLCertUtils.generateCertByBC(
                rootDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                rootKeyPair.getPublic(),
                rootExtensions
        );

        JcaX509ExtensionUtils jcaX509ExtensionUtils = new JcaX509ExtensionUtils();
        AuthorityKeyIdentifier authorityKeyIdentifier = jcaX509ExtensionUtils.createAuthorityKeyIdentifier(rootCert);

        //生成 server 密钥、证书
        String signDN = "CN=SIGN,OU=SERVER,O=XDJA,C=CN";
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());
        signExtensions.add(new Extension(Extension.authorityKeyIdentifier, false, authorityKeyIdentifier.getEncoded()));
        KeyPair signKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        X509Certificate signCert = GMSSLCertUtils.generateCertByBC(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                signKeyPair.getPublic(),
                signExtensions
        );

        String encDN = "CN=ENC,OU=SERVER,O=XDJA,C=CN";
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());
        encExtensions.add(new Extension(Extension.authorityKeyIdentifier, false, authorityKeyIdentifier.getEncoded()));
        KeyPair encKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        X509Certificate encCert = GMSSLCertUtils.generateCertByBC(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                encKeyPair.getPublic(),
                encExtensions
        );
        return new GMSSLKeyStore("password", rootKeyPair.getPrivate(), rootCert, signKeyPair.getPrivate(), signCert, encKeyPair.getPrivate(), encCert);
    }

    public static GMSSLKeyStore generateGMSSLKeyStoreByYunhsm(int privateKeyIndex, String privateKeyPassword) throws Exception {
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 1);
        Date noAfter = calendar.getTime();

        String rootDN = "CN=ROOTCA,O=XDJA,C=CN";
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());

        PublicKey rootPublicKey = GMSSLSM2KeyUtils.getSignPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey rootPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate rootCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, rootDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                rootPublicKey,
                rootExtensions
        );

        //生成 server 密钥、证书
        String signDN = "CN=SIGN,O=SERVER,O=XDJA,C=CN";
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());

        PublicKey signPublicKey = GMSSLSM2KeyUtils.getSignPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey signPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate signCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, signDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                signPublicKey,
                signExtensions
        );

        String encDN = "CN=ENC,O=SERVER,O=XDJA,C=CN";
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());

        PublicKey encPublicKey = GMSSLSM2KeyUtils.getEncryptPublicKeyByYunhsm(privateKeyIndex);
        PrivateKey encPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);

        X509Certificate encCert = GMSSLCertUtils.generateCertByYunhsm(
                rootDN, encDN,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                encPublicKey,
                encExtensions
        );

        return new GMSSLKeyStore("password", rootPrivateKey, rootCert, signPrivateKey, signCert, encPrivateKey, encCert);
    }
}
