package com.xdja.SafeKey.utils;

import com.xdja.SafeKey.*;
import com.xdja.SafeKey.utils.pool.MiniPcieConnection;
import com.xdja.SafeKey.utils.pool.MiniPcieConnectionPool;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;

/**
 * @ClassName MiniPcieXKFUtils
 * @Description TODO
 * @Date 2020/5/15 17:46
 * @Author FengZhen
 */
/*
MINIPCi-E
        1、SM1
        2、SM2内部私钥签名，外部公钥验签、SM2外部公钥加密，内部私钥解密
        3、SM2公私钥生成
        4、SM2证书导入，支持SM2双证
        5、SM2加密私钥密文结构体导入（建议MINIPCI-E卡要求的加密私钥密文结构体和PCI-E卡保持一致，减少对接的工作量）
        6、SM2明文公私钥导入
        7、SM3
        8、SM4
*/
public class MiniPcieXKFUtils {
    private static JNIAPI jniapi = new JNIAPI();
    private static MiniPcieConnectionPool connectionPool = MiniPcieConnectionPool.getInstance();
    private static Logger logger = LoggerFactory.getLogger(MiniPcieXKFUtils.class);

    public static byte[] generateRandom(int len) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[len];
        ret = jniapi.GenRandom(calculateHandle.getHandle(), len, out);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.GenRandom(calculateHandle.getHandle(), len, out);
        }
        checkRet("generateRandom", ret, calculateHandle);
        return out;
    }

    public static byte[] sm1(byte[] key, byte[] data, int flag, byte[] iv) throws Exception {
        int ret = 0;
        if (key.length != 16) {
            throw new IllegalArgumentException("sm1 requires a 128 bit key and the key is " + Hex.toHexString(key));
        }
        if (data.length % 16 != 0) {
            throw new IllegalArgumentException("sm1 encrypt data is not multiple of 16 and the data length is " + data.length);
        }
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[data.length];
        ret = jniapi.SM1KEY(calculateHandle.getHandle(), key, data, data.length, flag, out, iv);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM1KEY(calculateHandle.getHandle(), key, data, data.length, flag, out, iv);
        }
        checkRet("sm1", ret, calculateHandle);
        return out;
    }

    // TODO: 2020/5/16 补位
    public static byte[] sm3(byte[] data) {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[32];
        ret = jniapi.SM3Ex(calculateHandle.getHandle(), data, data.length, out);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM3Ex(calculateHandle.getHandle(), data, data.length, out);
        }
        checkRet("sm3", ret, calculateHandle);
        return out;
    }


    public static boolean createFile(MiniPcieFile file) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.CreateFile(calculateHandle.getHandle(), file);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.CreateFile(calculateHandle.getHandle(), file);
        }
        checkRet("createFile", ret, calculateHandle);
        if (ret == 0) {
            return true;
        }
        return false;
    }

    public static byte[] sm2Encrypt(byte[] data, Sm2PublicKey sm2PublicKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[data.length + 32 + 32 + 32 + 1];
        int[] length = new int[out.length];
        ret = jniapi.SM2Encrypt(calculateHandle.getHandle(), Hex.decode("0000"), sm2PublicKey, data, data.length, out, length);
        checkRet("sm2Encrypt", ret, calculateHandle);
        return out;
    }

    public static byte[] sm2EncryptGM(byte[] data, Sm2PublicKey sm2PublicKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[data.length + 32 + 32 + 32 + 1];
        int[] length = new int[out.length];
        ret = jniapi.SM2EncryptGM(calculateHandle.getHandle(), Hex.decode("0000"), sm2PublicKey, data, data.length, out, length);
        checkRet("sm2Encrypt", ret, calculateHandle);
        return out;
    }

    public static byte[] sm2Decrypt(byte[] cipher, byte[] priKeyId, String password) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        byte[] out = new byte[cipher.length - 32 - 32 - 32 - 1];
        int[] length = new int[out.length];
        ret = jniapi.SM2Decrypt(calculateHandle.getHandle(), priKeyId, cipher, cipher.length, out, length);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.SM2Decrypt(calculateHandle.getHandle(), priKeyId, cipher, cipher.length, out, length);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.SM2Decrypt(calculateHandle.getHandle(), priKeyId, cipher, cipher.length, out, length);
        }
        checkRet("sm2Decrypt", ret, calculateHandle);
        return out;
    }

    public static boolean verifyPin(byte[] pin) {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.VerifyPIN(calculateHandle.getHandle(), JNIAPI.ROLE_A, pin, pin.length);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.VerifyPIN(calculateHandle.getHandle(), JNIAPI.ROLE_A, pin, pin.length);
        }
        checkRet("verifyPin", ret, calculateHandle);
        if (ret == 0) {
            return true;
        }
        return false;
    }

    public static boolean verifyPin(byte[] pin, long handle) {
        int ret = jniapi.VerifyPIN(handle, JNIAPI.ROLE_A, pin, pin.length);
        if (ret == 0) {
            logger.info("verify pin success handle = " + handle);
            return true;
        }
        logger.info("verify pin failed handle = " + handle);
        return false;
    }

    public static MiniPcieFile getFileInfo(byte[] fileId) throws Exception {
        int ret = 0;
        MiniPcieFile miniPcieFile = new MiniPcieFile();
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.GetFileInfo(calculateHandle.getHandle(), fileId, miniPcieFile);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.GetFileInfo(calculateHandle.getHandle(), fileId, miniPcieFile);
        }
        checkRet("getFileInfo", ret, calculateHandle);
        if (ret == 0) {
            return miniPcieFile;
        }
        return null;
    }

    public static boolean writeSm2PublicKey(byte[] keyId, String password, Sm2PublicKey publicKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.WriteSm2PubKey(calculateHandle.getHandle(), keyId, publicKey);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.WriteSm2PubKey(calculateHandle.getHandle(), keyId, publicKey);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.WriteSm2PubKey(calculateHandle.getHandle(), keyId, publicKey);
        }
        checkRet("writeSm2PublicKey", ret, calculateHandle);
        if (ret == 0) {
            return true;
        }
        return false;
    }

    public static boolean writeSm2PrivateKey(byte[] keyId, String password, Sm2PrivateKey privateKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.WriteSm2PriKey(calculateHandle.getHandle(), keyId, privateKey);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.WriteSm2PriKey(calculateHandle.getHandle(), keyId, privateKey);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.WriteSm2PriKey(calculateHandle.getHandle(), keyId, privateKey);
        }
        checkRet("writeSm2PrivateKey", ret, calculateHandle);
        if (ret == 0) {
            return true;
        }
        return false;
    }

    public static Sm2PublicKey readSm2PublicKey(byte[] keyId) throws Exception {
        int ret = 0;
        Sm2PublicKey sm2PublicKey = new Sm2PublicKey();
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.ReadSm2PubKey(calculateHandle.getHandle(), keyId, sm2PublicKey);
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin("111111".getBytes(), calculateHandle.getHandle());
            ret = jniapi.ReadSm2PubKey(calculateHandle.getHandle(), keyId, sm2PublicKey);
        }
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.ReadSm2PubKey(calculateHandle.getHandle(), keyId, sm2PublicKey);
        }
        checkRet("readSm2PublicKey", ret, calculateHandle);
        if (ret == 0) {
            return sm2PublicKey;
        }
        return null;
    }

    public static byte[] sm2SignWithInternalHash(byte[] data, byte[] pubKeyId, byte[] priKeyId, String password) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        byte[] sign = new byte[64];
        int[] signLen = new int[1];
        ret = jniapi.SM2SignEx(calculateHandle.getHandle(), pubKeyId, priKeyId, data, data.length, sign, signLen);
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.SM2SignEx(calculateHandle.getHandle(), pubKeyId, priKeyId, data, data.length, sign, signLen);
        }
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM2SignEx(calculateHandle.getHandle(), pubKeyId, priKeyId, data, data.length, sign, signLen);
        }
        checkRet("sm2SignWithInternalHash", ret, calculateHandle);
        return getSignatureEncode(ret, sign);
    }

    public static boolean sm2VerifyWithInternalHash(byte[] data, byte[] sign, Sm2PublicKey sm2PublicKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] signature = getRealSign(sign);
        byte[] pubfids = {0x00, 0x00};
        ret = jniapi.SM2SignVerifyEx(calculateHandle.getHandle(), pubfids, sm2PublicKey, data, data.length, signature);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM2SignVerifyEx(calculateHandle.getHandle(), pubfids, sm2PublicKey, data, data.length, signature);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin("111111".getBytes(), calculateHandle.getHandle());
            ret = jniapi.SM2SignVerifyEx(calculateHandle.getHandle(), pubfids, sm2PublicKey, data, data.length, signature);
        }
        checkRet("sm2VerifyWithInternalHash", ret, calculateHandle);
        if (ret == 0) {
            return true;
        } else {
            logger.error("public key x " + Hex.toHexString(sm2PublicKey.getX()));
            logger.error("public key y " + Hex.toHexString(sm2PublicKey.getY()));
            logger.error("signature   signature.length : " + signature.length + "  code " + Hex.toHexString(signature));
            return false;
        }
    }

    private static byte[] getSignatureEncode(int ret, byte[] sign) throws IOException {
        if (ret == 0) {
            byte[] x = new byte[32];
            byte[] y = new byte[32];
            System.arraycopy(sign, 0, x, 0, 32);
            System.arraycopy(sign, 32, y, 0, 32);
            logger.info("sign " + Hex.toHexString(sign));
            return GMSSLX509Utils.derSignatureEncode(x, y);
        }
        return null;
    }

    //typedef enum
    //{
    //	SIGN_HASH = 0,	//签名数据已经过摘要运算，SM2算法时，表示预处理后的e值
    //	SIGN_NOHASH = 1	//签名数据未经过摘要运算
    //}SIGN_DATA_TYPE;
    public static byte[] sm2SignWithOutInternalHash(byte[] hash, byte[] priKeyId, String password) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        byte[] sign = new byte[64];
        int[] signLen = new int[1];
        ret = jniapi.SM2Sign(calculateHandle.getHandle(), null, priKeyId, JNIAPI.SIGN_HASH, hash, hash.length, sign, signLen);
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.SM2Sign(calculateHandle.getHandle(), null, priKeyId, JNIAPI.SIGN_HASH, hash, hash.length, sign, signLen);
        }
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM2Sign(calculateHandle.getHandle(), null, priKeyId, JNIAPI.SIGN_HASH, hash, hash.length, sign, signLen);
        }
        checkRet("sm2SignWithOutInternalHash", ret, calculateHandle);
        return getSignatureEncode(ret, sign);
    }

    // @param[in] sm2pubkey       签名用公钥，当pubkeyid为0x00 0x00时使用
    public static boolean sm2VerifyWithOutInternalHash(byte[] hash, byte[] sign, Sm2PublicKey sm2PublicKey) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        byte[] signature = getRealSign(sign);
//        byte[] pubfids = {0x00, 0x2a};
        byte[] pubfids = Hex.decode("0000");
        ret = jniapi.SM2SignVerify(calculateHandle.getHandle(), pubfids, JNIAPI.SIGN_HASH, sm2PublicKey, hash, hash.length, signature);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM2SignVerify(calculateHandle.getHandle(), pubfids, JNIAPI.SIGN_HASH, sm2PublicKey, hash, hash.length, signature);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin("111111".getBytes(), calculateHandle.getHandle());
            ret = jniapi.SM2SignVerify(calculateHandle.getHandle(), pubfids, JNIAPI.SIGN_HASH, sm2PublicKey, hash, hash.length, signature);
        }
        checkRet("sm2VerifyWithOutInternalHash", ret, calculateHandle);
        if (ret == 0) {
            return true;
        } else {
            return false;
        }
    }

    private static byte[] getRealSign(byte[] sign) throws Exception {
        BigInteger[] bigIntegers = GMSSLX509Utils.derSignatureDecode(sign);
        byte[] signature = new byte[64];
        byte[] r = GMSSLByteArrayUtils.changeByteArrayLength(bigIntegers[0].toByteArray(), 32);
        byte[] s = GMSSLByteArrayUtils.changeByteArrayLength(bigIntegers[1].toByteArray(), 32);
        GMSSLByteArrayUtils.printHexBinary(null, "r ", r);
        GMSSLByteArrayUtils.printHexBinary(null, "s ", s);
        System.arraycopy(r, 0, signature, 0, 32);
        System.arraycopy(s, 0, signature, 32, 32);
        return signature;
    }

    //* @param[in] pubkeyid        公钥文件ID,为0x00 0x00时表示公钥导出卡外
    //* @param[in] prikeyid        私钥文件ID,公私钥ID均为0x00 0x00时表示私钥导出卡外
    public static KeyPair generateSm2KeyPair() throws Exception {
        return generateEcKeyPair(Hex.decode("0000"), Hex.decode("0000"), GMSSLECUtils.SM2p256);
    }

    public static boolean generateSm2KeyPairInContent(byte[] pubKeyId, byte[] priKeyId) throws Exception {
        int ret = 0;
        Sm2PublicKey sm2PublicKey = new Sm2PublicKey();
        Sm2PrivateKey sm2PrivateKey = new Sm2PrivateKey();
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.GenSM2KeyPair(calculateHandle.getHandle(), pubKeyId, priKeyId, sm2PublicKey, sm2PrivateKey);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.GenSM2KeyPair(calculateHandle.getHandle(), pubKeyId, priKeyId, sm2PublicKey, sm2PrivateKey);
        }
        checkRet("generateEcKeyPair", ret, calculateHandle);
        if (ret == 0) {
            return true;
        }
        return false;
    }

    public static KeyPair generateEcKeyPair(byte[] pubKeyId, byte[] priKeyId, String stdName) throws Exception {
        int ret = 0;
        Sm2PublicKey sm2PublicKey = new Sm2PublicKey();
        Sm2PrivateKey sm2PrivateKey = new Sm2PrivateKey();
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        ret = jniapi.GenSM2KeyPair(calculateHandle.getHandle(), pubKeyId, priKeyId, sm2PublicKey, sm2PrivateKey);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.GenSM2KeyPair(calculateHandle.getHandle(), pubKeyId, priKeyId, sm2PublicKey, sm2PrivateKey);
        }
        checkRet("generateEcKeyPair", ret, calculateHandle);
        if (ret == 0) {
            return new KeyPair(sm2PublicKey.getPublicKey(stdName), sm2PrivateKey.getPrivateKey(stdName));
        }
        return null;
    }


    public static byte[] sm4(byte[] key, byte[] data, int flag, byte[] iv) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getCalculateHandle();
        byte[] out = new byte[data.length];
        ret = jniapi.SM4KEYEx(calculateHandle.getHandle(), key, data, data.length, flag, out, iv);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getCalculateHandle();
            ret = jniapi.SM4KEYEx(calculateHandle.getHandle(), key, data, data.length, flag, out, iv);
        }
        checkRet("sm4", ret, calculateHandle);
        return out;
    }


    public static boolean writeCert(byte[] certEncode, byte[] fidId, String password) throws Exception {
        int ret = 0;
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.WriteCert(calculateHandle.getHandle(), fidId, certEncode, certEncode.length);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.WriteCert(calculateHandle.getHandle(), fidId, certEncode, certEncode.length);
        }
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin(password.getBytes(), calculateHandle.getHandle());
            ret = jniapi.WriteCert(calculateHandle.getHandle(), fidId, certEncode, certEncode.length);
        }
        checkRet("writeCert", ret, calculateHandle);
        if (ret == XkfApiCode.XKR_OK.getCode()) {
            return true;
        } else {
            return false;
        }
    }

    public static byte[] readCert(byte[] fidId) throws Exception {
        int ret = 0;
        byte[] certbuff = new byte[2048];
        int[] certLen = {2048};
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        ret = jniapi.ReadCert(calculateHandle.getHandle(), fidId, certbuff, certLen);
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.ReadCert(calculateHandle.getHandle(), fidId, certbuff, certLen);
        }
        checkRet("readCert", ret, calculateHandle);
        if (ret == XkfApiCode.XKR_OK.getCode()) {
            return certbuff;
        } else {
            return certbuff;
        }
    }

    public static boolean deleteFile(byte[] fileId) throws Exception {
        MiniPcieConnection calculateHandle = connectionPool.getSm2Handle();
        int ret = jniapi.DeleteFile(calculateHandle.getHandle(), fileId);
        if (ret == XkfApiCode.XKR_NO_POWER.getCode()) {
            verifyPin("111111".getBytes(), calculateHandle.getHandle());
            ret = jniapi.DeleteFile(calculateHandle.getHandle(), fileId);
        }
        if (ret == XkfApiCode.XKR_NO_HANDLE.getCode()) {
            connectionPool.closeHandle(calculateHandle);
            calculateHandle = connectionPool.getSm2Handle();
            ret = jniapi.DeleteFile(calculateHandle.getHandle(), fileId);
        }
        checkRet("deleteFile", ret, calculateHandle);
        return true;
    }

    private static void checkRet(String method, int ret, MiniPcieConnection calculateHandle) {
        if (ret != 0) {
            logger.error(method + " error! use dev  = " + calculateHandle.getDevNum() + " handle = " + calculateHandle.getHandle() + " ret = " + ret + " error message  " + XkfApiCode.getErrorMessage(ret));
        } else {
            logger.info(method + " success! use dev  = " + calculateHandle.getDevNum() + " handle = " + calculateHandle.getHandle() + " ret = " + ret + " HEX = " + Integer.toHexString(ret));
        }
        connectionPool.release(calculateHandle);
    }
}
