package com.xdja.pki.ldap.dao;

import com.xdja.pki.ldap.RFC4519StyleUpperCase;
import com.xdja.pki.ldap.X509Utils;
import com.xdja.pki.ldap.config.OpenLDAPConfiguration;
import com.xdja.pki.ldap.config.StoreDRLException;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.CRLNumber;
import org.bouncycastle.asn1.x509.Extension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import sun.security.provider.certpath.X509CertificatePair;


import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;

/**
 * LDAPServer服务调用 访问LDAP服务
 * TODO: 需要考虑多线程并发访问问题
 */
@Repository
public class LDAPDAO implements IDAO {

    private Logger logger = LoggerFactory.getLogger(LDAPDAO.class);

    private SpringLDAPConnect ldapConnect;


    public LDAPDAO(@Autowired OpenLDAPConfiguration openLDAPConfiguration) {
        this.ldapConnect = new SpringLDAPConnect(
                openLDAPConfiguration.getHost(),
                openLDAPConfiguration.getPort(),
                openLDAPConfiguration.getRootDn(),
                openLDAPConfiguration.getRootPassword(),
                openLDAPConfiguration.getContainerName()
        );
    }

    public void init(X509Certificate rootCA) throws Exception {
        logger.debug("-----初始化服务器开始------");
        clear();
        addCertEntry(rootCA);
    }

    public void updateRootCACertificate(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) throws Exception {
        addCertEntry(oldWithNew);
        addCertEntry(newWithOld);
        addCertEntry(newWithNew);
    }

    public void sendCertificate(X509Certificate certificate) throws Exception {
        addCertEntry(certificate);
    }

    public void sendInvaildCertificateSN(String[] sns) {

    }

    public void sendCRL(int crlSegment, X509CRL crl) throws Exception {
        logger.info("clrSegment" + crlSegment);
        logger.info("发布的crl为" + crl);
        addCRLEntry(crlSegment, crl);
    }

    public void sendCrossCertificate(X509CertificatePair pair) throws Exception {
        X509Certificate forward = pair.getForward();
        X509Certificate reserve = pair.getReverse();
        if (forward != null && reserve == null) {
            sendForwardCert(forward);
        } else if (forward == null && reserve != null) {
            sendReserveCert(reserve);
        } else if (forward != null && reserve != null) {
            String dn = X509Utils.getSubjectByX509Cert(forward);
//            dn = X509Utils.transformCommaInDNValues(dn);
            X500Name name = new X500Name(RFC4519StyleUpperCase.INSTANCE, dn);
            String cn = X509Utils.getCommonName(name);
            ldapConnect.addCrossCertEntry(dn, cn, pair.getEncoded());
        } else {
            logger.error("can't insert");
            throw new Exception("can't insert");
        }
//            dn = forward.getSubjectDN().getName();
//            X500Name name = new X500Name(dn);
//            String cn = name.getCommonName();
//            this.ldapConnect.addCrossCertEntry(dn, cn, pair.getEncoded(), true);
//        } else {
//            X509Certificate re = pair.getReverse();
//            if (re != null) {
//                dn = re.getIssuerDN().getName();
//                X500Name name = new X500Name(dn);
//                String cn = name.getCommonName();
//                this.ldapConnect.addCrossCertEntry(dn, cn, pair.getEncoded(), false);
//            } else {
//                logger.error("交叉证书对为空");
//                throw new Exception("pair is null");
//            }
//        }
    }

    private void sendReserveCert(X509Certificate reserve) throws Exception {
        String dn = X509Utils.getIssuerByX509Cert(reserve);
//        dn = X509Utils.transformCommaInDNValues(dn);
        X500Name name = new X500Name(RFC4519StyleUpperCase.INSTANCE, dn);
        String cn = X509Utils.getCommonName(name);
        this.ldapConnect.addReserveCert(dn, cn, reserve);
    }

    private void sendForwardCert(X509Certificate forward) throws Exception {
        String dn = X509Utils.getSubjectByX509Cert(forward);
//        dn = X509Utils.transformCommaInDNValues(dn);
        X500Name name = new X500Name(RFC4519StyleUpperCase.INSTANCE, dn);
        String cn = X509Utils.getCommonName(name);
        this.ldapConnect.addForwardCert(dn, cn, forward);
    }

    private void clear() {
        this.ldapConnect.deleteAll();
    }

    private void addCertEntry(X509Certificate certificate) throws Exception {
        String dn = X509Utils.getSubjectByX509Cert(certificate);
        System.out.println(dn);
//        dn = X509Utils.transformCommaInDNValues(dn);
        LdapName ldapName = new LdapName(dn);

        Rdn endRdn = ldapName.getRdn(ldapName.size() - 1);
        String cn = (String) endRdn.getValue();
        if (X509Utils.isCACertificate(certificate)) {
            this.ldapConnect.addCACertEntry(dn, cn, certificate.getEncoded());
        } else {
            this.ldapConnect.addUserCertEntry(dn, cn, certificate.getEncoded());
        }
    }

    private void addCRLEntry(int clrSegment, X509CRL crl) throws Exception {
        String issuer = X509Utils.getIssuerByX509CRL(crl);
//        issuer = X509Utils.transformCommaInDNValues(issuer);
        LdapName ldapName = new LdapName(issuer);
        Rdn rdn = ldapName.getRdn(ldapName.size() - 1);
        String rdnString = rdn.toString();

        if (X509Utils.isARL(crl)) {
            String cn = "arl" + clrSegment;
            String dn = "cn=" + cn + ",o=" + X509Utils.transformCommaInDNValues(rdn.getValue().toString()) + "ARL" + issuer.replace(rdnString, "");
            logger.debug("开始向LDAP服务器中插入ARL");
            this.ldapConnect.addARLEntry(dn, cn, crl.getEncoded());
        } else if (X509Utils.isDRL(crl)) {
            addDRLEntry(clrSegment, crl);
        } else {
            String cn = "crl" + clrSegment;
            String dn = "cn=" + cn + ",o=" + X509Utils.transformCommaInDNValues(rdn.getValue().toString()) + "CRL" + issuer.replace(rdnString, "");
            logger.debug("开始向LDAP服务器中插入CRL");
            this.ldapConnect.addCRLEntry(dn, cn, crl.getEncoded());
        }
    }

    private void addDRLEntry(int clrSegment, X509CRL crl) throws Exception {
        String issuer = X509Utils.getIssuerByX509CRL(crl);
//        issuer = X509Utils.transformCommaInDNValues(issuer);
        LdapName ldapName = new LdapName(issuer);
        Rdn rdn = ldapName.getRdn(ldapName.size() - 1);
        String rdnString = rdn.toString();
        byte[] deltaCRLIndicator = crl.getExtensionValue(Extension.deltaCRLIndicator.getId());
        byte[] drlBaseOctet = DEROctetString.getInstance(deltaCRLIndicator).getOctets();
        //DRL的增量CRL指示器
        CRLNumber baseCrlNumber = CRLNumber.getInstance(drlBaseOctet);
        //先判断是不是CRL的DRL
        String cn = "crl" + clrSegment;
        String dn = "cn=" + cn + ",o=" + X509Utils.transformCommaInDNValues(rdn.getValue().toString()) + "CRL" + issuer.replace(rdnString, "");
        X509CRL getCrl = ldapConnect.searchCrlEntry(dn, "certificateRevocationList;binary");
        if (null == getCrl) {
            logger.info("get crl is null");
        } else {
            logger.info("get crl is {}", getCrl);
        }
        StoreDRLException status = StoreDRLException.ARL_AND_CRL_NOT_FOUND;
        if (getCrl != null) {
            //查到的CRL
            byte[] extensionValue = getCrl.getExtensionValue(Extension.cRLNumber.getId());
            byte[] octets = DEROctetString.getInstance(extensionValue).getOctets();
            //CRL的CRL数字
            CRLNumber crlInstance = CRLNumber.getInstance(octets);
            System.out.println(getCrl);
            System.out.println(crl);
            logger.info("crlInstance.getCRLNumber().  " + crlInstance.getCRLNumber());
            logger.info("baseCrlNumber.getCRLNumber(). " + baseCrlNumber.getCRLNumber());
            if (crlInstance.getCRLNumber().equals(baseCrlNumber.getCRLNumber())) {
                logger.debug("开始向LDAP服务器中插入DRL");
                this.ldapConnect.addDRLEntry(dn, cn, crl.getEncoded());
                return;
            } else {
                status = StoreDRLException.CRL_NOT_MATCH;
            }
        }
        //判断是不是ARL的DRL
        cn = "arl" + clrSegment;
        dn = "cn=" + cn + ",o=" + X509Utils.transformCommaInDNValues(rdn.getValue().toString()) + "ARL" + issuer.replace(rdnString, "");
        X509CRL getArl = ldapConnect.searchCrlEntry(dn, "authorityRevocationList;binary");
        if (null == getArl) {
            logger.info("get arl is null");
        } else {
            logger.info("get arl is {}", getArl);
        }
        if (getArl != null) {
            //查到的ARL
            byte[] extensionValue = getArl.getExtensionValue(Extension.cRLNumber.getId());
            byte[] octets = DEROctetString.getInstance(extensionValue).getOctets();
            CRLNumber crlInstance = CRLNumber.getInstance(octets);
            if (crlInstance.getCRLNumber().equals(baseCrlNumber.getCRLNumber())) {
                logger.debug("开始向LDAP服务器中插入DRL");
                this.ldapConnect.addDRLEntry(dn, cn, crl.getEncoded());
                return;
            } else {
                if (StoreDRLException.CRL_NOT_MATCH.equals(status)) {
                    status = StoreDRLException.CRL_AND_ARL_NOT_MATCH;
                } else {
                    status = StoreDRLException.ARL_NOT_MATCH;
                }
            }
        }
        logger.error(status.getDescription());
        throw new Exception(status.getDescription());
    }
}