package com.xdja.pki.asn1.issue;

import com.xdja.pki.ldap.X509Utils;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.x509.Certificate;


/***
 * PkixIssue :: = SEQUENCE {
 *     version       Version,               -- 版本号，目前为1
 *     type          INTEGER,               -- 类型
 *                                              0：向 LDAP 或 OCSP 更新根证书
 *                                              1：向 LDAP 发送证书
 *                                              2：向 LDAP 发送作废证书序列号
 *                                              3：向 LDAP 发送作废证书链
 *                                              4：向 OCSP 发送证书状态
 *                                              8：向 LDAP 发送交叉认证证书，LDAP 自己用根证书与此包中的第一个证书组成 PKCS#7 证书发布链
 *     transNonce   OCTET STRING OPTIONAL,  -- 包内随机数 （八位位组串）
 *     number       INTEGER OPTIONAL,       -- 包内证书或证书状态或证书撤销链数目
 *     time         GeneralizedTime,        -- 接收方比较此时间，根据约定时间延迟确定是否接收包内容
 *     cert         [0] SEQUENCE SIZE (1..MAX) OF CERTIFICATE OPTIONAL, -- 如果是证书，放在这项中，
 *                      如果是更新根证书将发布 3 个证书，依次为 OldWithNew、NewWithOld、NewWithNew
 *     certStatus   [1] SEQUENCE OF SEQUENCE {
 *         certId           CertID，
 *         BeforeTime       GeneralizedTime,
 *         EndTime          GeneralizedTime,
 *         status           INTEGER, -- 0 有效，1 无效，无效时有原因项
 *         statusTime       GeneralizedTime,
 *         statusReasonRode CRLReason OPTIONAL
 *     },
 *     mCRL         [2] SEQUENCE OF SEQUENCE {
 *          stateOrProvinceNum   INTEGER,           -- 区域区别号，CRL发布点可以很多，发布内容可以不容，只有唯一地点的为0
 *          CLRSegment           INTEGER,           -- 证书链的块号，为防止证书链太大，一般按证书号分块，同块内的证书在同一条链中
 *          CRLNumber            INTEGER,           -- 证书链的基本序列号
 *          DeltareCRLNumber     INTEGER OPTIONAL,  -- 证书链的增量序列号
 *          CertificateList      SEQUENCE {
 *              tbsCertList TBSCertList,
 *              signatureAlgorithm  AlgorithmIdentifier,
 *              signatureValue  BIT STRING
 *          },
 *      },
 * }
 *
 */
public class TBSIssueRequest extends TBSIssue {

    ASN1Sequence cert;
    ASN1Sequence certStatus;
    ASN1Sequence mCRL;

    public static TBSIssueRequest getInstance(Object o) {
        if (o instanceof TBSIssueRequest) {
            return (TBSIssueRequest) o;
        } else if (o != null) {
            return new TBSIssueRequest(ASN1Sequence.getInstance(o));
        }

        return null;
    }

    public TBSIssueRequest(ASN1Sequence asn1Sequence) {
        super(asn1Sequence);
        while (this.enumeration.hasMoreElements()) {
            ASN1TaggedObject tObj = (ASN1TaggedObject) this.enumeration.nextElement();
            switch (tObj.getTagNo()) {
                case 0:
                    this.cert = ASN1Sequence.getInstance(tObj, true);
                    break;
                case 1:
                    this.certStatus = ASN1Sequence.getInstance(tObj, true);
                    break;
                case 2:
                    this.mCRL = ASN1Sequence.getInstance(tObj, true);
                    break;
                default:
                    break;
            }
        }
    }

    public TBSIssueRequest(ASN1Integer type, ASN1OctetString transNonce, ASN1Integer number, ASN1GeneralizedTime time, Certificate[] cert, CertStatus[] certStatus, ASN1CRL[] mCRL) {
        super(type, transNonce, number, time);
        this.cert = X509Utils.toSequence(cert);
        this.certStatus = X509Utils.toSequence(certStatus);
        this.mCRL = X509Utils.toSequence(mCRL);
    }

    public Certificate[] getCert() {
        if (cert == null) {
            return null;
        }

        Certificate[] results = new Certificate[cert.size()];

        for (int i = 0; i < results.length; i++) {
            results[i] = Certificate.getInstance(cert.getObjectAt(i));
        }
        return results;
    }

    public CertStatus[] getCertStatus() {
        if (certStatus == null) {
            return null;
        }

        CertStatus[] results = new CertStatus[this.certStatus.size()];

        for (int i = 0; i < results.length; i++) {
            results[i] = CertStatus.getInstance(this.certStatus.getObjectAt(i));
        }
        return results;
    }

    public ASN1CRL[] getmCRL() {
        if (this.mCRL == null) {
            return null;
        }

        ASN1CRL[] results = new ASN1CRL[this.mCRL.size()];

        for (int i = 0; i < results.length; i++) {
            results[i] = ASN1CRL.getInstance(this.mCRL.getObjectAt(i));
        }
        return results;
    }

    public ASN1Primitive toASN1Primitive() {
        ASN1EncodableVector vector = this.getVector();
        addOptional(vector, 0, this.cert);
        addOptional(vector, 1, this.certStatus);
        addOptional(vector, 2, this.mCRL);
        return new DERSequence(vector);
    }

    private void addOptional(ASN1EncodableVector v, int tagNo, ASN1Encodable obj) {
        if (obj != null) {
            v.add(new DERTaggedObject(true, tagNo, obj));
        }
    }

}
