package com.xdja.pki.issue;

import com.xdja.pki.asn1.issue.PkixIssue;
import com.xdja.pki.asn1.issue.PkixIssueRequest;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Objects;

public class BasicPkixIssue {

    protected PkixIssue issue;

    private Logger logger = LoggerFactory.getLogger(BasicPkixIssue.class);

    BasicPkixIssue() {

    }

    boolean isSigned() {
        return this.issue.getSignature() != null;
    }

    public ASN1ObjectIdentifier getSignatureAlgOID() {
        if (!this.isSigned()) {
            return null;
        }
        return this.issue.getSignature().getSignatureAlgorithm().getAlgorithm();
    }

    public PkixIssue getPkixIssue() {
        return this.issue;
    }

    public TBSIssueType getTBSIssueType() throws Exception {
        return TBSIssueType.decode(this.issue.getTBSIssue().getType());
    }

    public byte[] getSignatureValues() {
        if (!this.isSigned()) {
            return null;
        }
        return this.issue.getSignature().getSignature().getOctets();
    }

    public boolean isSignatureValid(ContentVerifierProvider verifierProvider) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }

        try {
            ContentVerifier verifier = verifierProvider.get(this.issue.getSignature().getSignatureAlgorithm());
            OutputStream out = verifier.getOutputStream();
            out.write(this.issue.getSignatureInfo());
            return verifier.verify(this.getSignatureValues());
        } catch (Exception e) {
            logger.error("结构体验签失败");
            throw new Exception("exception processing signature: " + e, e);
        }
    }

    public boolean isSignatureValid(KeyPair keyPair) throws Exception {
        return isSignatureValid(keyPair.getPublic());
    }

    // TODO: 2019/5/28 改成 pki-utils 里边方法
    public boolean isSignatureValid(PublicKey key) throws Exception {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(key);
        return isSignatureValid(verifierProvider);
    }


    public boolean verifySignatureByBC(List<X509Certificate> certificates) throws Exception {
        for (int i = 0; i < certificates.size(); i++) {
            boolean flag = verifySignatureByBC(certificates.get(i).getPublicKey());
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public boolean verifySignatureBySdf(List<X509Certificate> certificates, SdfCryptoType sdfCryptoType) throws Exception {
        for (int i = 0; i < certificates.size(); i++) {
            boolean flag = verifySignatureBySdf(certificates.get(i).getPublicKey(), sdfCryptoType);
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public boolean verifySignatureBySdf(PublicKey key, SdfCryptoType sdfCryptoType) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }
        logger.debug("通过" + sdfCryptoType.name() + "进行验签");
        return GMSSLSM2SignUtils.verifyBySdf(sdfCryptoType, key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
    }

    public boolean verifySignatureByBC(PublicKey key) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }
        logger.debug("通过BC进行验签");
        return GMSSLSM2SignUtils.verifyByBC(key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
    }

    public byte[] getEncoded() throws IOException {
        return this.issue.getEncoded();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BasicPkixIssue that = (BasicPkixIssue) o;
        return Objects.equals(issue, that.issue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(issue);
    }

}
