package com.xdja.pki.issue;

import com.xdja.pki.asn1.issue.PkixIssue;
import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.utils.GMSSLECSignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSASignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Objects;

public class BasicPkixIssue {

    public PkixIssue issue; //更改属性为public，兼容OCSP维护算法OID

    private Logger logger = LoggerFactory.getLogger(BasicPkixIssue.class);

    BasicPkixIssue() {

    }

    boolean isSigned() {
        return this.issue.getSignature() != null;
    }

    public ASN1ObjectIdentifier getSignatureAlgOID() {
        if (!this.isSigned()) {
            return null;
        }
        return this.issue.getSignature().getSignatureAlgorithm().getAlgorithm();
    }

    public PkixIssue getPkixIssue() {
        return this.issue;
    }

    public TBSIssueType getTBSIssueType() throws Exception {
        return TBSIssueType.decode(this.issue.getTBSIssue().getType());
    }

    public byte[] getSignatureValues() {
        if (!this.isSigned()) {
            return null;
        }
        return this.issue.getSignature().getSignature().getOctets();
    }

    public boolean isSignatureValid(ContentVerifierProvider verifierProvider) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }

        try {
            ContentVerifier verifier = verifierProvider.get(this.issue.getSignature().getSignatureAlgorithm());
            OutputStream out = verifier.getOutputStream();
            out.write(this.issue.getSignatureInfo());
            return verifier.verify(this.getSignatureValues());
        } catch (Exception e) {
            logger.error("结构体验签失败");
            throw new Exception("exception processing signature: " + e, e);
        }
    }

    public boolean isSignatureValid(KeyPair keyPair) throws Exception {
        return isSignatureValid(keyPair.getPublic());
    }

    // TODO: 2019/5/28 改成 pki-utils 里边方法
    public boolean isSignatureValid(PublicKey key) throws Exception {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(key);
        return isSignatureValid(verifierProvider);
    }


    public boolean verifySignatureByBC(List<X509Certificate> issueCerts, String sigAlgName) throws Exception {
        for (int i = 0; i < issueCerts.size(); i++) {
            boolean flag = verifySignatureByBC(issueCerts.get(i).getPublicKey(),sigAlgName);
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public boolean verifySignatureBySdf(List<X509Certificate> certificates, SdfCryptoType sdfCryptoType) throws Exception {
        for (int i = 0; i < certificates.size(); i++) {
            boolean flag = verifySignatureBySdf(certificates.get(i).getPublicKey(), certificates.get(i).getSigAlgName(), sdfCryptoType);
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public boolean verifySignatureBySdf(PublicKey key, String sigAlgName, SdfCryptoType sdfCryptoType) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }
        logger.debug("通过" + sdfCryptoType.name() + "进行验签");
        if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            return GMSSLSM2SignUtils.verifyBySdf(sdfCryptoType, key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())
                || sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            return GMSSLRSASignUtils.verifyByYunHsm(sigAlgName, key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())){ //add NIST HSM signAlgorithm
            return GMSSLECSignUtils.verifyByYunHsm(key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues(), sigAlgName);
        } else {
            logger.error("暂未未找到" + sigAlgName + "类型验签方式");
            throw new Exception("can't get verify sign with " + sigAlgName + " type");
        }
    }

    public boolean verifySignatureByBC(PublicKey key, String sigAlgName) throws Exception {
        if (!this.isSigned()) {
            logger.error("该结构体没有被签名");
            throw new Exception("attempt to verify signature on unsigned object");
        }
        logger.debug("通过BC进行验签");
        if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            return GMSSLSM2SignUtils.verifyByBC(key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())
                || sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            return GMSSLRSASignUtils.verifyByBC(sigAlgName, key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())){ //add NIST signAlgorithm
            return GMSSLBCSignUtils.verifySignature(sigAlgName, key, this.issue.getTBSIssue().getEncoded(ASN1Encoding.DER), this.getSignatureValues());
        } else {
            logger.error("暂未未找到" + sigAlgName + "类型验签方式");
            throw new Exception("can't get verify sign with " + sigAlgName + " type");
        }
    }

    public byte[] getEncoded() throws IOException {
        return this.issue.getEncoded();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BasicPkixIssue that = (BasicPkixIssue) o;
        return Objects.equals(issue, that.issue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(issue);
    }

}
