package com.xdja.pki.issue;

import com.xdja.pki.asn1.issue.TBSIssue;
import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.utils.GMSSLECSignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSASignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.ocsp.Signature;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x9.X9ObjectIdentifiers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.util.List;

public class BasicPkixIssueBuilder {
    private Logger logger = LoggerFactory.getLogger(BasicPkixIssue.class);
    protected KeyPair keyPair;
    protected List<X509Certificate> caCerts;
    protected int privateKeyIndex;
    protected String privateKeyPassword;
    protected SdfCryptoType sdfCryptoType;
    protected boolean isSignByBC;

    public BasicPkixIssueBuilder(KeyPair keyPair, List<X509Certificate> caCerts) {
        this.keyPair = keyPair;
        this.caCerts = caCerts;
        this.isSignByBC = true;
    }

    public BasicPkixIssueBuilder(int privateKeyIndex, String privateKeyPassword, List<X509Certificate> caCerts, SdfCryptoType sdfCryptoType)  {
        this.privateKeyIndex = privateKeyIndex;
        this.privateKeyPassword = privateKeyPassword;
        this.caCerts = caCerts;
        this.sdfCryptoType = sdfCryptoType;
        this.isSignByBC = false;
    }

    public BasicPkixIssueBuilder(int privateKeyIndex, String privateKeyPassword, List<X509Certificate> caCerts, SdfCryptoType sdfCryptoType, KeyPair keyPair)  {
        this.privateKeyIndex = privateKeyIndex;
        this.privateKeyPassword = privateKeyPassword;
        this.caCerts = caCerts;
        this.sdfCryptoType = sdfCryptoType;
        if (privateKeyPassword == null || keyPair != null){
            this.isSignByBC = true;
        } else {
            this.isSignByBC = false;
        }
        this.keyPair = keyPair;
    }

    // TODO: 2019/3/11 签名算法从证书中读取
    protected Signature getSignature(TBSIssue tbsIssue, String signAlgorithm) throws Exception {
        if (this.isSignByBC) {
            return getSignatureByBC(tbsIssue, signAlgorithm);
        } else {
            return getSignatureBySdf(tbsIssue, signAlgorithm);
        }
    }

    // TODO: 2019/3/11 签名算法从证书中读取
    protected Signature getSignatureByBC(TBSIssue tbsIssue, String signAlgorithm) throws Exception {
        logger.debug(" getSignatureByBC");
        AlgorithmIdentifier signatureAlgorithm = getSignatureWithString(signAlgorithm);
        byte[] data = tbsIssue.getEncoded(ASN1Encoding.DER);
        byte[] signByBC;
        if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            signByBC = GMSSLSM2SignUtils.signByBC(keyPair.getPrivate(), data);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())) { //add NIST signAlgorithm
            signByBC = GMSSLBCSignUtils.generateSignature(signAlgorithm, keyPair.getPrivate(), data);
        } else {
            signByBC = GMSSLRSASignUtils.signByBC(signAlgorithm, keyPair.getPrivate(), data);
        }
        return new Signature(signatureAlgorithm, new DERBitString(signByBC));
    }

    // TODO: 2019/3/11 签名算法从证书中读取
    protected Signature getSignatureBySdf(TBSIssue tbsIssue, String signAlgorithm) throws Exception {
        logger.debug(" getSignatureBySdf {}", this.sdfCryptoType.name());
        AlgorithmIdentifier signatureAlgorithm = getSignatureWithString(signAlgorithm);
        byte[] data = tbsIssue.getEncoded(ASN1Encoding.DER);
        byte[] signBySdf;
        logger.info("get SignAlg is {}", signAlgorithm);
        if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            logger.info("choice sm2");
            signBySdf = GMSSLSM2SignUtils.signBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())) { // add Nist HSM signAlgorithm
            logger.info("choice nist");
            signBySdf = GMSSLECSignUtils.signByYunHsm(privateKeyIndex, privateKeyPassword, data, signAlgorithm);
        } else {
            logger.info("choice rsa");
            signBySdf = GMSSLRSASignUtils.signByYunHsm(signAlgorithm, privateKeyIndex, privateKeyPassword, data);
        }
        return new Signature(signatureAlgorithm, new DERBitString(signBySdf));
    }

    private AlgorithmIdentifier getSignatureWithString(String signAlgorithm) throws Exception {
        AlgorithmIdentifier signatureAlgorithm;
        if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            signatureAlgorithm = new AlgorithmIdentifier(PKCSObjectIdentifiers.sha256WithRSAEncryption);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())) {
            signatureAlgorithm = new AlgorithmIdentifier(PKCSObjectIdentifiers.sha1WithRSAEncryption);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            signatureAlgorithm = new AlgorithmIdentifier(GMObjectIdentifiers.sm2sign_with_sm3);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())){  //add NIST signAlgorithm
            signatureAlgorithm = new AlgorithmIdentifier(X9ObjectIdentifiers.ecdsa_with_SHA256);
        } else {
            throw new Exception("can sign with " + signAlgorithm + " type");
        }
        return signatureAlgorithm;
    }
}
