package com.xdja.pki.issue;

import com.xdja.pki.asn1.issue.ASN1CRL;
import com.xdja.pki.asn1.issue.CertStatus;
import com.xdja.pki.asn1.issue.TBSIssueRequest;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.operator.GMSSLDigestCalculatorProvider;
import com.xdja.pki.ldap.X509Utils;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.ocsp.Signature;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.operator.DefaultDigestAlgorithmIdentifierFinder;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;
import org.bouncycastle.operator.DigestCalculator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.provider.certpath.X509CertificatePair;

import java.io.IOException;
import java.security.KeyPair;
import java.security.NoSuchProviderException;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.List;


// TODO: 2019/2/21 签名运算后期需要调用加密机
public class PkixIssueReqBuilder extends BasicPkixIssueBuilder {

    private Logger logger = LoggerFactory.getLogger(PkixIssueReqBuilder.class);

    public PkixIssueReqBuilder(KeyPair keyPair,
                               List<X509Certificate> caCerts) {
        super(keyPair, caCerts);
    }

    public PkixIssueReqBuilder(int privateKeyIndex,
                               String privateKeyPassword,
                               List<X509Certificate> caCerts,
                               SdfCryptoType sdfCryptoType) {
        super(privateKeyIndex, privateKeyPassword, caCerts, sdfCryptoType);
    }

    public PkixIssueReq build(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) throws Exception {
        if (X509Utils.verifyCert(this.caCerts, newWithOld)
                && X509Utils.verifyCert(newWithNew, newWithOld.getPublicKey())
                && X509Utils.verifyCert(oldWithNew, newWithNew.getPublicKey())) {
            this.caCerts.add(newWithNew);
            return generate(TBSIssueType.UPDATE_ROOT_CERTIFICATE, new Certificate[]{X509Utils.convertCertificate(oldWithNew),
                    X509Utils.convertCertificate(newWithOld), X509Utils.convertCertificate(newWithNew)}, null, null);
        }
        throw new Exception("among sn=" + oldWithNew.getSerialNumber() + " sn=" + newWithOld.getSerialNumber() +" sn=" + newWithNew.getSerialNumber() +" cert signature is not valid");
    }


    public PkixIssueReq build(X509Certificate cert) throws Exception {
        if (X509Utils.verifyCert(this.caCerts, cert)) {
            return generate(TBSIssueType.SEND_CERTIFICATE, new Certificate[]{X509Utils.convertCertificate(cert)}, null, null);
        }
        throw new Exception("sn=" + cert.getSerialNumber() +" cert signature is not valid");
    }

    public PkixIssueReq buildSendCerts(List<X509Certificate> certs) throws Exception {
        Certificate[] certificates = new Certificate[certs.size()];
        for (int i = 0; i < certs.size(); i++) {
            if (X509Utils.verifyCert(this.caCerts, certs.get(i))) {
                certificates[i] = X509Utils.convertCertificate(certs.get(i));
            } else {
                throw new Exception("sn=" + certs.get(i).getSerialNumber() + " cert signature is not valid");
            }
        }
        return generate(TBSIssueType.SEND_CERTIFICATE, certificates, null, null);
    }

    public PkixIssueReq build(int crlSegment, X509CRL crl) throws Exception {
        if (X509Utils.verifyCRL(crl, this.caCerts)) {
            return generate(TBSIssueType.SEND_CRL, null, null, new ASN1CRL[]{generateASN1CRL(crlSegment, crl)});
        }
        throw new Exception("crlSegment=" + crlSegment +" crl signature is not valid");
    }

    public PkixIssueReq build(int stateOrProvinceNum, int crlSegment, X509CRL crl) throws Exception {
        if (X509Utils.verifyCRL(crl, this.caCerts)) {
            return generate(TBSIssueType.SEND_CRL, null, null, new ASN1CRL[]{generateASN1CRL(stateOrProvinceNum, crlSegment, crl)});
        }
        throw new Exception("stateOrProvinceNum=" + stateOrProvinceNum + "crlSegment=" + crlSegment +" crl signature is not valid");
    }

    public PkixIssueReq buildSendCrls(List<PkixIssueCRL> list) throws Exception {
        ASN1CRL[] asn1CRLS = new ASN1CRL[list.size()];
        for (int i = 0; i < list.size(); i++) {
            if (X509Utils.verifyCRL(list.get(i).getCrl(), this.caCerts)) {
                if (0 == list.get(i).getStateOrProvinceNum()) {
                    asn1CRLS[i] = generateASN1CRL(list.get(i).getCrlSegment(), list.get(i).getCrl());
                } else {
                    asn1CRLS[i] = generateASN1CRL(list.get(i).getStateOrProvinceNum(), list.get(i).getCrlSegment(), list.get(i).getCrl());
                }
            } else {
                throw new Exception("stateOrProvinceNum=" + list.get(i).getStateOrProvinceNum() + "crlSegmnet=" + list.get(i).getCrlSegment() + " crl signature is not valid");
            }

        }
        return generate(TBSIssueType.SEND_CRL, null, null, asn1CRLS);
    }

    public PkixIssueReq build(TBSIssueCRLReason reason, X509Certificate certificate) throws Exception {
        if (X509Utils.verifyCert(this.caCerts, certificate)) {
            return generate(TBSIssueType.SEND_CERTIFICATE_STATUS, null, new CertStatus[]{generateCertStatus(reason.getType(),
                    certificate)}, null);
        }
        throw new Exception("sn=" + certificate.getSerialNumber() + " cert signature is not valid");
    }

    /*
     * @MethodName: build
     * @Description: 生成证书状态方法
     * @Param: reason 失效原因
     * @Param: certificate 证书
     * @Param: date 证书状态时间
     * @Return: com.xdja.pki.issue.PkixIssueReq
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/3/12 21:17
    **/
    public PkixIssueReq build(TBSIssueCRLReason reason, X509Certificate certificate, Date date) throws Exception {
        if (X509Utils.verifyCert(this.caCerts, certificate)) {
            return generate(TBSIssueType.SEND_CERTIFICATE_STATUS, null, new CertStatus[]{generateCertStatus(reason.getType(),
                    certificate, date)}, null);
        }
        throw new Exception("sn=" + certificate.getSerialNumber() + "signature is not valid");
    }

    public PkixIssueReq buildSendCertStatuses(List<PkixIssueCertStatus> list) throws Exception {
        CertStatus[] statuses = new CertStatus[list.size()];
        for (int i = 0; i < list.size(); i++) {
            if (X509Utils.verifyCert(this.caCerts, list.get(i).getCertificate())) {
                statuses[i] = generateCertStatus(list.get(i).getReason().getType(), list.get(i).getCertificate(), list.get(i).getStatusTime(), list.get(i).getCertStatus());
            } else {
                throw new Exception("sn=" + list.get(i).getCertificate().getSerialNumber() + " cert signature is not valid");
            }
        }
        return generate(TBSIssueType.SEND_CERTIFICATE_STATUS, null, statuses, null);
    }

    // TODO: 2019/4/1 交叉证书forward dn与当前证书dn相同时候 无法验证
    public PkixIssueReq build(X509CertificatePair pair) throws Exception {
        if (pair.getForward() == null && pair.getReverse() == null) {
            throw new Exception("can not be all null");
        }
        //reserve先验公钥和CA是否相同 证明非自签 不同继续 在进行验签证明
        if (pair.getForward() == null) {
            if (X509Utils.verifyCert(this.caCerts, pair.getReverse())
//                    && !pair.getReverse().getPublicKey().equals(this.certificate.getPublicKey())) {
                    && !X509Utils.verifyPublicKey(this.caCerts, pair.getReverse())) {
                return generate(TBSIssueType.SEND_CROSS_CERTIFICATE, new Certificate[]{
                        X509Utils.convertCertificate(pair.getReverse())
                }, null, null);
            }
            throw new Exception("sn=" + pair.getReverse().getSerialNumber() +" cert signature is not valid");
        }
        if (pair.getReverse() == null) {
            //forward先验公钥和CA是否相同  在进行验签证明非自签
//            if (pair.getForward().getPublicKey().equals(this.certificate.getPublicKey())
            if (X509Utils.verifyPublicKey(this.caCerts, pair.getForward())
                    && !X509Utils.verifyCert(this.caCerts, pair.getForward())) {
                return generate(TBSIssueType.SEND_CROSS_CERTIFICATE, new Certificate[]{
                        X509Utils.convertCertificate(pair.getForward())
                }, null, null);
            }
            throw new Exception("sn=" + pair.getForward().getSerialNumber()+ " cert signature is not valid");
        }
        if (X509Utils.verifyCert(this.caCerts, pair.getReverse())
//                && !pair.getReverse().getPublicKey().equals(this.certificate.getPublicKey())
                && !X509Utils.verifyPublicKey(this.caCerts, pair.getReverse())
//                && pair.getForward().getPublicKey().equals(this.certificate.getPublicKey())
                && X509Utils.verifyPublicKey(this.caCerts, pair.getForward())
                && !X509Utils.verifyCert(this.caCerts, pair.getForward())
        ) {
            return generate(TBSIssueType.SEND_CROSS_CERTIFICATE, new Certificate[]{
                    X509Utils.convertCertificate(pair.getForward()),
                    X509Utils.convertCertificate(pair.getReverse())
            }, null, null);
        }
        throw new Exception("among sn=" + pair.getReverse().getSerialNumber() + " sn=" + pair.getForward().getSerialNumber() +" cert signature is not valid");

    }

    private ASN1CRL generateASN1CRL(int crlSegment, X509CRL crl) throws CRLException, CertificateException, NoSuchProviderException, IOException {
        ASN1Integer stateOrProvinceNum = new ASN1Integer(0);
        ASN1Integer crlSegmentASN1 = new ASN1Integer(crlSegment);
        ASN1Integer cRLNumber = new ASN1Integer(crl.getExtensionValue(Extension.cRLNumber.getId()));
        ASN1Integer deltareCRLNumber = null;
        if (X509Utils.isDRL(crl)) {
            byte[] bytes = crl.getExtensionValue(Extension.deltaCRLIndicator.getId());
            if (bytes != null) {
                deltareCRLNumber = ASN1Integer.getInstance(DEROctetString.getInstance(bytes).getOctets());
            }
        }

        CertificateList certificateList = CertificateList.getInstance(crl.getEncoded());

        return new ASN1CRL(stateOrProvinceNum, crlSegmentASN1, cRLNumber, deltareCRLNumber, certificateList);
    }

    private ASN1CRL generateASN1CRL(int stOrProvinceNum, int crlSegment, X509CRL crl) throws CRLException, CertificateException, NoSuchProviderException, IOException {
        ASN1Integer stateOrProvinceNum = new ASN1Integer(stOrProvinceNum);
        ASN1Integer crlSegmentASN1 = new ASN1Integer(crlSegment);
        ASN1Integer cRLNumber = new ASN1Integer(crl.getExtensionValue(Extension.cRLNumber.getId()));
        ASN1Integer deltareCRLNumber = null;
        if (X509Utils.isDRL(crl)) {
            byte[] bytes = crl.getExtensionValue(Extension.deltaCRLIndicator.getId());
            if (bytes != null) {
                deltareCRLNumber = ASN1Integer.getInstance(DEROctetString.getInstance(bytes).getOctets());
            }
        }

        CertificateList certificateList = CertificateList.getInstance(crl.getEncoded());

        return new ASN1CRL(stateOrProvinceNum, crlSegmentASN1, cRLNumber, deltareCRLNumber, certificateList);
    }

    /**
     * @MethodName: generateCertStatus
     * @Description: 按传入证书状态生成证书
     * @Param: reason
     * @Param: cert
     * @Param: date
     * @Param: certStatus
     * @Return: com.xdja.pki.asn1.issue.CertStatus
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/4/13 9:17
    **/
    private CertStatus generateCertStatus(int reason, X509Certificate cert, Date date, Integer certStatus) throws Exception {
        X509Certificate issueCert = null;
        for (X509Certificate ct : this.caCerts) {
            if (X509Utils.verifyCert(ct, cert)) {
                issueCert = ct;
            }
        }
        if (issueCert == null){
            throw new Exception("can`t find this certificate " + X509Utils.getSubjectByX509Cert(cert) + " issue cert from ca certs, please check init ca certs");
        }
        String sigAlgName = issueCert.getSigAlgName();
        AlgorithmIdentifier sigAlgID = new DefaultSignatureAlgorithmIdentifierFinder().find(sigAlgName);
        AlgorithmIdentifier algorithmIdentifier = new DefaultDigestAlgorithmIdentifierFinder().find(sigAlgID);
        DigestCalculator digestCalculator = new GMSSLDigestCalculatorProvider().get(algorithmIdentifier);
        CertificateID certId = new CertificateID(
                digestCalculator,
                new X509CertificateHolder(issueCert.getEncoded()),
                cert.getSerialNumber()
        );

        ASN1GeneralizedTime beforeTime = new ASN1GeneralizedTime(cert.getNotBefore());
        ASN1GeneralizedTime endTime = new ASN1GeneralizedTime(cert.getNotAfter());
        //根据传入转态生成证书状态
        ASN1Integer asn1Status = new ASN1Integer(certStatus);
        //按照传入时间
        ASN1GeneralizedTime statusTime = new ASN1GeneralizedTime(date);
        CRLReason crlReason = CRLReason.lookup(reason);

        return new CertStatus(certId.toASN1Primitive(), beforeTime, endTime, asn1Status, statusTime, crlReason);
    }

    /*
     * @MethodName: generateCertStatus
     * @Description: 按传入时间生成证书状态
     * @Param: reason
     * @Param: cert
     * @Param: date
     * @Return: com.xdja.pki.asn1.issue.CertStatus
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/3/12 20:57
    **/
    private CertStatus generateCertStatus(int reason, X509Certificate cert, Date date) throws Exception {
        X509Certificate issueCert = null;
        for (X509Certificate ct : this.caCerts) {
            if (X509Utils.verifyCert(ct, cert)) {
                issueCert = ct;
            }
        }
        if (issueCert == null){
            throw new Exception("can`t find this certificate " + X509Utils.getSubjectByX509Cert(cert) + " issue cert from ca certs, please check init ca certs");
        }
        // TODO: 2019/3/27 hash是否使用密码机 （可不使用）
//                new BcDigestCalculatorProvider().get(new DefaultDigestAlgorithmIdentifierFinder().find(this.hashAlgorithm)),
//        String sigAlgName = cert.getSigAlgName();
        String sigAlgName = issueCert.getSigAlgName();
        AlgorithmIdentifier sigAlgID = new DefaultSignatureAlgorithmIdentifierFinder().find(sigAlgName);
        AlgorithmIdentifier algorithmIdentifier = new DefaultDigestAlgorithmIdentifierFinder().find(sigAlgID);
        DigestCalculator digestCalculator = new GMSSLDigestCalculatorProvider().get(algorithmIdentifier);
        CertificateID certId = new CertificateID(
                digestCalculator,
                new X509CertificateHolder(issueCert.getEncoded()),
                cert.getSerialNumber()
        );

        ASN1GeneralizedTime beforeTime = new ASN1GeneralizedTime(cert.getNotBefore());
        ASN1GeneralizedTime endTime = new ASN1GeneralizedTime(cert.getNotAfter());
        //证书状态
        ASN1Integer asn1Status = TBSIssueCRLStatus.INVALID.encode();
        //按照传入时间
        ASN1GeneralizedTime statusTime = new ASN1GeneralizedTime(date);
        CRLReason crlReason = CRLReason.lookup(reason);

        return new CertStatus(certId.toASN1Primitive(), beforeTime, endTime, asn1Status, statusTime, crlReason);
    }

    private CertStatus generateCertStatus(int reason, X509Certificate cert) throws Exception {
        X509Certificate issueCert = null;
        for (X509Certificate ct : this.caCerts) {
            if (X509Utils.verifyCert(ct, cert)) {
                issueCert = ct;
            }
        }
        if (issueCert == null){
            throw new Exception("can`t find this certificate " + X509Utils.getSubjectByX509Cert(cert) + " issue cert from ca certs, please check init ca certs");
        }
        // TODO: 2019/3/27 hash是否使用密码机 （可不使用）
//                new BcDigestCalculatorProvider().get(new DefaultDigestAlgorithmIdentifierFinder().find(this.hashAlgorithm)),
//        String sigAlgName = cert.getSigAlgName();
        String sigAlgName = issueCert.getSigAlgName();
        AlgorithmIdentifier sigAlgID = new DefaultSignatureAlgorithmIdentifierFinder().find(sigAlgName);
        AlgorithmIdentifier algorithmIdentifier = new DefaultDigestAlgorithmIdentifierFinder().find(sigAlgID);
        DigestCalculator digestCalculator = new GMSSLDigestCalculatorProvider().get(algorithmIdentifier);
        CertificateID certId = new CertificateID(
                digestCalculator,
                new X509CertificateHolder(issueCert.getEncoded()),
                cert.getSerialNumber()
        );

        ASN1GeneralizedTime beforeTime = new ASN1GeneralizedTime(cert.getNotBefore());
        ASN1GeneralizedTime endTime = new ASN1GeneralizedTime(cert.getNotAfter());
        //证书状态
        ASN1Integer asn1Status = TBSIssueCRLStatus.INVALID.encode();
        ASN1GeneralizedTime statusTime = new ASN1GeneralizedTime(new Date(System.currentTimeMillis()));
        CRLReason crlReason = CRLReason.lookup(reason);

        return new CertStatus(certId.toASN1Primitive(), beforeTime, endTime, asn1Status, statusTime, crlReason);
    }

    private PkixIssueReq generate(TBSIssueType type, Certificate[] cert, CertStatus[] certStatus, ASN1CRL[] mCRL) throws Exception {
        ASN1Integer typeASN1 = type.encode();
        ASN1OctetString transNonce = PkixIssueUtils.generateTransNonce();
        ASN1GeneralizedTime time = PkixIssueUtils.generateTime();
        ASN1Integer number;
        switch (type) {
            case UPDATE_ROOT_CERTIFICATE:
            case SEND_CERTIFICATE:
            case SEND_CROSS_CERTIFICATE:
                number = new ASN1Integer(cert.length);
                break;
            case SEND_CRL:
                number = new ASN1Integer(mCRL.length);
                break;
            case SEND_CERTIFICATE_STATUS:
                number = new ASN1Integer(certStatus.length);
                break;
            default:
                throw new Exception("unknown type " + type);
        }

        TBSIssueRequest pkixIssueInfo = new TBSIssueRequest(typeASN1, transNonce, number, time, cert, certStatus, mCRL);
        Signature signature = this.getSignature(pkixIssueInfo,this.caCerts.get(0).getSigAlgName());
        return new PkixIssueReq(pkixIssueInfo, signature);
    }
}
