package com.xdja.pki.ldap.sdk.ca;


import com.xdja.pki.asn1.issue.TBSIssueRequest;
import com.xdja.pki.asn1.issue.TBSIssueResponse;
import com.xdja.pki.issue.*;
import com.xdja.pki.ldap.config.HttpRequestHeaderConfig;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;
import sun.security.provider.certpath.X509CertificatePair;

import java.io.IOException;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.List;

public class LDAPCASDK {

    private PkixIssueReqBuilder builder;
    //ldap访问的url
    private String ldapUrl;
    //ocsp访问的url
    private String ocspUrl;
    // ldap服务器签名证书
    private X509Certificate ldapSignCert;
    // ocsp服务器签名证书
    private X509Certificate ocspSignCert;
    /**
     * @param caCert        CA根证书
     * @param cakeyPair     CA根密钥对
     * @param hashAlgorithm ocsp使用的HASH算法
     * @param ldapSignCert  ldap服务器签名证书
     * @param ldapUrl       访问ldapControlller的url
     * @param ocspUrl       访问ocspController的url
     */
    public LDAPCASDK(X509Certificate caCert, KeyPair cakeyPair, String hashAlgorithm, String ldapUrl,
                     String ocspUrl, X509Certificate ldapSignCert, X509Certificate ocspSignCert) {
        this.ldapSignCert = ldapSignCert;
        this.ocspSignCert = ocspSignCert;
        this.ldapUrl = ldapUrl;
        this.ocspUrl = ocspUrl;
        builder = new PkixIssueReqBuilder(cakeyPair, caCert, hashAlgorithm);
    }

    private RestTemplate restTemplate = new RestTemplate();
    //错误原因
    private String reason;

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 0：向LDAP更新根证书
     *
     * @param oldWithNew 新CA密钥给老证书签名的证书
     * @param newWithOld 老CA密钥给新证书签名的证书
     * @param newWithNew 新CA密钥给新证书签名的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse updateRootCACertificateToLDAP(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(oldWithNew, newWithOld, newWithNew);
        } catch (Exception e) {
            reason = "can't build updateRootCACertificateToLDAP PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 0：向OCSP更新根证书
     *
     * @param oldWithNew 新CA密钥给老证书签名的证书
     * @param newWithOld 老CA密钥给新证书签名的证书
     * @param newWithNew 新CA密钥给新证书签名的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse updateRootCACertificateToOCSP(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(oldWithNew, newWithOld, newWithNew);
        } catch (Exception e) {
            reason = "can't build updateRootCACertificateToOCSP PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 1：向LDAP发送证书
     *
     * @param certificate 向LDAP发送的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse sendCertificate(X509Certificate certificate) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(certificate);
        } catch (Exception e) {
            reason = "can't build sendCertificate PkixisssueReq";
            return new LDAPResponse(false, reason);
        }

        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 1：向LDAP发送证书
     *
     * @param certificates 向LDAP发送的证书列表List
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertificate(List<X509Certificate> certificates) {
        PkixIssueReq req = null;
        try {
            req = this.builder.buildSendCerts(certificates);
        } catch (Exception e) {
            reason = "can't build sendCertificateList PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 3：向LDAP发送作废证书链
     *
     * @param crlSegment 向LDAP发送的CRL的块号，一般按照证书号分块
     * @param crl        向LDAP发送的X509CRL类型的CRL （包含ARL，全量CRL及其对应的增量CRL）
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCRL(int crlSegment, X509CRL crl) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(crlSegment, crl);
        } catch (Exception e) {
            reason = "can't build sendCRL  PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }


    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 3：向LDAP发送作废证书链
     *
     * @param list PkixIssueCRL <int, X509CRL>
     *             第一个Integer对应的是CRL块号，一般按照证书号分块
     *             第二个参数X509CRL类型的CRL（包含ARL，全量CRL及其对应的增量CRL）
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCRL(List<PkixIssueCRL> list) {
        PkixIssueReq req = null;
        try {
            req = this.builder.buildSendCrls(list);
        } catch (Exception e) {
            reason = "can't build sendCRLList PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 4：向OCSP发送证书状态
     *
     * @param crlReason 失效原因  可以使用TBSIssueCRLReason枚举进行传值
     * @param cert      发送的证书
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertStatus(TBSIssueCRLReason crlReason, X509Certificate cert) {
        PkixIssueReq req = null;
        try {
            req = builder.build(crlReason, cert);
        } catch (Exception e) {
            reason = "can't build sendCertStatus PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 4：向OCSP发送证书状态
     *
     * @param list PkixIssueCertStatus
     *             包含crlReason  失效原因  可以使用TBSIssueCRLReason枚举进行传值
     *             cert    发送的证书
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertStatus(List<PkixIssueCertStatus> list) {
        PkixIssueReq req = null;
        try {
            req = builder.buildSendCertStatuses(list);
        } catch (Exception e) {
            reason = "can't build sendCertStatusList PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param pair 交叉证书对
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificate(X509CertificatePair pair) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            reason = "can't build sendCrossCertificate PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param forward 交叉证书对的IssueToThisCA且IssueByThisCA为空
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificateIssueToThisCA(X509Certificate forward) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(forward, null);
        } catch (CertificateException e) {
            reason = "can't use IssueToThisCA  build X509CertificatePair";
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            reason = "can't build sendCrossCertificateToCA PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param reverse 交叉证书对的IssueByThisCA且IssueToThisCA为空
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificateIssueByThisCA(X509Certificate reverse) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(null, reverse);
        } catch (CertificateException e) {
            reason = "can't use IssueByThisCA  build X509CertificatePair";
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            reason = "can't build sendCrossCertificateByCA PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param forward IssueToThisCA
     * @param reverse IssueByThisCA 两个证书相互信任
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificate(X509Certificate forward, X509Certificate reverse) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(forward, reverse);
        } catch (CertificateException e) {
            reason = "can't use these two certs build X509CertificatePair";
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            reason = "can't build sendCrossCertificate PkixisssueReq";
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl);
    }

    private LDAPResponse run(PkixIssueReq req, String url) {
        HttpHeaders requestHeaders = HttpRequestHeaderConfig.getHeaders();
        HttpEntity<byte[]> requestEntity = null;
        try {
            requestEntity = new HttpEntity<>(req.getEncoded(), requestHeaders);
        } catch (IOException e) {
            reason = "pkixIssueReq getEncode is failure";
            return new LDAPResponse(false, reason);
        }
        ResponseEntity<byte[]> responseEntity;
        try{
           responseEntity = this.restTemplate.postForEntity(url, requestEntity, byte[].class);
        } catch (Exception e){
            reason = "Connection refused";
         //   e.printStackTrace();
            return new LDAPResponse(false, reason);
        }


        //4xx或者5xx
        if (responseEntity.getStatusCode().is4xxClientError()) {
            reason = "bad request";
            return new LDAPResponse(false, reason);
        }
        if (responseEntity.getStatusCode().is5xxServerError()) {
            reason = "Internal Server Error";
            return new LDAPResponse(false, reason);
        }
        PkixIssueResp resp;
        try {
            resp = new PkixIssueResp(responseEntity.getBody());
        } catch (Exception e) {
            reason = "can't resolve this response struct";
            return new LDAPResponse(false, reason);
        }
        //返回状态是否正常
        TBSIssueResponse respIssue = (TBSIssueResponse) resp.getPkixIssue().getTBSIssue();
        TBSIssueRequest reqIssue = (TBSIssueRequest) req.getPkixIssue().getTBSIssue();
        try {
            if (resp.getStatus().getType() == 1) {
                reason = "response status code is error, please resend";
                return new LDAPResponse(false, reason);
            }
        } catch (Exception e) {
            reason = "unknown response status code type ";
            return new LDAPResponse(false, reason);
        }

        if (!verifyType(reqIssue, respIssue)) {
            reason = "request struct's type is not equals response struct's type";
            return new LDAPResponse(false, reason);
        }
        if (!verifyNumber(reqIssue, respIssue)) {
            reason = "request struct's number is not equals response struct's number";
            return new LDAPResponse(false, reason);
        }
        if (!verifyTranNonce(reqIssue, respIssue)) {
            reason = "request struct's transNonce is not equals response struct's tranNonce";
            return new LDAPResponse(false, reason);
        }

        try {
            if (!verifyTime(respIssue)) {
                reason = "timeout";
                return new LDAPResponse(false, reason);
            }
        } catch (ParseException e) {
            reason = "can't get response body's time";
            return new LDAPResponse(false, reason);
        }

        try {
            if (!verifySign(resp, reqIssue.getType().getValue().intValue())) {
                reason = "signature is not valid";
                return new LDAPResponse(false, reason);
            }
        } catch (Exception e) {
            reason = "can't verify signature";
            return new LDAPResponse(false, reason);
        }
        return new LDAPResponse(true, "success");
    }

    //验证time
    private boolean verifyTime(TBSIssueResponse respIssue) throws ParseException {
        Long nowTime = System.currentTimeMillis();
        Long getTime = respIssue.getTime().getDate().getTime();
        if (nowTime - getTime >= 1000 * 15) {
            return false;
        }
        return true;
    }

    //验证tranNonce是否相等
    private boolean verifyTranNonce(TBSIssueRequest reqIssue, TBSIssueResponse respIssue) {
        if (reqIssue.getTransNonce().equals(respIssue.getTransNonce())) {
            return true;
        }
        return false;
    }

    //验证number是否相等
    private boolean verifyNumber(TBSIssueRequest reqIssue, TBSIssueResponse respIssue) {
        if (reqIssue.getTransNonce().equals(respIssue.getTransNonce())) {
            return true;
        }
        return false;
    }

    //验证type是否相等
    private boolean verifyType(TBSIssueRequest reqIssue, TBSIssueResponse respIssue) {
        if (reqIssue.getType().equals(respIssue.getType())) {
            return true;
        }
        return false;
    }

    //验证type是否相等
    private boolean verifySign(PkixIssueResp resp, int type) throws Exception {
        PublicKey publicKey;
        if (type == 4) {
             publicKey = this.ocspSignCert.getPublicKey();
        } else {
            publicKey = this.ldapSignCert.getPublicKey();
        }
        if (resp.isSignatureValid(publicKey)) {
            return true;
        }
        return false;
    }
}
