package com.xdja.pki.ldap.sdk.ca;


import com.xdja.pki.asn1.issue.TBSIssueRequest;
import com.xdja.pki.asn1.issue.TBSIssueResponse;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.http.GMSSLHttpClient;
import com.xdja.pki.gmssl.http.GMSSLHttpsClient;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpRequest;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpResponse;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpException;
import com.xdja.pki.issue.*;
import com.xdja.pki.ldap.config.HttpRequestHeaderConfig;
import com.xdja.pki.ldap.sdk.ca.bean.BCRequestBean;
import com.xdja.pki.ldap.sdk.ca.bean.RequestType;
import com.xdja.pki.ldap.sdk.ca.bean.YunHsmRequestBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.provider.certpath.X509CertificatePair;

import java.io.IOException;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

public class LDAPCASDK {

    private PkixIssueReqBuilder builder;
    //ldap访问的url
    private String ldapUrl;
    //ocsp访问的url
    private String ocspUrl;
    // ldap服务器签名证书
    private List<X509Certificate> ldapSignCerts;
    // ocsp服务器签名证书
    private List<X509Certificate> ocspSignCerts;
    //
    private int time;


    private boolean isSignByBC;
    private SdfCryptoType sdfCryptoType = SdfCryptoType.YUNHSM;
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<X509Certificate> caCerts;

    public void setTime(int time) {
        this.time = time;
    }

    /**
     * @param caCerts       CA根证书 列表
     * @param cakeyPair     CA根密钥对
     * @param ldapSignCerts ldap服务器签名证书列表
     * @param ocspSignCerts ocsp服务器签名证书列表
     * @param ldapUrl       访问ldapControlller的url
     * @param ocspUrl       访问ocspController的url
     */
    public LDAPCASDK(List<X509Certificate> caCerts, KeyPair cakeyPair, String ldapUrl,
                     String ocspUrl, List<X509Certificate> ldapSignCerts, List<X509Certificate> ocspSignCerts) {
        this.ldapSignCerts = ldapSignCerts;
        this.ocspSignCerts = ocspSignCerts;
        this.ldapUrl = ldapUrl;
        this.ocspUrl = ocspUrl;
        this.isSignByBC = true;
        this.caCerts = caCerts;
        builder = new PkixIssueReqBuilder(cakeyPair, caCerts);
    }

    /**
     * @param caCerts            CA根证书
     * @param privateKeyIndex    CA根证书密钥索引
     * @param privateKeyPassword CA根证书密钥访问控制码
     * @param sdfCryptoType      加密类型 SdfCryptoType.PCIE SdfCryptoType.YUNHSM
     * @param ldapSignCerts      ldap服务器签名证书列表
     * @param ocspSignCerts      ocsp服务器签名证书列表
     * @param ldapUrl            访问ldapControlller的url
     * @param ocspUrl            访问ocspController的url
     */
    public LDAPCASDK(List<X509Certificate> caCerts, int privateKeyIndex, String privateKeyPassword, SdfCryptoType sdfCryptoType, String ldapUrl,
                     String ocspUrl, List<X509Certificate> ldapSignCerts, List<X509Certificate> ocspSignCerts) {
        this.ldapSignCerts = ldapSignCerts;
        this.ocspSignCerts = ocspSignCerts;
        this.ldapUrl = ldapUrl;
        this.ocspUrl = ocspUrl;
        this.isSignByBC = false;
        this.caCerts = caCerts;
        this.sdfCryptoType = sdfCryptoType;
        builder = new PkixIssueReqBuilder(privateKeyIndex, privateKeyPassword, caCerts, sdfCryptoType);
    }

    public LDAPCASDK(BCRequestBean request) {
        this.ldapSignCerts = request.getLdapSignCerts();
        this.ocspSignCerts = request.getOcspSignCerts();
        this.ldapUrl = request.getLdapUrl();
        this.ocspUrl = request.getOcspUrl();
        this.isSignByBC = true;
        this.caCerts = request.getCaCerts();
        builder = new PkixIssueReqBuilder(request.getCakeyPair(), request.getCaCerts());
    }

    public LDAPCASDK(YunHsmRequestBean request) {
        this.ldapSignCerts = request.getLdapSignCerts();
        this.ocspSignCerts = request.getOcspSignCerts();
        this.ldapUrl = request.getLdapUrl();
        this.ocspUrl = request.getOcspUrl();
        this.isSignByBC = false;
        this.sdfCryptoType = request.getSdfCryptoType();
        this.caCerts = request.getCaCerts();
        builder = new PkixIssueReqBuilder(request.getPrivateKeyIndex(), request.getPrivateKeyPassword(), request.getCaCerts(), sdfCryptoType);
    }

    //错误原因
    private String reason;

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 0：向LDAP更新根证书
     *
     * @param oldWithNew 新CA密钥给老证书签名的证书
     * @param newWithOld 老CA密钥给新证书签名的证书
     * @param newWithNew 新CA密钥给新证书签名的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse updateRootCACertificateToLDAP(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(oldWithNew, newWithOld, newWithNew);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build updateRootCACertificateToLDAP PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 0：向OCSP更新根证书
     *
     * @param oldWithNew 新CA密钥给老证书签名的证书
     * @param newWithOld 老CA密钥给新证书签名的证书
     * @param newWithNew 新CA密钥给新证书签名的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse updateRootCACertificateToOCSP(X509Certificate oldWithNew, X509Certificate newWithOld, X509Certificate newWithNew) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(oldWithNew, newWithOld, newWithNew);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build updateRootCACertificateToOCSP PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl, true);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 1：向LDAP发送证书
     *
     * @param certificate 向LDAP发送的证书
     * @return LDAPResponse 暂时返回相应的返回结构体包含是否成功和相应的原因
     */
    public LDAPResponse sendCertificate(X509Certificate certificate) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(certificate);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCertificate PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 1：向LDAP发送证书
     *
     * @param certificates 向LDAP发送的证书列表List
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertificate(List<X509Certificate> certificates) {
        PkixIssueReq req = null;
        try {
            req = this.builder.buildSendCerts(certificates);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCertificateList PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 3：向LDAP发送作废证书链
     *
     * @param crlSegment 向LDAP发送的CRL的块号，一般按照证书号分块
     * @param crl        向LDAP发送的X509CRL类型的CRL （包含ARL，全量CRL及其对应的增量CRL）
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCRL(int crlSegment, X509CRL crl) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(crlSegment, crl);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCRL  PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }


    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 3：向LDAP发送作废证书链
     *
     * @param list PkixIssueCRL <int, X509CRL>
     *             第一个Integer对应的是CRL块号，一般按照证书号分块
     *             第二个参数X509CRL类型的CRL（包含ARL，全量CRL及其对应的增量CRL）
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCRL(List<PkixIssueCRL> list) {
        PkixIssueReq req = null;
        try {
            req = this.builder.buildSendCrls(list);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCRLList PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 4：向OCSP发送证书状态
     *
     * @param crlReason 失效原因  可以使用TBSIssueCRLReason枚举进行传值
     * @param cert      发送的证书
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertStatus(TBSIssueCRLReason crlReason, X509Certificate cert) {
        PkixIssueReq req = null;
        try {
            req = builder.build(crlReason, cert);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCertStatus PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl, true);
    }

    /*
     * @MethodName: sendCertStatus
     * @Description: GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * type 4：向OCSP发送证书状态
     * @Param: crlReason 失效原因
     * @Param: cert 发送的证书
     * @Param: date statusTime 证书状态时间
     * @Return: com.xdja.pki.ldap.sdk.ca.LDAPResponse
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/3/12 21:14
    **/
    public LDAPResponse sendCertStatus(TBSIssueCRLReason crlReason, X509Certificate cert, Date date) {
        PkixIssueReq req = null;
        try {
            req = builder.build(crlReason, cert, date);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCertStatus PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl, true);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 4：向OCSP发送证书状态
     *
     * @param list PkixIssueCertStatus
     *             包含crlReason  失效原因  可以使用TBSIssueCRLReason枚举进行传值
     *             cert    发送的证书
     *             statusTime 证书状态时间
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCertStatus(List<PkixIssueCertStatus> list) {
        PkixIssueReq req = null;
        try {
            req = builder.buildSendCertStatuses(list);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCertStatusList PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ocspUrl, true);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param pair 交叉证书对
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificate(X509CertificatePair pair) {
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCrossCertificate PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param forward 交叉证书对的IssueToThisCA且IssueByThisCA为空
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificateIssueToThisCA(X509Certificate forward) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(forward, null);
        } catch (CertificateException e) {
            logger.error("不能使用该证书构造交叉证书对", e);
            reason = "can't use IssueToThisCA  build X509CertificatePair, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCrossCertificateToCA PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param reverse 交叉证书对的IssueByThisCA且IssueToThisCA为空
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificateIssueByThisCA(X509Certificate reverse) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(null, reverse);
        } catch (CertificateException e) {
            logger.error("不能使用该证书构造交叉证书对", e);
            reason = "can't use IssueByThisCA  build X509CertificatePair, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCrossCertificateByCA PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    /**
     * GM/T 0014-2012 5.3 CA与LDAP服务间相关协议 5.3.2 发布协议
     * TBSISSUE SEQUENCE
     * type 8：向LDAP发送交叉认证证书
     *
     * @param forward IssueToThisCA
     * @param reverse IssueByThisCA 两个证书相互信任
     * @return LDAPResponse 返回相应的返回结构体 包含是否成功和相应的原因
     */
    public LDAPResponse sendCrossCertificate(X509Certificate forward, X509Certificate reverse) {
        X509CertificatePair pair = null;
        try {
            pair = new X509CertificatePair(forward, reverse);
        } catch (CertificateException e) {
            logger.error("不能使用这两个证书构造交叉证书对", e);
            reason = "can't use these two certs build X509CertificatePair, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        PkixIssueReq req = null;
        try {
            req = this.builder.build(pair);
        } catch (Exception e) {
            logger.error("构建PkixIsuse结构体失败", e);
            reason = "can't build sendCrossCertificate PkixisssueReq, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        return run(req, ldapUrl, false);
    }

    private LDAPResponse run(PkixIssueReq req, String url, boolean isOCSPRes) {
        GMSSLHttpRequest request = new GMSSLHttpRequest();
        request.setUrl(url);
        request.setHeaders(HttpRequestHeaderConfig.getHeaders());
        GMSSLHttpClient httpClient;
        if (url.startsWith("https")) {
            try {
                X509Certificate[] certs = new X509Certificate[caCerts.size()];
                for (int i = 0; i < caCerts.size(); i++) {
                    certs[i] = caCerts.get(i);
                }
                String signAlgName = caCerts.get(0).getSigAlgName();
                //SM2硬算法走GMSSLSDFYUNHSMv11通道
                if((!this.isSignByBC) && signAlgName.contains("SM2")){
                    httpClient = new GMSSLHttpsClient(certs, this.sdfCryptoType == SdfCryptoType.YUNHSM);
                } else {
                    httpClient = new GMSSLHttpsClient(certs);
                }
            } catch (GMSSLHttpException e) {
                logger.error("构造 trust keystore 异常", e);
                reason = "generate trust keystore is failure, " + e.getMessage();
                return new LDAPResponse(false, reason);
            }
        } else if (url.startsWith("http")) {
            httpClient = new GMSSLHttpClient();
        } else {
            logger.error("传入的url格式异常" + url);
            reason = "url type is error url = " + url;
            return new LDAPResponse(false, reason);
        }
        GMSSLHttpResponse responseEntity = null;
        try {
            request.setBody(req.getEncoded());
        } catch (IOException e) {
            logger.error("PkixIsuse转换二进制失败", e);
            reason = "pkixIssueReq getEncode is failure, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        try {
            logger.debug("start send post request, url = " + url);
            responseEntity = httpClient.post(request);
        } catch (Exception e) {
            logger.error("连接服务器失败, ", e);
            reason = "Connection refused, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        //4xx或者5xx
        if (responseEntity.getStatusCode() >= 400 && responseEntity.getStatusCode() < 500) {
            logger.error("请求参数有误");
            reason = "bad request";
            return new LDAPResponse(false, reason);
        }
        if (responseEntity.getStatusCode() >= 500 && responseEntity.getStatusCode() < 600) {
            logger.error("服务器初始化错误");
            reason = "Internal Server Error";
            return new LDAPResponse(false, reason);
        }
        PkixIssueResp resp;
        try {
            resp = new PkixIssueResp(responseEntity.getBody());
        } catch (Exception e) {
            logger.error("不能解析PkixIssueRessp结构体 ", e);
            reason = "can't resolve this response struct, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        //返回状态是否正常
        TBSIssueResponse respIssue = (TBSIssueResponse) resp.getPkixIssue().getTBSIssue();
        TBSIssueRequest reqIssue = (TBSIssueRequest) req.getPkixIssue().getTBSIssue();
        try {
            if (resp.getStatus().getType() == 1) {
                logger.error("返回状态码有误，请重新发送数据 ");
                reason = "response status code is error, please resend";
                return new LDAPResponse(false, reason);
            }
        } catch (Exception e) {
            logger.error("未知的返回状态", e);
            reason = "unknown response status code type , " + e.getMessage();
            return new LDAPResponse(false, reason);
        }

        if (!PkixIssueResp.verifyType(reqIssue, respIssue)) {
            logger.error("请求结构体和响应结构体的type不相同");
            reason = "request struct's type is not equals response struct's type";
            return new LDAPResponse(false, reason);
        }
        if (!PkixIssueResp.verifyNumber(reqIssue, respIssue)) {
            logger.error("请求结构体和响应结构体的number不相同");
            reason = "request struct's number is not equals response struct's number";
            return new LDAPResponse(false, reason);
        }
        if (!PkixIssueResp.verifyTranNonce(reqIssue, respIssue)) {
            logger.error("请求结构体和相应结构体的随机数不相同");
            reason = "request struct's transNonce is not equals response struct's tranNonce";
            return new LDAPResponse(false, reason);
        }

        try {
            if (!PkixIssueResp.verifyTime(respIssue, this.time)) {
                logger.error("收到相应的时间超过了设定时间");
                reason = "timeout";
                return new LDAPResponse(false, reason);
            }
        } catch (ParseException e) {
            logger.error("无法从响应结构体得到相应时间", e);
            reason = "can't get response body's time, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }

        try {
            if (!PkixIssueResp.verifySign(resp, isOCSPRes, this.ocspSignCerts, this.ldapSignCerts, this.isSignByBC, this.sdfCryptoType)) {
                logger.error("响应结构体签名无效");
                reason = "signature is not valid";
                return new LDAPResponse(false, reason);
            }
        } catch (Exception e) {
            logger.error("无法进行验签", e);
            reason = "can't verify signature, " + e.getMessage();
            return new LDAPResponse(false, reason);
        }
        logger.info("----------操作成功-----------");
        return new LDAPResponse(true, "success");
    }
}
