package com.xdja.pki.ldap.sdk.ca;

import com.sun.jndi.ldap.LdapURL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.naming.InvalidNameException;
import javax.naming.NamingException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

public class LDAPUrlUtils {

    private static Logger logger = LoggerFactory.getLogger(LDAPUrlUtils.class);

    private LDAPUrlUtils() {

    }

    public static String getHostFromLdapUri(String url) {
        LdapURL ldapURL = null;
        try {
            ldapURL = new LdapURL(url);
            return ldapURL.getHost();
        } catch (NamingException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static int getIPFromLdapUri(String url) throws NamingException {
        LdapURL ldapURL = null;
        try {
            ldapURL = new LdapURL(url);
            return ldapURL.getPort();
        } catch (NamingException e) {
            logger.error("传入的url不是一个正确的LDAPURL", e);
            throw new NamingException("不支持传入的url");
        }
    }

    /**
     * 生成 CA证书 CRLDistributionPoints 扩展项 的LDAP地址
     * CRLDistributionPoints CA
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       颁发者DN
     * @param crlSegment    ARL分块号
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=arl0,o=RootCAARL,O=XDJA,C=CN?authorityRevocationList;binary
     * @throws InvalidNameException new LdapName时产生的异常
     */
    public static String genCertArlLdapUri(String host, int port, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(host, port, issueDn, crlSegment, "authorityRevocationList;binary", true);
        logger.debug("-------创建CA证书的ARL的LDAPUri------- " + uri);
        return uri;
    }
    public static String genCertArlLdapUri(String url, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(url, issueDn, crlSegment, "authorityRevocationList;binary", true);
        logger.debug("-------创建CA证书的ARL的LDAPUri------- " + uri);
        return uri;
    }

    /**
     * 生成 CA证书  FreshestCRL 扩展项 的LDAP地址
     * FreshestCRL
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       颁发者DN
     * @param crlSegment    DRL分块号
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=arl0,o=RootCACRL,O=XDJA,C=CN?deltaRevocationList;binary
     * @throws InvalidNameException new LdapName时产生的异常
     */
    public static String genCertArlDrlLdapUri(String host, int port, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(host, port, issueDn, crlSegment, "deltaRevocationList;binary", true);
        logger.debug("-------创建CA证书的DRL的LDAPUri-------" + uri);
        return uri;
    }

    public static String genCertArlDrlLdapUri(String url, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(url, issueDn, crlSegment, "deltaRevocationList;binary", true);
        logger.debug("-------创建CA证书的DRL的LDAPUri-------" + uri);
        return uri;
    }


    /**
     * 生成 用户证书 CRLDistributionPoints 扩展项 的LDAP地址
     * CRLDistributionPoints User
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       颁发者DN
     * @param crlSegment    CRL分块号
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=crl0,o=RootCACRL,O=XDJA,C=CN?certificateRevocationList;binary
     * @throws InvalidNameException new LdapName时产生的异常
     */
    public static String genCertCrlLdapUri(String host, int port, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(host, port, issueDn, crlSegment, "certificateRevocationList;binary", false);
        logger.debug("-------创建用户证书的CRL的LDAPUrl-------" + uri);
        return uri;
    }

    public static String genCertCrlLdapUri(String url, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(url, issueDn, crlSegment, "certificateRevocationList;binary", false);
        logger.debug("-------创建用户证书的CRL的LDAPUrl-------" + uri);
        return uri;
    }


    /**
     * 生成 用户证书 FreshestCRL 扩展项 的LDAP 地址
     * FreshestCRL
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       颁发者DN
     * @param crlSegment    DRL分块号
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=crl0,o=RootCACRL,O=XDJA,C=CN?deltaRevocationList;binary
     * @throws InvalidNameException new LdapName时产生的异常
     */
    public static String genCertCrlDrlLdapUri(String host, int port, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(host, port, issueDn, crlSegment, "deltaRevocationList;binary", false);
        logger.debug("-------创建用户证书的DRL的LDAPUrl-------" + uri);
        return uri;
    }
    public static String genCertCrlDrlLdapUri(String url, String issueDn, int crlSegment, String containerName) throws InvalidNameException {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCRLExtension(url, issueDn, crlSegment, "deltaRevocationList;binary", false);
        logger.debug("-------创建用户证书的DRL的LDAPUrl-------" + uri);
        return uri;
    }


    /**
     * 生成证书的颁发者机构信息 Authority Information Access 扩展项 的LDAP 地址
     * Authority Information Access  不区分CA证书和用户证书
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       颁发者DN
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=root,O=XDJA,C=CN?cACertificate;binary,crossCertificatePair;binary
     */
    public static String genCertAuthorityLdapUri(String host, int port, String issueDn, String containerName) {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCertExtension(host, port, issueDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建证书颁发机构信息的LDAPUrl-------" + uri);
        return uri;
    }
    public static String genCertAuthorityLdapUri(String url, String issueDn, String containerName) {
        issueDn = checkDn(issueDn, containerName);
        String uri = setCertExtension(url, issueDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建证书颁发机构信息的LDAPUrl-------" + uri);
        return uri;
    }


    /**
     * 生成CA证书的主体访问信息 Subject Information Access 扩展项 的LDAP 地址
     * Subject Information Access  CA
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param subjectDn     CA证书的主体DN
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=root,O=XDJA,C=CN?cACertificate;binary,crossCertificatePair;binary
     */
    public static String genCACertSubjectLdapUri(String host, int port, String subjectDn, String containerName) {
        subjectDn = checkDn(subjectDn, containerName);
        String uri = setCertExtension(host, port, subjectDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建CA证书主体信息的LDAPUrl-------" + uri);
        return uri;
    }
    public static String genCACertSubjectLdapUri(String url, String subjectDn, String containerName) {
        subjectDn = checkDn(subjectDn, containerName);
        String uri = setCertExtension(url, subjectDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建CA证书主体信息的LDAPUrl-------" + uri);
        return uri;
    }

    /**
     * 生成用户证书的主体访问信息 Subject Information Access 扩展项 的LDAP 地址
     * Subject Information Access  User
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param subjectDn     CA证书的主体DN
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=user0,o=研发中心,O=XDJA,C=CN?userCertificate;binary
     */
    public static String genUserCertSubjectLdapUri(String host, int port, String subjectDn, String containerName) {
        subjectDn = checkDn(subjectDn, containerName);
        String uri = setCertExtension(host, port, subjectDn, "userCertificate;binary");
        logger.debug("-------创建用户证书主体信息的LDAPUrl-------" + uri);
        return uri;
    }
    public static String genUserCertSubjectLdapUri(String url, String subjectDn, String containerName) {
        subjectDn = checkDn(subjectDn, containerName);
        String uri = setCertExtension(url, subjectDn, "userCertificate;binary");
        logger.debug("-------创建用户证书主体信息的LDAPUrl-------" + uri);
        return uri;
    }


    /**
     * 生成CRL(不区分ARL，CRL，DRL)的颁发机构访问信息 Authority Information Access 扩展项 的LDAP 地址
     * Authority Information Access  CRL
     *
     * @param host          从LDAP服务的ip地址
     * @param port          从LDAP服务的端口号
     * @param issueDn       CA证书的主体DN
     * @param containerName 颁发者证书的baseDN
     * @return ldap://11.12.85.57:389/cn=root,O=XDJA,C=CN?cACertificate;binary,crossCertificatePair;binary
     */
    public static String genCRLAuthorityLdapUri(String host, int port, String issueDn, String containerName) {

        issueDn = checkDn(issueDn, containerName);
        String uri = setCertExtension(host, port, issueDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建CRL类型颁发机构信息的LDAPUrl-------" + uri);
        return uri;
    }
    public static String genCRLAuthorityLdapUri(String url, String issueDn, String containerName) {

        issueDn = checkDn(issueDn, containerName);
        String uri = setCertExtension(url, issueDn, "cACertificate;binary,crossCertificatePair;binary");
        logger.debug("-------创建CRL类型颁发机构信息的LDAPUrl-------" + uri);
        return uri;
    }

    private static String setCRLExtension(String host, int port, String issueDn, int crlSegment, String name, boolean isArl) throws InvalidNameException {
        LdapName ldapName = new LdapName(issueDn);
        Rdn rdn = ldapName.getRdn(ldapName.size() - 1);
        String rdnString = rdn.toString();
        if (isArl) {
            String cn = "arl" + crlSegment;
            String dn = "cn=" + cn + ",o=" + rdn.getValue() + "ARL" + issueDn.replace(rdnString, "");
            return "ldap://" + host + ":" + port + "/" + dn + "?" + name;
        } else {
            String cn = "crl" + crlSegment;
            String dn = "cn=" + cn + ",o=" + rdn.getValue() + "CRL" + issueDn.replace(rdnString, "");
            return "ldap://" + host + ":" + port + "/" + dn + "?" + name;
        }
    }

    private static String setCRLExtension(String url, String issueDn, int crlSegment, String name, boolean isArl) throws InvalidNameException {
        LdapName ldapName = new LdapName(issueDn);
        Rdn rdn = ldapName.getRdn(ldapName.size() - 1);
        String rdnString = rdn.toString();
        if (isArl) {
            String cn = "arl" + crlSegment;
            String dn = "cn=" + cn + ",o=" + rdn.getValue() + "ARL" + issueDn.replace(rdnString, "");
            return url + "/" + dn + "?" + name;
        } else {
            String cn = "crl" + crlSegment;
            String dn = "cn=" + cn + ",o=" + rdn.getValue() + "CRL" + issueDn.replace(rdnString, "");
            return url + "/" + dn + "?" + name;
        }
    }

    private static String setCertExtension(String host, int port, String dn, String name) {
        String url = "ldap://" + host + ":" + port + "/" + dn + "?" + name;
        return url;
    }

    private static String setCertExtension(String url, String dn, String name) {
        return url + "/" + dn + "?" + name;
    }

    private static String checkDn(String dn, String containerName) {
        if (!dn.toUpperCase().endsWith(containerName.toUpperCase())) {
            dn = dn + "," + containerName;
        }
        dn = dn.replace("dnqualifier".toUpperCase(), "dmdName".toUpperCase());
        return dn;
    }
}
