package com.xdja.pki.ldap.utils;

import com.sun.jndi.ldap.LdapURL;
import com.xdja.pki.asn1.x509.SubjectInformationAccess;
import com.xdja.pki.ldap.X509Utils;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.provider.X509CertPairParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.DirContextAdapter;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.DefaultDirObjectFactory;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.query.LdapQueryBuilder;
import sun.security.provider.certpath.X509CertificatePair;

import javax.naming.NamingEnumeration;
import javax.naming.directory.Attributes;
import java.io.ByteArrayInputStream;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;


public class LDAPUserUtils {

    private static CertificateFactory certificateFactory;
    private static Logger logger = LoggerFactory.getLogger(LDAPUserUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
        try {
            certificateFactory = CertificateFactory.getInstance("X.509", BouncyCastleProvider.PROVIDER_NAME);
        } catch (CertificateException | NoSuchProviderException e) {
            e.printStackTrace();
        }
    }

    private static Attributes search(String url) throws Exception {
        LdapURL ldapURL = new LdapURL(url);
        String[] attrs = ldapURL.getAttributes().split(",");
        LdapContextSource lcs = new LdapContextSource();
        lcs.setUrl("ldap://" + ldapURL.getHost() + ":" + ldapURL.getPort());
        lcs.setDirObjectFactory(DefaultDirObjectFactory.class);
        lcs.afterPropertiesSet();
        LdapTemplate ldapTemplate;
        try {
            ldapTemplate = new LdapTemplate(lcs);
            logger.info("连接服务器成功，开始查询相关内容");
        } catch (Exception e) {
            logger.error("不能连接到服务器");
            throw new Exception("can't connect server");
        }
        Attributes attributes = ldapTemplate.searchForObject(LdapQueryBuilder.query().base(ldapURL.getDN())
                        .attributes(attrs).filter("(objectClass=*)"),
                new ContextMapper<Attributes>() {
                    @Override
                    public Attributes mapFromContext(Object ctx) {
                        return ((DirContextAdapter) ctx).getAttributes();
                    }
                });
        return attributes;
        //ctx -> ((DirContextAdapter) ctx).getAttributes()
    }

    public static List<X509Certificate> searchCert(String url, boolean isCA) throws Exception {
        String search;
        if (isCA) {
            search = "cACertificate;binary";
        } else {
            search = "userCertificate;binary";
        }
        Attributes attributes = search(url);
        List<X509Certificate> certificates = new ArrayList<>();
        NamingEnumeration<String> iDs = attributes.getIDs();
        while (iDs.hasMore()) {
            String id = iDs.next();
            if (id.equalsIgnoreCase(search)) {
                NamingEnumeration<?> all = attributes.get(id).getAll();
                while (all.hasMore()) {
                    X509Certificate certificate = (X509Certificate) certificateFactory.generateCertificate(new ByteArrayInputStream((byte[]) all.next()));
                    certificates.add(certificate);
                }
            }
        }
        return certificates;
    }

    public static X509CRL searchX509CRL(Attributes attributes, String flag) throws Exception {
        NamingEnumeration<String> iDs = attributes.getIDs();
        while (iDs.hasMore()) {
            String id = iDs.next();
            if (id.equalsIgnoreCase(flag)) {
                NamingEnumeration<?> all = attributes.get(id).getAll();
                while (all.hasMore()) {
                    return (X509CRL) certificateFactory.generateCRL(new ByteArrayInputStream((byte[]) all.next()));
                }
            }
        }
        logger.error("没有查到相关 " + flag);
        throw new Exception("没有查到相关" + flag);
    }

    public static X509CRL searchCRL(String url) throws Exception {
        Attributes attributes = search(url);
        return searchX509CRL(attributes, "certificateRevocationList;binary");
    }

    public static X509CRL searchDRL(String url) throws Exception {
        Attributes attributes = search(url);
        return searchX509CRL(attributes, "deltaRevocationList;binary");
    }

    public static X509CRL searchARL(String url) throws Exception {
        Attributes attributes = search(url);
        return searchX509CRL(attributes, "authorityRevocationList;binary");
    }

    public static List<X509CertificatePair> searchCrossCert(String url) throws Exception {
        Attributes attributes = search(url);
        List<X509CertificatePair> pairs = new ArrayList<>();
        NamingEnumeration<String> iDs = attributes.getIDs();
        while (iDs.hasMore()) {
            String id = iDs.next();
            if (id.equalsIgnoreCase("crossCertificatePair;binary")) {
                NamingEnumeration<?> all = attributes.get(id).getAll();
                while (all.hasMore()) {
                    X509CertPairParser pairParser = new X509CertPairParser();
                    pairParser.engineInit(new ByteArrayInputStream((byte[]) all.next()));
                    X509CertificatePair pair = X509Utils.convertCertificatePair((org.bouncycastle.x509.X509CertificatePair) pairParser.engineRead());
                    pairs.add(pair);
                }
            }
        }
        return pairs;
    }


    public static String getSubjectURLWithCert(X509Certificate certificate) throws Exception {
        TBSCertificate tbs = TBSCertificate.getInstance(certificate.getTBSCertificate());
        Extensions extensions = tbs.getExtensions();
        return subjectCertCommon(extensions, "1.3.6.1.5.5.7.48.5");
    }

    public static String getIssueURLWithCert(X509Certificate certificate) throws Exception {
        TBSCertificate tbs = TBSCertificate.getInstance(certificate.getTBSCertificate());
        Extensions extensions = tbs.getExtensions();
        return issueCertCommon(extensions, AccessDescription.id_ad_caIssuers.getId());
    }


    public static String getCRLURLWithCert(X509Certificate certificate) throws Exception {
        byte[] extensionValue = certificate.getExtensionValue(Extension.cRLDistributionPoints.getId());
        logger.debug("开始查询证书中cRLDistributionPoints扩展项");
        return CrlCommon(extensionValue);
    }

    public static String getDRLURLWithCert(X509Certificate certificate) throws Exception {
        byte[] extensionValue = certificate.getExtensionValue(Extension.freshestCRL.getId());
        logger.debug("开始查询证书中freshestCRL扩展项");
        return CrlCommon(extensionValue);
    }

    public static String getIssueURLWithCRL(X509CRL crl) throws Exception {
        TBSCertList tbs = TBSCertList.getInstance(crl.getTBSCertList());
        Extensions extensions = tbs.getExtensions();
        return issueCertCommon(extensions, AccessDescription.id_ad_caIssuers.getId());
    }

    private static String CrlCommon(byte[] extensionValue) throws Exception {
        CRLDistPoint crlDistPoint = CRLDistPoint.getInstance(JcaX509ExtensionUtils.parseExtensionValue(extensionValue));
        DistributionPoint[] distributionPoints = crlDistPoint.getDistributionPoints();
        for (DistributionPoint distributionPoint : distributionPoints) {
            DistributionPointName distributionPoint1 = distributionPoint.getDistributionPoint();
            GeneralNames instance = GeneralNames.getInstance(distributionPoint1.getName().toASN1Primitive());
            GeneralName[] names = instance.getNames();
            for (GeneralName generalName : names) {
                if (generalName.getTagNo() == 6) {
                    DERIA5String str = DERIA5String.getInstance(generalName.getName());
                    return new String(str.getOctets(), "utf-8");
                }
            }
        }
        logger.error("这个证书没有相关扩展项");
        throw new Exception("this cert is not contains cRLDistributionPoints");
    }

    private static String issueCertCommon(Extensions extensions, String id) throws Exception {
        AuthorityInformationAccess access = AuthorityInformationAccess.fromExtensions(extensions);
        AccessDescription[] descs = access.getAccessDescriptions();
        for (AccessDescription desc : descs) {
            if (desc.getAccessMethod().getId().equalsIgnoreCase(id)) {
                GeneralName generalName = GeneralName.getInstance(desc.getAccessLocation());
                DERIA5String instance = DERIA5String.getInstance(generalName.getName());
                return new String(instance.getOctets(), "utf-8");
            }
        }
        logger.error("这个证书没有Authority Information Access扩展项");
        throw new Exception("this cert is not contains Authority Information Access");
    }

    private static String subjectCertCommon(Extensions extensions, String id) throws Exception {
        SubjectInformationAccess access = SubjectInformationAccess.fromExtensions(extensions);
        AccessDescription[] descs = access.getAccessDescriptions();
        for (AccessDescription desc : descs) {
            if (desc.getAccessMethod().getId().equalsIgnoreCase(id)) {
                GeneralName generalName = GeneralName.getInstance(desc.getAccessLocation());
                DERIA5String instance = DERIA5String.getInstance(generalName.getName());
                return new String(instance.getOctets(), "utf-8");
            }
        }
        logger.error("这个证书没有Subject Information Accesss扩展项");
        throw new Exception("this cert is not contains Subject Information Access");
    }
}