package com.xdja.pki.ldap.sdk.user;

import com.xdja.pki.ldap.utils.LDAPUserUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.provider.certpath.X509CertificatePair;

import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class LDAPUserTemplate {
    private static Logger logger = LoggerFactory.getLogger(LDAPUserTemplate.class);


    /**
     * 根据用户证书查询CRL分发节点
     * certificateRevocationList;binary   userCertificate
     *
     * @param certificate 用户证书
     * @return X509CRL
     * @throws Exception 查询时产生的异常
     */
    public static X509CRL searchCrlByUserCert(X509Certificate certificate) throws Exception {
        logger.debug("----------------根据用户证书查询CRL---------------");
        String url = LDAPUserUtils.getCRLURLWithCert(certificate);
        return LDAPUserUtils.searchCRL(url);
    }

    /**
     * 根据CA证书查询ARL分发节点
     * authorityRevocationList;binary   caCertificate
     *
     * @param certificate CA证书
     * @return X509CRL
     * @throws Exception 查询时产生的异常
     */
    public static X509CRL searchArlByCACert(X509Certificate certificate) throws Exception {
        logger.debug("----------------根据CA证书查询ARL---------------");
        String url = LDAPUserUtils.getCRLURLWithCert(certificate);
        return LDAPUserUtils.searchARL(url);
    }

    /**
     * 根据证书查询DRL分发节点
     * deltaRevocationList;binary   caCertificate
     *
     * @param certificate X509证书(CA和用户都可以)
     * @return X509CRL
     * @throws Exception 查询时产生的异常
     */
    public static X509CRL searchDrlByCert(X509Certificate certificate) throws Exception {
        //  String url = "ldap://11.12.85.57:389/cn=crl0,o=rootCAcrl,O=XDJA,C=CN?deltaRevocationList;binary";
        logger.debug("----------------根据证书查询DRL---------------");
        String url = LDAPUserUtils.getDRLURLWithCert(certificate);
        return LDAPUserUtils.searchDRL(url);
    }

    /**
     * 根据CA证书查询CA证书主体信息
     * CACertificate,CrossCertificatePair   caSubject
     *
     * @param certificate CA证书
     * @return Map （包含CACertificate和CrossCertificatePair）
     * @throws Exception 查询时产生的异常
     */
    public static Map searchCASubCertByCACert(X509Certificate certificate) throws Exception {
        logger.debug("----------------根据CA证书查询CA证书主体信息---------------");
        String url = LDAPUserUtils.getSubjectURLWithCert(certificate);
        return searchIssue(url);
    }

    /**
     * 根据用户证书查询用户证书主体信息
     * userCertificate;binary  UserSubject
     *
     * @param certificate 用户证书
     * @return List
     * @throws Exception 查询时产生的异常
     */
    public static List<X509Certificate> searchUserSubCertByCert(X509Certificate certificate) throws Exception {
        logger.debug("----------------根据用户证书查询用户证书主体信息---------------");
        String url = LDAPUserUtils.getSubjectURLWithCert(certificate);
        return LDAPUserUtils.searchCert(url, false);
    }


    /**
     * 根据证书查询证书授权机构信息
     * CACertificate,CrossCertificatePair  Issue
     *
     * @param certificate X509证书(CA和用户都可以)
     * @return Map （包含CACertificate和CrossCertificatePair）
     * @throws Exception 查询时产生的异常
     */
    public static Map searchIssueCertByCert(X509Certificate certificate) throws Exception {
        logger.debug("----------------根据证书查询证书授权机构信息---------------");
        String url = LDAPUserUtils.getIssueURLWithCert(certificate);
        return searchIssue(url);
    }

    /**
     * 根据CRL查询证书授权机构信息
     * CACertificate,CrossCertificatePair  Issue
     *
     * @param crl X509CRL(包含ARL，CRL，DRL)
     * @return Map （包含CACertificate和CrossCertificatePair）
     * @throws Exception 查询时产生的异常
     */
    public static Map searchIssueCertByCRL(X509CRL crl) throws Exception {
        logger.debug("----------------根据CRL查询证书授权机构信息---------------");
        String url = LDAPUserUtils.getIssueURLWithCRL(crl);
        return searchIssue(url);
    }

    private static Map searchIssue(String url) throws Exception {
        List<X509Certificate> certificates = LDAPUserUtils.searchCert(url, true);
        List<X509CertificatePair> pairs = LDAPUserUtils.searchCrossCert(url);
        Map map = new HashMap();
        map.put("CACertificate", certificates);
        map.put("CrossCertificatePair", pairs);
        return map;
    }
}

