package com.xdja.pki.ldap.service;

import com.xdja.pki.asn1.issue.CertStatus;
import com.xdja.pki.issue.PkixIssueCRL;
import com.xdja.pki.issue.PkixIssueReq;
import com.xdja.pki.issue.PkixIssueRespBuilder;
import com.xdja.pki.issue.TBSIssueResponseStatus;
import com.xdja.pki.ldap.X509Utils;
import com.xdja.pki.ldap.config.LDAPConfiguration;
import com.xdja.pki.ldap.dao.IDAO;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import sun.security.provider.certpath.X509CertificatePair;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.List;

@Service
public class OpenLDAPService {

    @Autowired
    private LDAPConfiguration ldapConfiguration;

    @Autowired
    private IDAO idao;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    public byte[] pkixIssue(PkixIssueReq req, PkixIssueRespBuilder respBuilder) throws Exception {
        // insert to ldap
        System.out.println(Hex.toHexString(ldapConfiguration.getSignKey().getPrivate().getEncoded()));
        switch (req.getTBSIssueType()) {
            case UPDATE_ROOT_CERTIFICATE: {
                List<X509Certificate> list = req.getCertificateList();
                logger.debug("UPDATE_ROOT_CERTIFICATE LENGTH " + list.size());
                if (list.size() != 3) {
                    logger.error("更新根证书传递数量有误");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
                try {
                    idao.updateRootCACertificate(req.getCertificateList().get(0), req.getCertificateList().get(1), req.getCertificateList().get(2));
                } catch (Exception e) {
                    logger.error("更新根证书失败", e);
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }

                X509Certificate caNew = req.getCertificateList().get(2);
                ldapConfiguration.updateCaCert(caNew);
                logger.info("更新根证书成功");
                break;
            }
            case SEND_CERTIFICATE: {
                List<X509Certificate> list = req.getCertificateList();
                logger.debug("SEND_CERTIFICATE LENGTH " + list.size());
                for (X509Certificate cert : list) {
                    try {
                        if (X509Utils.isCACertificate(cert)) {
                            ldapConfiguration.updateCaCert(cert);
                        }
                        idao.sendCertificate(cert);
                    } catch (Exception e) {
                        logger.error("发布证书失败", e);
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }
                }
                logger.info("发布证书成功");
                break;
            }
            case SEND_CRL: {
                List<PkixIssueCRL> list = req.getCRLList();
                logger.debug("SEND_CRL " + list.size());
                for (PkixIssueCRL pkixIssueCRL : list) {
                    try {
                        idao.sendCRL(pkixIssueCRL.getCrlSegment(), pkixIssueCRL.getCrl());
                    } catch (Exception e) {
                        logger.error("发布CRL失败", e);
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }
                }
                logger.info("发布crl成功");
                break;
            }
            case SEND_CROSS_CERTIFICATE: {
                List<X509Certificate> list = req.getCertificateList();
                logger.debug("SEND_CROSS_CERTIFICATE LENGTH " + list.size());
                List<X509Certificate> caCerts = ldapConfiguration.getCaCerts();
                //先验签，在对比DN
                if (list.size() == 1) {
                    //证明是issueByThisCA
                    if (X509Utils.verifyCert(caCerts, list.get(0))) {
                        try {
                            logger.debug("发布IssueByThisCA");
                            idao.sendCrossCertificate(new X509CertificatePair(null, list.get(0)));
                        } catch (Exception e) {
                            logger.error("发布交叉证书失败", e);
                            return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                        }
                    } else {
                        try {
                            logger.debug("发布IssueByToCA");
                            idao.sendCrossCertificate(new X509CertificatePair(list.get(0), null));
                        } catch (Exception e) {
                            logger.error("发布交叉证书失败", e);
                            return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                        }
                    }
                } else if (list.size() == 2) {
                    X509CertificatePair pair;
                    try {
                        pair = new X509CertificatePair(list.get(0), list.get(1));
                    } catch (CertificateException e) {
                        // if this two certificate is note cross certificate pair
                        logger.error("发布交叉证书失败，无法通过这两个证书构建交叉证书对", e);
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }
                    try {
                        idao.sendCrossCertificate(pair);
                    } catch (Exception e) {
                        logger.error("发布交叉证书失败", e);
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }

                } else {
                    logger.error("发布交叉证书失败");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
                break;
            }
            case SEND_CERTIFICATE_STATUS: {
                List<CertStatus> certStatusList = req.getCertStatusList();
                break;
            }
            default:
                logger.error("未知的PkixIssue类型");
                throw new Exception("bad request unknown type " + req.getTBSIssueType());
        }
        return respBuilder.build(req, TBSIssueResponseStatus.Normal, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
    }
}
