package com.xdja.pki.ldap.controller;

import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.issue.PkixIssueReq;
import com.xdja.pki.issue.PkixIssueRespBuilder;
import com.xdja.pki.issue.TBSIssueResponseStatus;
import com.xdja.pki.ldap.X509Utils;
import com.xdja.pki.ldap.config.LDAPConfiguration;
import com.xdja.pki.ldap.service.OpenLDAPService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.List;

@RestController
@RequestMapping("/api/v1/ldapserver")
public class LDAPController {

    @Autowired
    private LDAPConfiguration ldapConfiguration;

    @Autowired
    private OpenLDAPService openLDAPService;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @PostMapping("/pkixissue")
    public byte[] pkixissue(@RequestBody byte[] body, HttpServletResponse response) {
        List<X509Certificate> caCerts;
        PkixIssueRespBuilder respBuilder;
        logger.info("------------访问服务器成功------------");
        try {
            caCerts = ldapConfiguration.getCaCerts();
            if ("YUNHSM".equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                        ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.YUNHSM);
                logger.debug("使用加密机验签");
            } else if ("PCIE".equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                        ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.PCIE);
                logger.debug("使用PCIE卡验签");
            } else if ("BC".equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getSignKey(), caCerts);
                logger.debug("使用BC模式验签");
            } else {
                logger.error("配置文件验签方式配置有误");
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                return new byte[0];
            }
        } catch (Exception e) {
            logger.error("读取配置证书失败", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }

        PkixIssueReq req;
        try {
            req = new PkixIssueReq(body);
            logger.info("请求结构体类型为 " + req.getTBSIssueType());
            if ("PCIE".equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.PCIE)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
            } else if ("YUNHSM".equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.YUNHSM)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
            } else {
                if (!req.verifySignatureByBC(caCerts, caCerts.get(0).getSigAlgName())) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
            }
        } catch (Exception e) {
            logger.error("无法解析请求结构体 ", e);
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            return new byte[0];
        }

        try {
            return openLDAPService.pkixIssue(req, respBuilder);
        } catch (Exception e) {
            logger.error("向LDAP存储失败 ", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }
    }

}
