package com.xdja.pki.ldap.controller;

import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.utils.*;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import com.xdja.pki.issue.PkixIssueReq;
import com.xdja.pki.issue.PkixIssueRespBuilder;
import com.xdja.pki.issue.TBSIssueResponseStatus;
import com.xdja.pki.ldap.CryptoTypeStr;
import com.xdja.pki.ldap.X509Utils;
import com.xdja.pki.ldap.config.ErrorEnum;
import com.xdja.pki.ldap.config.LDAPConfiguration;
import com.xdja.pki.ldap.service.OpenLDAPService;
import org.bouncycastle.asn1.ocsp.Signature;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.*;

@RestController
@RequestMapping("/api/v1/ldapserver")
public class LDAPController {

    @Autowired
    private LDAPConfiguration ldapConfiguration;

    @Autowired
    private OpenLDAPService openLDAPService;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    //@PostMapping("/pkixissue")
    @Deprecated
    public byte[] pkixissue(@RequestBody byte[] body, HttpServletResponse response) {
        List<X509Certificate> caCerts;
        PkixIssueRespBuilder respBuilder;
        logger.info("------------访问服务器成功------------");
        try {
            caCerts = ldapConfiguration.getCaCerts();
            if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                if (ldapConfiguration.getPrivateKeyPassword() == null) {
                    KeyPair keyPair = ldapConfiguration.getSignKey();
                    respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                            ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.YUNHSM, keyPair);
                    if (logger.isDebugEnabled()) {
                        logger.debug("使用加密机验签");
                    }
                } else {
                    respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                            ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.YUNHSM);
                    if (logger.isDebugEnabled()) {
                        logger.debug("使用加密机验签");
                    }
                }
            } else if (CryptoTypeStr.PCIE.equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                        ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.PCIE);
                if (logger.isDebugEnabled()) {
                    logger.debug("使用PCIE卡验签");
                }
            } else if (CryptoTypeStr.BC.equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getSignKey(), caCerts);
                if (logger.isDebugEnabled()) {
                    logger.debug("使用BC模式验签");
                }
            } else {
                logger.error("配置文件验签方式配置有误");
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                return new byte[0];
            }
        } catch (Exception e) {
            logger.error("读取配置证书失败", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }

        PkixIssueReq req;
        try {
            req = new PkixIssueReq(body);
            logger.info("请求结构体类型为 {}", req.getTBSIssueType());

            if (CryptoTypeStr.PCIE.equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.PCIE)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
            } else if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(ldapConfiguration.getCryptoType())) {
                if (ldapConfiguration.getPrivateKeyPassword() != null) {
                    if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.YUNHSM)) {
                        logger.error("请求结构体签名无效");
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }
                } else {
                    if (!req.verifySignatureByBC(caCerts)) {
                        logger.error("请求结构体签名无效");
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                    }
                }
            } else {
                if (!req.verifySignatureByBC(caCerts)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapConfiguration.getSignCert().getSigAlgName()).getEncoded();
                }
            }
        } catch (Exception e) {
            logger.error("无法解析请求结构体 ", e);
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            return new byte[0];
        }

        try {
            return openLDAPService.pkixIssue(req, respBuilder);
        } catch (Exception e) {
            logger.error("向LDAP存储失败 ", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }
    }

    @PostMapping("/pkixissue")
    public byte[] pkixIssue(@RequestBody byte[] body, HttpServletRequest request, HttpServletResponse response) {
        List<X509Certificate> caCerts;
        List<Object> singleCAToLdap;
        PkixIssueRespBuilder respBuilder;
        X509Certificate caCertInSignature;
        String caKeyStr;
        PkixIssueReq req;
        Map<String, Object> caToLdapMap;
        List<List<X509Certificate>> nestCaList;
        try {
            req = new PkixIssueReq(body);
        } catch (Exception e) {
            logger.error("无法解析请求结构体 ", e);
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            return new byte[0];
        }
        try {
            Signature caSignature = req.issue.getSignature();
            caCertInSignature = X509Utils.getCertFromSignature(caSignature);
            caKeyStr = X509Utils.getKeyFromCertificate(caCertInSignature);
            nestCaList = ldapConfiguration.getAllCAListFromPath(ldapConfiguration.getCaCert());
            caToLdapMap = ldapConfiguration.getCAToLDAPMap(nestCaList);
            singleCAToLdap = (List<Object>) caToLdapMap.get(caKeyStr);
            if (singleCAToLdap == null || singleCAToLdap.isEmpty()) {
                singleCAToLdap = X509Utils.getListFromCAToOCSPMap(caToLdapMap, nestCaList, caKeyStr);
            }
            if (singleCAToLdap == null || singleCAToLdap.isEmpty()) {
                logger.error("签名CA证书未配置");
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                return new byte[0];
            }
            caCerts = (List<X509Certificate>) singleCAToLdap.get(0);
        } catch (Exception e) {
            logger.error("获取签名CA证书异常", e);
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            return new byte[0];
        }
        logger.info("------------访问服务器成功------------");
        String configCryptoType = ldapConfiguration.getCryptoType();
        X509Certificate ldapSignCert = (X509Certificate) singleCAToLdap.get(1);
        String cryptoType;
        if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(configCryptoType) && !ldapSignCert.getSigAlgName().contains("SM2")) {
            cryptoType = CryptoTypeStr.BC;
        } else {
            cryptoType = configCryptoType;
        }
        //初始化模块
        ldapConfiguration.initInstance(cryptoType);
        Object kpOrPk = singleCAToLdap.get(2);
        try {
            if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(cryptoType)) {
                if (ldapConfiguration.getPrivateKeyPassword() == null) {
                    respBuilder = new PkixIssueRespBuilder(ldapConfiguration.getPrivateKeyIndex(),
                            ldapConfiguration.getPrivateKeyPassword(), caCerts, SdfCryptoType.YUNHSM, (KeyPair) kpOrPk);
                    if (logger.isDebugEnabled()) {
                        logger.debug("使用加密机验签");
                    }
                } else {
                    String[] privateKey = (String[]) kpOrPk;
                    respBuilder = new PkixIssueRespBuilder(Integer.parseInt(privateKey[0]), privateKey[1], caCerts, SdfCryptoType.YUNHSM);
                    if (logger.isDebugEnabled()) {
                        logger.debug("使用加密机验签");
                    }
                }
            } else if (CryptoTypeStr.PCIE.equalsIgnoreCase(cryptoType)) {
                String[] privateKey = (String[]) kpOrPk;
                respBuilder = new PkixIssueRespBuilder(Integer.parseInt(privateKey[0]), privateKey[1], caCerts, SdfCryptoType.PCIE);
                if (logger.isDebugEnabled()) {
                    logger.debug("使用PCIE卡验签");
                }
            } else if (CryptoTypeStr.BC.equalsIgnoreCase(cryptoType)) {
                respBuilder = new PkixIssueRespBuilder((KeyPair) kpOrPk, caCerts);
                if (logger.isDebugEnabled()) {
                    logger.debug("使用BC模式验签");
                }
            } else {
                logger.error("配置文件验签方式配置有误");
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                return new byte[0];
            }
        } catch (Exception e) {
            logger.error("读取配置证书失败", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }

        try {
            logger.info("请求结构体类型为 {}", req.getTBSIssueType());
            if (CryptoTypeStr.PCIE.equalsIgnoreCase(cryptoType)) {
                if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.PCIE)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapSignCert).getEncoded();
                }
            } else if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(cryptoType)) {
                if (ldapConfiguration.getPrivateKeyPassword() != null) {
                    if (!req.verifySignatureBySdf(caCerts, SdfCryptoType.YUNHSM)) {
                        logger.error("请求结构体签名无效");
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapSignCert).getEncoded();
                    }
                } else {
                    if (!req.verifySignatureByBC(caCerts)) {
                        logger.error("请求结构体签名无效");
                        return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapSignCert).getEncoded();
                    }
                }
            } else {
                if (!req.verifySignatureByBC(caCerts)) {
                    logger.error("请求结构体签名无效");
                    return respBuilder.build(req, TBSIssueResponseStatus.Error, ldapSignCert).getEncoded();
                }
            }
        } catch (Exception e) {
            logger.error("无法解析请求结构体 ", e);
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            return new byte[0];
        }

        try {
            return openLDAPService.pkixIssue(req, respBuilder, ldapSignCert);
        } catch (Exception e) {
            logger.error("向LDAP存储失败 ", e);
            response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            return new byte[0];
        }
    }

    /**
     * 测试LDAP连通
     *
     * @param resp
     * @return
     * @throws
     */
    @RequestMapping(value = "/pkixissue/test", method = RequestMethod.POST)
    public Object caConnectLDAPTest(@RequestParam String sign, String keyStr, HttpServletResponse resp) {
        try {
            List<List<X509Certificate>> nestCaList = ldapConfiguration.getAllCAListFromPath(ldapConfiguration.getCaCert());
            Map<String, Object> caToLdapMap = ldapConfiguration.getCAToLDAPMap(nestCaList);
            List<Object> singleCAToLdap = (List<Object>) caToLdapMap.get(keyStr);
            if (singleCAToLdap == null || singleCAToLdap.isEmpty()) {
                singleCAToLdap = X509Utils.getListFromCAToOCSPMap(caToLdapMap, nestCaList, keyStr);
            }
            String cryptoType = ldapConfiguration.getCryptoType();
            List<X509Certificate> caCerts = (List<X509Certificate>) singleCAToLdap.get(0);
            if (caCerts == null || caCerts.isEmpty()) {
                logger.error("签名CA证书未配置");
                return ErrorEnum.LDAP_INTERNAL_EXCEPTION.resp(resp);
            }
            String sdfCryptoType;
            if (CryptoTypeStr.YUNHSM.equalsIgnoreCase(cryptoType) && !caCerts.get(0).getSigAlgName().contains("SM2")) {
                sdfCryptoType = CryptoTypeStr.BC;
            } else {
                sdfCryptoType = cryptoType;
            }
            //将空格转为+
            sign = sign.replace(" ", "+");
            boolean flag = verifyTestSign(sign, sdfCryptoType, caCerts);
            if (!flag) {
                logger.error("测试LDAP连通验签失败");
                return ErrorEnum.SIGN_VERIFY_FAIL.resp(resp);
            } else {
                logger.info("测试LDAP连通验签通过");
            }

        } catch (Exception e) {
            logger.error("测试LDAP连通出现异常");
            return ErrorEnum.LDAP_INTERNAL_EXCEPTION.resp(resp);
        }
        logger.info("测试LDAP连通性通过");
        return null;
    }

    private boolean verifyTestSign(String sign, String sdfCryptoType, List<X509Certificate> caCerts) throws Exception {
        String sigAlgName = caCerts.get(caCerts.size() - 1).getSigAlgName();
        String sourceData = "LDAP";
        if (CryptoTypeStr.BC.equalsIgnoreCase(sdfCryptoType)) {
            if (logger.isDebugEnabled()) {
                logger.debug("使用BC进行验签");
            }
            return verifySignatureByBC(caCerts, sigAlgName, sourceData, sign);
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("使用HSM进行验签");
            }
            SdfCryptoType sdfCrypto = SdfCryptoType.YUNHSM;
            return verifySignatureBySdf(caCerts, sdfCrypto, sourceData, sign);
        }
    }

    public boolean verifySignatureByBC(List<X509Certificate> issueCerts, String sigAlgName, String sourceData, String sign) throws Exception {
        for (int i = 0; i < issueCerts.size(); i++) {
            boolean flag = verifySignatureByBC(issueCerts.get(i).getPublicKey(), sigAlgName, sourceData, sign);
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public boolean verifySignatureBySdf(List<X509Certificate> certificates, SdfCryptoType sdfCryptoType, String sourceData, String sign) throws Exception {
        for (int i = 0; i < certificates.size(); i++) {
            boolean flag = verifySignatureBySdf(certificates.get(i).getPublicKey(), certificates.get(i).getSigAlgName(), sdfCryptoType, sourceData, sign);
            if (flag) {
                return true;
            }
        }
        return false;
    }

    private boolean verifySignatureBySdf(PublicKey key, String sigAlgName, SdfCryptoType sdfCryptoType, String sourceData, String sign) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("通过 {} 进行验签", sdfCryptoType.name());
        }
        String base64Source = Base64.toBase64String(sourceData.getBytes());
        if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            return GMSSLSM2SignUtils.verifyBySdf(sdfCryptoType, key, base64Source, sign);
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())
                || sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            return GMSSLRSASignUtils.verifyByYunHsm(sigAlgName, key, base64Source, sign);
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())) { //add NIST HSM signAlgorithm
            return GMSSLECSignUtils.verifyByYunHsm(key, base64Source, sign, sigAlgName);
        } else {
            logger.error("暂未未找到 {} 类型验签方式", sigAlgName);
            throw new Exception(String.format("can't get verify sign with %s type", sigAlgName));
        }
    }

    private boolean verifySignatureByBC(PublicKey key, String sigAlgName, String sourceData, String sign) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("通过BC进行验签");
        }
        String base64Source = Base64.toBase64String(sourceData.getBytes());
        if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            return GMSSLSM2SignUtils.verifyByBC(key, base64Source, sign);
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())
                || sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            return GMSSLRSASignUtils.verifyByBC(sigAlgName, key, base64Source, sign);
        } else if (sigAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName())) { //add NIST signAlgorithm
            return GMSSLBCSignUtils.verifySignature(sigAlgName, key, GMSSLByteArrayUtils.base64Decode(base64Source), GMSSLByteArrayUtils.base64Decode(sign));
        } else {
            logger.error("暂未未找到 {} 类型验签方式", sigAlgName);
            throw new Exception(String.format("can't get verify sign with %s type", sigAlgName));
        }
    }
}
