package com.xdja.pki.apache.client.utils;

import com.xdja.pki.apache.client.core.ApacheHttpException;
import com.xdja.pki.apache.client.core.ClientKeyStoreConfig;
import com.xdja.pki.apache.client.core.KeyStroeConvertException;
import com.xdja.pki.gmssl.GMSSLContext;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.http.bean.GMSSLProtocol;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.io.File;
import java.io.InputStream;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * apache提供的安全通道工具类
 * @author syg
 *
 */
public class ApacheClientHttpUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(ApacheClientHttpUtils.class);

    public static volatile CloseableHttpClient client = null;

    private static HttpContext localContext = new BasicHttpContext();

    private static HttpClientContext context = HttpClientContext.adapt(localContext);

    private static RequestConfig requestConfig=null;

    /**
     * 发送请求
     * @param reqBody
     * @param paramsMap
     * @param url
     * @param contentType
     * @param protectionAlgName
     * @param isHttps
     * @param requestMethod
     * @param clientKeyStoreConfig
     * @return
     */
    public static CloseableHttpResponse sendApacheClientRequest(byte[] reqBody, Map<String, String> paramsMap,Map<String,String> headerMap, String url, String contentType, String protectionAlgName, boolean isHttps, String requestMethod,ClientKeyStoreConfig clientKeyStoreConfig) throws ApacheHttpException {
        CloseableHttpResponse clientResponse = ApacheClientHttpUtils.exeHttpsRequest(url, reqBody, paramsMap,headerMap,contentType, protectionAlgName,isHttps,requestMethod, false,clientKeyStoreConfig);
        return clientResponse;
    }


    /**
     * 发送请求
     * @param reqBody
     * @param paramsMap
     * @param url
     * @param contentType
     * @param protectionAlgName
     * @param isHttps
     * @param requestMethod
     * @param clientKeyStoreConfig
     * @return
     */
    public static CloseableHttpResponse sendApacheClientRequest(byte[] reqBody, Map<String, String> paramsMap,Map<String,String> headerMap, String url, String contentType, String protectionAlgName, boolean isHttps, String requestMethod,boolean isUseHsm, ClientKeyStoreConfig clientKeyStoreConfig) throws ApacheHttpException {
        CloseableHttpResponse clientResponse = ApacheClientHttpUtils.exeHttpsRequest(url, reqBody, paramsMap,headerMap,contentType, protectionAlgName,isHttps,requestMethod, isUseHsm,clientKeyStoreConfig);
        return clientResponse;
    }

    /**
     * 执行apache发送https请求
     * @param url               目标url
     * @param message           post请求body体数据
     * @param paramsMap         get请求params参数
     * @param contentType       请求消息类型
     * @param signAlgName       签名算法
     * @param isHttps           是否使用https  0-否 1-是
     * @param requestMethod     请求方式  GET 、POST
     * @param clientKeyStoreConfig    keyStore配置类
     * @return
     */
    public static CloseableHttpResponse exeHttpsRequest(String url, byte[] message, Map<String, String> paramsMap,Map<String,String> headerMap, String contentType, String signAlgName, boolean isHttps, String requestMethod,boolean isUseHsm,ClientKeyStoreConfig clientKeyStoreConfig) throws ApacheHttpException {
        // 获取client实例
        ApacheClientHttpUtils.getClient(isHttps,signAlgName, isUseHsm,clientKeyStoreConfig);
        CloseableHttpResponse response = null;
        try {
            // 指定报文头Content-type、User-Agent
            URIBuilder uri = null;
            if (isHttps){
                uri = new URIBuilder("https://" + url);
            }else {
                uri = new URIBuilder("http://" + url);
            }
            logger.info("当前请求的url=========================="+uri.toString());
            HttpUriRequest httpUriRequest = null;
            ByteArrayEntity bodyEntity = null;

            if ("post".equalsIgnoreCase(requestMethod)){
                httpUriRequest = new HttpPost(uri.build());
                bodyEntity = new ByteArrayEntity(message);
                ((HttpPost) httpUriRequest).setEntity(bodyEntity);
            }else {
                // 请求带参数
                if (null != paramsMap && paramsMap.size() != 0) {
                    List<NameValuePair> list = new LinkedList<NameValuePair>();
                    for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
                        BasicNameValuePair param = new BasicNameValuePair(entry.getKey(), entry.getValue());
                        list.add(param);
                    }
                    uri.setParameters(list);
                }
                if ("get".equalsIgnoreCase(requestMethod)) {
                    httpUriRequest = new HttpGet(uri.build());
                    ((HttpGet) httpUriRequest).setConfig(requestConfig);
                } else if ("put".equalsIgnoreCase(requestMethod)) {
                    httpUriRequest = new HttpPut(uri.build());
                    bodyEntity = new ByteArrayEntity(message);
                    ((HttpPut) httpUriRequest).setEntity(bodyEntity);
                    ((HttpPut) httpUriRequest).setConfig(requestConfig);
                } else if ("delete".equalsIgnoreCase(requestMethod)) {
                    httpUriRequest = new HttpDelete(uri.build());
                    ((HttpDelete) httpUriRequest).setConfig(requestConfig);
                }
            }
            if (StringUtils.isNotBlank(contentType)){
                httpUriRequest.setHeader("content-type", contentType);
            }
            httpUriRequest.setHeader("timestamp", String.valueOf(System.currentTimeMillis()));

            //组装header
            if (!CollectionUtils.isEmpty(headerMap)) {
                for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                    httpUriRequest.setHeader(entry.getKey(), entry.getValue());
                }
            }
            // 执行请求操作，并拿到结果（同步阻塞）
            response = client.execute(httpUriRequest);

        } catch (Exception e) {
            logger.error("apache发送client请求异常:",e);
            throw new ApacheHttpException("apache发送client请求异常");
        }
        return response;
    }

    /**
     * 创建SSL上下文
     * @param signAlgName      签名算法
     * @param clientKeyStoreConfig   keystore配置
     * @return
     * @throws Exception
     */
    private static SSLContext createVerifySSL(String signAlgName,boolean isUseHsm, ClientKeyStoreConfig clientKeyStoreConfig) throws ApacheHttpException {
        SSLContext sslcontext = null;
        if (StringUtils.isBlank(signAlgName)) {
            return sslcontext;
        }
        String algName = null;
        if("SM3withSM2".equalsIgnoreCase(signAlgName)) {
            if (isUseHsm) {
                algName = GMSSLProtocol.GMSSLSDFYUNHSMv11.getValue();
            } else {
                algName = GMSSLProtocol.GMSSLv11.getValue();
            }
        }else{
            algName = GMSSLProtocol.TLSV12.getValue();
        }
        KeyStore clientKeyStore = null;
        KeyStore trustKeyStore = null;
        try {
            if (clientKeyStoreConfig==null) {
                if ("SM3withSM2".equalsIgnoreCase(signAlgName)) {
                    sslcontext = GMSSLContext.getClientInstance("GMSSLv1.1").getSslContext();
                } else {
                    sslcontext = GMSSLContext.getClientInstance("TLSv1.2").getSslContext();
                }

                X509TrustManager trustManager = new X509TrustManager() {
                    public void checkClientTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                    }
                    public void checkServerTrusted(X509Certificate[] x509Certificates, String authType) throws CertificateException {
                        if (x509Certificates == null || x509Certificates.length < 1 || authType == null || authType.length() < 1) {
                            throw new IllegalArgumentException();
                        }
                        String subject = x509Certificates[0].getSubjectX500Principal().getName();
                        logger.info("Auto-trusted server certificate chain for: " + subject);
                    }

                    public X509Certificate[] getAcceptedIssuers() {
                        return new X509Certificate[0];
                    }
                };
                sslcontext.init(null, new TrustManager[]{trustManager}, null);
                return sslcontext;
            }

            if (StringUtils.isNotBlank(clientKeyStoreConfig.getTrustKeyStorePath()) && StringUtils.isNotBlank(clientKeyStoreConfig.getTrustKeyStorePwd())) {
                File tf = new File(clientKeyStoreConfig.getTrustKeyStorePath());
                if (tf.exists()) {
                    try {
                        trustKeyStore = ApacheClientHttpUtils.getKeyStore(clientKeyStoreConfig.getTrustKeyStorePath(), clientKeyStoreConfig.getTrustKeyStorePwd().toCharArray());
                    } catch (Exception e) {
                        logger.error("ApacheClientHttpUtils.getKeyStore==========", e);
                        throw new KeyStroeConvertException("keyStore转换失败！");
                    }
                }
                if (StringUtils.isAnyBlank(clientKeyStoreConfig.getClientKeyStorePath(), clientKeyStoreConfig.getClientKeyStorePwd())) {
                    sslcontext = GMSSLContext.getClientInstance(clientKeyStore, null, trustKeyStore, algName).getSslContext();
                    return sslcontext;
                }
                // 双向认证配置
                File sf = new File(clientKeyStoreConfig.getClientKeyStorePath());
                if (sf.exists()) {
                    try {
                        clientKeyStore = ApacheClientHttpUtils.getKeyStore(clientKeyStoreConfig.getClientKeyStorePath(), clientKeyStoreConfig.getClientKeyStorePwd().toCharArray());
                    } catch (Exception e) {
                        logger.error("ApacheClientHttpUtils.getKeyStore==========", e);
                        throw new KeyStroeConvertException("keyStore转换失败！");
                    }
                }
                sslcontext = GMSSLContext.getClientInstance(clientKeyStore, clientKeyStoreConfig.getClientKeyStorePwd().toCharArray(), trustKeyStore, algName).getSslContext();
                return sslcontext;
            }
        } catch (Exception e) {
            logger.error("ApacheClientHttpUtils.createVerifySSL==========",e);
            throw new ApacheHttpException("构造SSLContext异常");
        }
        return null;
    }

    /**
     * 获取client
     * @param isHttps         是否使用https通道  0-否  1-是
     * @param signAlgName     签名算法名称  e.g. SM3WithSM2
     * @param clientKeyStoreConfig  密钥库配置类
     * @return
     */
    public static CloseableHttpClient getClient(boolean isHttps, String signAlgName,boolean isUseHsm, ClientKeyStoreConfig clientKeyStoreConfig) throws ApacheHttpException {
        Registry<ConnectionSocketFactory> socketFactoryRegistry = null;
        if (client == null) {
            if (isHttps) {
                SSLContext sslcontext = null;
                // 采用绕过验证的方式处理https请求
                sslcontext = ApacheClientHttpUtils.createVerifySSL(signAlgName, isUseHsm,clientKeyStoreConfig);

                // 设置协议http和https对应的处理socket链接工厂的对象
                socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                        .register("https", new SSLConnectionSocketFactory(sslcontext,
                                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER)).build();
            } else {
                // 设置协议http和https对应的处理socket链接工厂的对象
                socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                        .register("http", PlainConnectionSocketFactory.INSTANCE).build();
            }
            PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
            connManager.setMaxTotal(500);
            connManager.setDefaultMaxPerRoute(500);

//            // 检查连接的状态，由客户端通过配置去主动关闭其认为是失效的链接
//            IdleConnectionMonitorThread idleConnectionMonitor = new IdleConnectionMonitorThread(connManager);
//            idleConnectionMonitor.start();

            // 指定链接 keep alive策略 告诉httpclient 链接大概什么时候过期 可以关闭
            ConnectionKeepAliveStrategy myStrategy = new ConnectionKeepAliveStrategy() {
                public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                    // Honor 'keep-alive' header
                    HeaderElementIterator it = new BasicHeaderElementIterator(
                            response.headerIterator(HTTP.CONN_KEEP_ALIVE));
                    while (it.hasNext()) {
                        HeaderElement he = it.nextElement();
                        String param = he.getName();
                        String value = he.getValue();
                        if (value != null && param.equalsIgnoreCase("timeout")) {
                            try {
                                return Long.parseLong(value) * 1000;
                            } catch (NumberFormatException ignore) {
                            }
                        }
                    }
                    return 30 * 1000;
                }
            };
            // 创建自定义的httpclient对象
            client = HttpClients.custom().setKeepAliveStrategy(myStrategy).setConnectionManager(connManager).build();
            // ConnectionRequestTimeout 使用连接池时，从连接池获取连接的超时时间
            // ConnectTimeout 建立连接时间，三次握手完成时间
            // SocketTimeout 数据传输过程中数据包之间间隔的最大时间
            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(6000).setSocketTimeout(20000).setConnectTimeout(6000).build();

//            HttpClientContext.create();
        }
        return client;
    }

    /**
     * 获取keysStore对象
     * @param filePath   文件路径
     * @param pw         文件访问密码
     * @return
     * @throws Exception
     */
    private static KeyStore getKeyStore(String filePath, char[] pw) throws Exception {
        KeyStore keyStore;
        try {
            InputStream in = GMSSLX509Utils.readInputStreamFromPath(filePath);
            keyStore = KeyStore.getInstance("JKS");
            keyStore.load(in, pw);
            return keyStore;
        } catch (Exception e) {
            try {
                InputStream in = GMSSLX509Utils.readInputStreamFromPath(filePath);
                keyStore = KeyStore.getInstance("BKS", "BC");
                keyStore.load(in, pw);
                return keyStore;
            } catch (Exception e1) {
                InputStream in = GMSSLX509Utils.readInputStreamFromPath(filePath);
                keyStore = KeyStore.getInstance("pkcs12","BC");
                keyStore.load(in, pw);
                return keyStore;
            }
        }
    }
}
