package com.xdja.pki.ra.openapi.aop;


import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.commonenum.ReqAttributeEnum;
import com.xdja.pki.ra.core.util.params.CheckUtils;
import com.xdja.pki.ra.core.util.SignUtils;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.manager.dao.CustomerDao;
import com.xdja.pki.ra.openapi.bean.SystemFlagBean;
import com.xdja.pki.ra.service.manager.certapply.bean.ErrorMsg;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.util.ContentCachingRequestWrapper;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLDecoder;
import java.security.PublicKey;


/**
 * @author: ggp
 * @Date: 2019/11/2 10:17
 * @Description:
 */
@Component
@Aspect
public class RaSdkInterceptAspect {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CustomerDao customerDao;

    @Value("${ra-openapi.response.overtime}")
    private String OverTime;

    @Pointcut("@annotation(SignIntercep)")
    public void pointcut() {
    }

    @Around("pointcut()")
    public Object signIntercept(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        logger.debug("a-openapi-normal---------------------开始拦截到RA-SDK的请求");
        /**
         * 获取当前请求上下文
         */
        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();

        HttpServletRequest request = servletRequestAttributes.getRequest();
        HttpServletResponse response = servletRequestAttributes.getResponse();


        /**
         * 获取请求body中内容
         */
        ContentCachingRequestWrapper requestWrapper;
        HttpServletRequestWrapper httpRequestWrapper = (HttpServletRequestWrapper) request;
        if (!(httpRequestWrapper instanceof ContentCachingRequestWrapper)) {
            requestWrapper = (ContentCachingRequestWrapper) httpRequestWrapper.getRequest();
        } else {
            requestWrapper = (ContentCachingRequestWrapper) httpRequestWrapper;
        }
        String body = new String(requestWrapper.getContentAsByteArray());
        String systemFlag = request.getHeader(ReqAttributeEnum.SYSTEM_FLAG.name);
        String signValue = request.getHeader(ReqAttributeEnum.SING_VALUE.name);
        String timestamp = request.getHeader(ReqAttributeEnum.TIMESTAMP.name);
        String signAlg = request.getHeader(ReqAttributeEnum.SIGN_ALG.name);
        String signSn  = request.getHeader(ReqAttributeEnum.SIGN_SN.name);
        String url = URLDecoder.decode(request.getRequestURI(), "UTF-8");
        logger.debug("请求地址,url:[{}]", url);
        logger.debug("验签参数，systemFlag:[{}],sign:[{}],timestamp:[{}],signAlg:[{}],signSn:[{}]", systemFlag, signValue, timestamp, signAlg,signSn);
        logger.debug("请求body:[{}]", body);

        if (!CheckUtils.checkParamsNotNull(systemFlag, signValue, timestamp)) {
            logger.error("缺少必要参数");
            response.setStatus(400);
            return getErrorResult(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }

        if (checkOverTime(request.getHeader(ReqAttributeEnum.TIMESTAMP.name))) {
            logger.error("sdk响应超时");
            response.setStatus(400);
            return getErrorResult(ErrorEnum.RA_SDK_REQUEST_TIMEOUT);
        }

        /**
         * 去掉包名（前端请求tomcat会加）
         */
        String source =  url.replaceFirst("/ra-openapi","") + body;
        String certStr = null;
        try {
            certStr = customerDao.getSysCertBySystemFlagAndSn(systemFlag,signSn);
        } catch (Exception e) {
            logger.error("正常状态的该系统证书不存在=========：systemFlag:[{}],signSn:[{}]",systemFlag,signSn);
            logger.error("正常状态的该系统证书不存在", e);
            response.setStatus(400);
            return getErrorResult(ErrorEnum.NORMAL_CERT_IS_NOT_EXIST);
        }
        PublicKey key = CertUtils.getCertFromStr(certStr).getPublicKey();
        if (!SignUtils.verify(signAlg, key, source, signValue)) {
            logger.error("sdk请求验签失败");
            logger.debug("signSn:[{}]",signSn);
            logger.debug("source:[{}]", source);
            logger.debug("sign:[{}]", signValue);
            response.setStatus(400);
            return getErrorResult(ErrorEnum.REQUEST_BODY_AND_SIGN_BODY_DISACCORD);
        }
        SystemFlagBean systemFlagBean = new SystemFlagBean();
        systemFlagBean.setSystemFlag(systemFlag);
        Object[] args = proceedingJoinPoint.getArgs();
        for (int i = 0; i < args.length ; i++) {
            if(null != args[i] && args[i].getClass() == SystemFlagBean.class){
                args[i] = systemFlagBean;
            }
        }
        return proceedingJoinPoint.proceed(args);
    }

    public static void main(String[] args) throws Exception {
        //验签参数，systemFlag:[test2019],sign:[MEYCIQDNef3Y4lvs6/OFOp+DY0/Z2W/mXEf7O3ud6BfVTTiL8gIhAIm6BlEoIwEkGyBgF6ci7M+LGZRZRRtFKV2sz5PpOJKr],timestamp:[1577502068126],signAlg:[1.2.156.10197.1.501],signSn:[10000056]
       String certStr = "-----BEGIN CERTIFICATE-----\n" +
               "MIIDFTCCArqgAwIBAgIEEAAAVzAKBggqgRzPVQGDdTA2MQswCQYDVQQGEwJDTjEXMBUGA1UECgwOQ2FzY29UZXN0U3ViQ2ExDjAMBgNVBAMMBXN1YkNhMB4XDTE5MTIyNzA4NTIwMVoXDTQ5MTIwOTA4NTIwMVowRzELMAkGA1UEBhMCQ04xFzAVBgNVBAoMDkNhc2NvVGVzdFN1YkNhMRAwDgYDVQQDDAdjYXNjb1JhMQ0wCwYDVQQDDAR0ZXN0MFkwEwYHKoZIzj0CAQYIKoEcz1UBgi0DQgAEpQbDwI+iNJTDhy4m+avytsJuWDREOrgT3B3bcU7CGr1MZaNW+O1bEQR8jFvBHx6rMSMEXxG76TvtcrG7CMyzVaOCAaMwggGfMA4GA1UdDwEB/wQEAwIDODCBgAYDVR0uBHkwdzB1oHOgcYZvbGRhcDovLzExLjEyLjEwNy4xMDA6Mzg5L089Q2FzY29UZXN0Q0EsQz1DTi9jbj1jcmwwLG89c3ViQ2FDUkwsTz1DYXNjb1Rlc3RTdWJDYSxDPUNOP2RlbHRhUmV2b2NhdGlvbkxpc3Q7YmluYXJ5MGIGA1UdIwRbMFmAFDm5j5xeORqC" +
               "9a8JXnE9PgGg4T3coTukOTA3MQswCQYDVQQGEwJDTjEUMBIGA1UECgwLQ2FzY29UZXN0Q0ExEjAQBgNVBAMMCWNhc2NvUm9vdIIEEAAADTCBhgYDVR0fBH8wfTB7oHmgd4Z1bGRhcDovLzExLjEyLjEwNy4xMDA6Mzg5L089Q2FzY29UZXN0Q0EsQz1DTi9jbj1jcmwwLG89c3ViQ2FDUkwsTz1DYXNjb1Rlc3RTdWJDYSxDPUNOP2NlcnRpZmljYXRlUmV2b2NhdGlvbkxpc3Q7YmluYXJ5MB0GA1UdDgQWBBSmbACtC8srsgw6myJAJbzWVsPJyDAKBggqgRzPVQGDdQNJADBGAiEA1ZSIxxNZNPwGGYmLuprYgYQEpsSsf9+j7Bo7l2Qs4qsCIQD6TNQjRFiaqVnMONGRL+zO+eG9OWhei9fiDku9Q1yrOg==\n" +
               "-----END CERTIFICATE-----";
        String signAlg = "1.2.156.10197.1.501";
        PublicKey key = CertUtils.getCertFromStr(certStr).getPublicKey();
        String sourceData = "/v1/ra-openapi/user/1{\"personUser\":{\"licenseNumber\":\"TldTest\",\"licenseType\":1,\"personName\":\"jiekouTest\",\"sex\":1,\"status\":0}}";
        String signValue = "MEYCIQDNef3Y4lvs6/OFOp+DY0/Z2W/mXEf7O3ud6BfVTTiL8gIhAIm6BlEoIwEkGyBgF6ci7M+LGZRZRRtFKV2sz5PpOJKr";
        boolean  verity = SignUtils.verify(signAlg, key, sourceData, signValue);
        System.out.println(verity);
    }
    /**
     * 判断是否超时
     *
     * @param timeStamp
     * @return
     */
    private boolean checkOverTime(String timeStamp) {
        long reqTime = Long.parseLong(timeStamp);
        long currentTime = System.currentTimeMillis();
        long overTime = Integer.parseInt(OverTime) * 60L * 1000;

        if (currentTime - reqTime > overTime) {
            return true;
        }
        return false;
    }

    /**
     * 获取body字符串
     *
     * @param reader
     * @return
     */
    private String read(Reader reader) {
        try {
            StringWriter writer = new StringWriter();
            char[] buffer = new char[4096];
            int n;
            while (-1 != (n = reader.read(buffer))) {
                writer.write(buffer, 0, n);
            }
            return writer.toString();
        } catch (IOException e) {
            throw new IllegalStateException("获取body字符串失败", e);
        }
    }
    private Object getErrorResult(ErrorEnum errorEnum){
        ErrorMsg errorMsg = new ErrorMsg();
        errorMsg.setErrorCode(errorEnum.code);
        errorMsg.setErrorMsg(errorEnum.desc);
        return errorMsg;
    }
}
