package com.xdja.pki.ra.openapi.normal;

import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.openapi.aop.SignIntercep;
import com.xdja.pki.ra.openapi.bean.SystemFlagBean;
import com.xdja.pki.ra.openapi.normal.Utils.CheckParam;
import com.xdja.pki.ra.service.manager.certapply.CertApplyManagerService;
import com.xdja.pki.ra.service.manager.certapply.bean.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;


/**
 * @author: ggp
 * @Date: 2019/10/22 18:06
 * @Description:
 */
@RestController
public class CertApplyManagerController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CertApplyManagerService certApplyManagerService;

    /**
     * 获取RA系统的BaseDns
     *
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/baseDn", method = RequestMethod.GET)
    public Object getRaBaseDn(HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用获取RA系统BaseDn");

        Result result = certApplyManagerService.getRABaseDn();
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 获取第三方系统拥有的模板信息
     *
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/template/{systemName}", method = RequestMethod.GET)
    public Object getOtherSystemTemplates(@PathVariable String systemName, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用获取第三方模板列表");
        Result result = CheckParam.checkAuthority(systemName, systemFlagBean.getSystemFlag());
        if (!result.isSuccess()) {
            return CheckParam.resultDeal(Result.success(null), resp);
        }
        result = certApplyManagerService.getTemplateInfoList(systemName);
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 签发申请-发起
     *
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/issue", method = RequestMethod.POST)
    public Object issueApplyLaunch(@RequestBody IssueApply issueApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用签发申请-发起");
        if(logger.isDebugEnabled()) {
            logger.debug("请求参数issueApply:[{}]", issueApply);
        }

        Result result = CheckParam.checkParam(issueApply);
        if (!result.isSuccess()) {
            logger.error("参数校验失败:{}", JsonUtils.object2Json(result));
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.issueApplyLaunch(issueApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 更新申请-发起
     *
     * @param updateApply
     * @param resp
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/update", method = RequestMethod.POST)
    public Object updateApplyLaunch(@RequestBody UpdateApply updateApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用更新申请-发起");
        logger.debug("请求参数updateApply:[{}]", updateApply);

        Result result = CheckParam.checkParam(updateApply);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.updateApplyLaunch(updateApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 撤销申请-发起
     *
     * @param revokeApply
     * @param resp
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/revoke", method = RequestMethod.POST)
    public Object revokeApplyLaunch(@RequestBody RevokeApply revokeApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用撤销申请-发起");
        logger.debug("请求参数revokeApply:[{}]", revokeApply);

        Result result = CheckParam.checkParam(revokeApply);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.revokeApplyLaunch(revokeApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 恢复申请-发起
     *
     * @param revokeApply 发起恢复申请时使用的类对象
     * @param resp        HttpServletResponse
     * @return ApplyRep类型的签发申请返回对象
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/recovery", method = RequestMethod.POST)
    public Object recoveryApplyLaunch(@RequestBody RecoveryApply revokeApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用恢复申请-发起revokeApply:[{}]", revokeApply);
        Result result = CheckParam.checkParam(revokeApply);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.recoveryApplyLaunch(revokeApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 冻结/解冻申请-发起
     *
     * @param freezeApply    发起冻结/解冻申请时使用的类对象
     * @param systemFlagBean 系统标识
     * @param resp           HttpServletResponse
     * @return ApplyRep类型的签发申请返回对象
     */
    @SignIntercep
    @PostMapping(value = "/v1/ra-openapi/apply/freeze")
    public Object freezeApplyLaunch(@RequestBody FreezeApply freezeApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用冻结/解冻申请-发起 freezeApply:[{}]", freezeApply);
        Result result = CheckParam.checkParam(freezeApply);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.freezeApplyLaunch(freezeApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 签发/更新/撤销申请-审核
     *
     * @param applyType  申请类型	1-签发申请 2-更新申请 3-撤销申请 4-恢复申请
     * @param applyNo    申请编号	发起申请时，返回的申请编号
     * @param checkApply 审核申请对象	审核各种申请审核类对象
     * @param resp       HttpServletResponse
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/check/{applyType}/{applyNo}", method = RequestMethod.PUT)
    public Object applyCheck(@PathVariable Integer applyType, @PathVariable String applyNo, @RequestBody CheckApply checkApply, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用申请审核");
        logger.debug("请求参数applyType:[{}]", applyType);
        logger.debug("请求参数applyNo:[{}]", applyNo);
        logger.debug("请求参数checkApply:[{}]", checkApply);

        Result result = CheckParam.checkParam(applyType, checkApply);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.applyCheck(applyType, applyNo, checkApply, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 申请制证执行
     *
     * @param applyType
     * @param applyNo
     * @param userCertReq
     * @param resp
     * @return
     */
    @RequestMapping(value = "/v1/ra-openapi/apply/{applyType}/{applyNo}", method = RequestMethod.POST)
    public Object applyCertCarry(@PathVariable Integer applyType, @PathVariable String applyNo, @RequestBody UserCertReq userCertReq, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用申请制证执行");
        logger.debug("请求参数applyType:[{}]", applyType);
        logger.debug("请求参数applyNo:[{}]", applyNo);
        logger.debug("请求参数userCertReq:[{}]", userCertReq);
        /**
         * 走cmp协议
         */
        return null;
    }

    /**
     * 证书签发确认消息
     *
     * @param applyNo
     * @return
     */
    @RequestMapping(value = "/v1/ra-openapi/apply/confirm/{applyNo}", method = RequestMethod.POST)
    public Object certIssueConfirmMsg(@PathVariable String applyNo, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用证书签发确认消息");
        logger.debug("请求参数applyNo:[{}]", applyNo);

        Result result = certApplyManagerService.certIssueConfirmMsg(applyNo);
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 证书签发错误消息
     *
     * @param applyNo
     * @return
     */
    @RequestMapping(value = "/v1/ra-openapi/apply/error/{applyNo}", method = RequestMethod.POST)
    public Object certIssueErrorMsg(@PathVariable String applyNo, @Valid @RequestBody ErrorMsg errorMsg, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用证书签发确认消息");
        logger.debug("请求参数applyNo:[{}]", applyNo);

        Result result = certApplyManagerService.certApplyErrorMsg(applyNo, errorMsg);
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 查询证书申请信息
     *
     * @param applyType 申请类型	1-签发申请 2-更新申请 3-撤销申请 4-恢复申请
     * @param applyNo   申请编号	发起申请时，返回的申请编号
     * @return CertApplyInfo类型的证书申请信息 根据申请类型的不同 返回该结构体中不同的类属性内容
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/{applyType}/{applyNo}", method = RequestMethod.GET)
    public Object getCertApplyInfo(@PathVariable Integer applyType, @PathVariable String applyNo, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用查询证书申请信息");
        logger.debug("请求参数applyType:[{}]", applyType);
        logger.debug("请求参数applyNo:[{}]", applyNo);
        Result result = CheckParam.checkApplyType(applyType);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.certApplyQuery(applyType, applyNo, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 根据申请编号查询申请状态
     *
     * @param applyNo
     * @param systemFlagBean
     * @param resp
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/{applyNo}", method = RequestMethod.GET)
    public Object certApplyQuery(@PathVariable String applyNo, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用查询证书申请信息");
        logger.debug("请求参数applyNo:[{}]", applyNo);

        Result result = certApplyManagerService.certApplyQuery(applyNo, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

    /**
     * 修改证书申请信息
     *
     * @param applyType
     * @param applyNo
     * @param editCertApplyInfo
     * @return
     */
    @SignIntercep
    @RequestMapping(value = "/v1/ra-openapi/apply/{applyType}/{applyNo}", method = RequestMethod.PUT)
    public Object updateCertApplyInfo(@PathVariable Integer applyType, @PathVariable String applyNo, @RequestBody EditCertApplyInfo editCertApplyInfo, SystemFlagBean systemFlagBean, HttpServletResponse resp) {
        logger.info("ra-openapi-normal---------------------开始调用修改证书申请信息");
        logger.debug("请求参数applyType:[{}]", applyType);
        logger.debug("请求参数applyNo:[{}]", applyNo);
        logger.debug("请求参数editCertApplyInfo:[{}]", editCertApplyInfo);

        Result result = CheckParam.checkParam(applyType, editCertApplyInfo);
        if (!result.isSuccess()) {
            return CheckParam.resultSort(result, resp);
        }
        result = certApplyManagerService.certApplyUpdate(applyType, applyNo, editCertApplyInfo, systemFlagBean.getSystemFlag());
        return CheckParam.resultDeal(result, resp);
    }

}
