package com.xdja.pki.ra.openapi.normal.Utils;

import com.xdja.pki.ra.core.common.ErrorBean;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.*;
import com.xdja.pki.ra.core.util.params.CheckUtils;
import com.xdja.pki.ra.core.util.params.PatternCheckUtils;
import com.xdja.pki.ra.service.manager.baseuser.bean.DeviceUser;
import com.xdja.pki.ra.service.manager.baseuser.bean.OrganUser;
import com.xdja.pki.ra.service.manager.baseuser.bean.PersonUser;
import com.xdja.pki.ra.service.manager.baseuser.bean.UserInfo;
import com.xdja.pki.ra.service.manager.certapply.bean.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletResponse;

/**
 * @author: ggp
 * @Date: 2019/12/3 14:21
 * @Description:
 */
public class CheckParam {
    private static final Logger logger = LoggerFactory.getLogger(CheckParam.class);


    /*****************************************************************************************************
     * 用户参数校验 开始
     */


    /**
     * 检查参数是否合法
     *
     * @param userType
     * @param userInfo
     * @return
     */
    public static Result requestParamCheck(Integer userType, UserInfo userInfo, boolean isUpdate) {
        Result result = new Result();
        if (userType == UserTypeEnum.PERSON_USER.id) {
            PersonUser personUser = userInfo.getPersonUser();
            if (null == personUser) {
                logger.error("用户类型和传入参数不匹配");
                return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_MATCH_ARGS);
            }
            if (!isUpdate) {
                if (!CheckUtils.checkParamsNotNull(personUser)) {
                    logger.error("个人用户缺少必要参数");
                    return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
                }
            }
            /**
             * 校验参数格式
             */
            result = checkPerson(personUser);
            if (!result.isSuccess()) {
                return result;
            }
        } else if (userType == UserTypeEnum.ORGAN_USER.id) {
            OrganUser organUser = userInfo.getOrganUser();
            if (null == organUser) {
                logger.error("用户类型和传入参数不匹配");
                return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_MATCH_ARGS);
            }
            if (!isUpdate) {
                if (!CheckUtils.checkParamsNotNull(organUser)) {
                    logger.error("机构用户缺少必要参数");
                    return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
                }
            }
            /**
             * 校验参数格式
             */
            result = checkOrganUser(organUser);
            if (!result.isSuccess()) {
                return result;
            }
        } else if (userType == UserTypeEnum.DEVICE_USER.id) {
            DeviceUser deviceUser = userInfo.getDeviceUser();
            if (null == deviceUser) {
                logger.error("用户类型和传入参数不匹配");
                return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_MATCH_ARGS);
            }
            if (!isUpdate) {
                if (!CheckUtils.checkParamsNotNull(deviceUser)) {
                    logger.error("设备用户缺少必要参数");
                    return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
                }
            }
            /**
             * 校验参数格式
             */
            result = checkDeviceUser(deviceUser);
            if (!result.isSuccess()) {
                return result;
            }
        } else {
            logger.error("不支持的用户类型:[{}]", userType);
            return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_SUPPORT);
        }
        return result;
    }


    /**
     * @param personUser
     * @return
     */
    public static Result checkParameter(PersonUser personUser) {
        return checkParameter(personUser.getEmail(), personUser.getPostalCode(), personUser.getSex(), personUser.getTelNumber(), personUser.getLicenseType(), personUser.getLicenseNumber());
    }

    /**
     * 校验参数
     *
     * @param email
     * @param postalCode
     * @param sex
     * @param telNumber
     * @return
     */
    public static Result checkParameter(String email, String postalCode, Integer sex, String telNumber, Integer licenseType, String licenseNumber) {
        if (null != email) {
            if (!PatternCheckUtils.checkEmail(email)) {
                return Result.failure(ErrorEnum.EMAIL_FORMAT_IS_ERROR);
            }
        }
        if (null != postalCode) {
            if (!PatternCheckUtils.checkPostalCode(postalCode)) {
                return Result.failure(ErrorEnum.POSTCODE_FORMAT_IS_ERROR);
            }
        }
        if (null != sex) {
            if (!PatternCheckUtils.checkSex(sex)) {
                return Result.failure(ErrorEnum.SEX_FORMAT_IS_ERROR);
            }
        }
        if (null != telNumber) {
            if (!PatternCheckUtils.checkTelNumber(telNumber)) {
                return Result.failure(ErrorEnum.TEL_NUMBER_IS_ERROR);
            }
        }
        if (null != licenseType) {
            if (!PersonUserTypeEnum.contain(licenseType)) {
                logger.error("不支持的个人用户类型:[{}]", licenseType);
                return Result.failure(ErrorEnum.PERSON_USER_TYPE_IS_NOT_SUPPORT);
            }
        }
        if (null != licenseType && null != licenseNumber) {
            Result result = checkPersonLicenseNumber(licenseType, licenseNumber);
            if (!result.isSuccess()) {
                return result;
            }
        }
        return Result.success(null);
    }

    /**
     * 校验个人用户参数
     *
     * @param personUser
     * @return
     */
    public static Result checkPerson(PersonUser personUser) {
        Result result = checkParameter(personUser.getEmail(), personUser.getPostalCode(), personUser.getSex(), personUser.getTelNumber(), personUser.getLicenseType(), personUser.getLicenseNumber());
        if (!result.isSuccess()) {
            return result;
        }
        return result;
    }

    /**
     * 校验个人用户证件号
     *
     * @param licenseType
     * @param licenseNumber
     * @return
     */
    public static Result checkPersonLicenseNumber(Integer licenseType, String licenseNumber) {
        if (licenseType == PersonUserTypeEnum.IDENTITY_CARD.id) {
            if (!PatternCheckUtils.checkIdCard(licenseNumber)) {
                return Result.failure(ErrorEnum.ID_CARD_FORMAT_IS_ERROR);
            }
        } else if (licenseType == PersonUserTypeEnum.OFFICIAL_CARD.id) {
            if (!PatternCheckUtils.checkMilitaryOfficer(licenseNumber)) {
                return Result.failure(ErrorEnum.MILITARY_OFFICER_FORMAT_IS_ERROR);
            }
        } else if (licenseType == PersonUserTypeEnum.PASSPORT.id) {
            if (!PatternCheckUtils.checkPassPort(licenseNumber)) {
                return Result.failure(ErrorEnum.PASSPORT_FORMAT_IS_ERROR);
            }
        } else if (licenseType == PersonUserTypeEnum.OTHER.id) {
            return Result.success(null);
        } else {
            logger.error("不支持的个人用户类型:[{}]", licenseType);
            return Result.failure(ErrorEnum.PERSON_USER_TYPE_IS_NOT_SUPPORT);
        }
        return Result.success(null);
    }

    /**
     * 校验机构用户参数
     */
    public static Result checkOrganUser(OrganUser organUser) {
        /**
         * 校验参数格式
         */
        Result result = checkParameter(organUser.getEmail(), organUser.getPostalCode(), null, null, null, null);
        if (!result.isSuccess()) {
            return result;
        }

        if (null != organUser.getLicenseType()) {
            result = checkOrganType(organUser.getLicenseType());
            if (!result.isSuccess()) {
                return result;
            }
        }
        if (null != organUser.getLicenseType() && null != organUser.getLicenseNumber()) {
            result = checkOrganUserLicenseNumber(organUser.getLicenseType(), organUser.getLicenseNumber());
            if (!result.isSuccess()) {
                return result;
            }
        }
        if (null != organUser.getUserInfo()) {
            return checkParameter(organUser.getUserInfo());

        }
        return Result.success(null);
    }

    /**
     * 校验机构用户证件号
     *
     * @param licenseType
     * @param licenseNumber
     * @return
     */
    public static Result checkOrganUserLicenseNumber(Integer licenseType, String licenseNumber) {
        Result result = checkOrganType(licenseType);
        if (!result.isSuccess()) {
            return result;
        }
        if (!PatternCheckUtils.checkEnterprise(licenseNumber)) {
            logger.error("机构用户证件号不能超过30个字符");
            return Result.failure(ErrorEnum.ORGAN_LICENSE_NUMBER_IS_ERROR);
        }
        return Result.success(null);
    }


    /**
     * 校验设备用户参数
     *
     * @param deviceUser
     * @return
     */
    public static Result checkDeviceUser(DeviceUser deviceUser) {

        Result result = null;
        if (null != deviceUser.getLicenseType()) {
            result = checkDeviceType(deviceUser.getLicenseType());
            if (!result.isSuccess()) {
                return result;
            }
        }
        if (null != deviceUser.getLicenseType() && null != deviceUser.getLicenseNumber()) {
            result = checkDeviceUserLicenseNumber(deviceUser.getLicenseType(), deviceUser.getLicenseNumber());
            if (!result.isSuccess()) {
                return result;
            }
        }
        if (null != deviceUser.getUserInfo()) {
            return checkParameter(deviceUser.getUserInfo());
        }
        return result;
    }

    /**
     * 校验设备唯一标识符
     *
     * @param licenseType
     * @param licenseNumber
     * @return
     */
    public static Result checkDeviceUserLicenseNumber(Integer licenseType, String licenseNumber) {
        Result result = checkDeviceType(licenseType);
        if (!result.isSuccess()) {
            return result;
        }
        if (!PatternCheckUtils.checkDeviceNumber(licenseNumber)) {
            logger.error("设备唯一标识符不能超过60个字符");
            return Result.failure(ErrorEnum.DEVICE_USER_LICENSE_NUMBER_LENGTH_IS_TOO_LONG);
        }
        return result;
    }

    /**
     * 校验
     *
     * @param userType
     * @param licenseType
     * @param licenseNumber
     * @param status
     * @return
     */
    public static Result CheckParameter(Integer userType, Integer licenseType, String licenseNumber, Integer status) {
        /**
         * 校验停用参数
         */
        if (!UserStatusEnum.contain(status)) {
            logger.error("不支持的用户状态类型:[{}]", status);
            return Result.failure(ErrorEnum.USER_STATUS_TYPE_IS_NOT_SUPPORT);
        }
        return checkParameter(userType, licenseType, licenseNumber);
    }

    /**
     * 校验参数
     *
     * @param userType
     * @param licenseType
     * @return
     */
    public static Result checkParameter(Integer userType, Integer licenseType, String licenseNumber) {
        Result result = checkUserType(userType);
        if (!result.isSuccess()) {
            return result;
        }
        if (userType == UserTypeEnum.PERSON_USER.id) {
            result = checkPersonLicenseNumber(licenseType, licenseNumber);
            if (!result.isSuccess()) {
                return result;
            }
        } else if (userType == UserTypeEnum.ORGAN_USER.id) {
            result = checkOrganUserLicenseNumber(licenseType, licenseNumber);
            if (!result.isSuccess()) {
                return result;
            }
        } else if (userType == UserTypeEnum.DEVICE_USER.id) {
            result = checkDeviceUserLicenseNumber(licenseType, licenseNumber);
            if (!result.isSuccess()) {
                return result;
            }
        } else {
            return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_SUPPORT);
        }
        return result;
    }

    /**
     * 检查用户类型
     *
     * @param userType
     * @return
     */
    public static Result checkUserType(Integer userType) {
        Result result = new Result();
        if (!UserTypeEnum.contain(userType)) {
            logger.error("不支持的用户类型:[{}]", userType);
            return Result.failure(ErrorEnum.USER_TYPE_IS_NOT_SUPPORT);
        }

        return result;
    }

    /**
     * 检查个人用户类型
     *
     * @param identType
     * @return
     */
    public static Result checkPersonType(Integer identType) {
        Result result = new Result();
        if (!PersonUserTypeEnum.contain(identType)) {
            logger.error("不支持的个人用户类型:[{}]", identType);
            return Result.failure(ErrorEnum.PERSON_USER_TYPE_IS_NOT_SUPPORT);
        }
        return result;
    }

    /**
     * 检查机构用户类型
     *
     * @param identType
     * @return
     */
    public static Result checkOrganType(Integer identType) {
        if (!OrganTypeEnum.contain(identType)) {
            logger.error("不支持的用户机构类型:[{}]", identType);
            return Result.failure(ErrorEnum.ORGAN_USER_TYPE_IS_NOT_SUPPORT);
        }
        return Result.success(null);
    }

    /**
     * 检查设备用户类型
     *
     * @param identType
     * @return
     */
    public static Result checkDeviceType(Integer identType) {
        if (!DeviceTypeEnum.contain(identType)) {
            logger.error("不支持的设备用户类型:[{}]", identType);
            return Result.failure(ErrorEnum.DEVICE_USER_TYPE_IS_NOT_SUPPORT);
        }
        return Result.success(null);
    }


    /*****************************************************************************************************
     * 用户参数校验 结束
     */

    /*****************************************************************************************************
     * 证书申请参数校验 开始
     */

    /**
     * 校验获取模板列表是否跨系统
     *
     * @param systemName
     * @param systemFlag
     * @return
     */
    public static Result checkAuthority(String systemName, String systemFlag) {
        if (!systemFlag.equals(systemName)) {
            logger.error("不支持跨系统访问:[{}],[{}]", systemName, systemFlag);
            return Result.failure(ErrorEnum.THE_CROSS_SYSTEM_ACCESS_IS_NOT_SUPPORT);
        }
        return Result.success(null);
    }

    /**
     * 校验签发申请参数
     *
     * @param issueApply
     * @return
     */
    public static Result checkParam(IssueApply issueApply) {
        logger.info("checkParam---------------------校验签发申请参数");
        if (!CheckUtils.checkParamsNotNull(issueApply)) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        if (null != issueApply.getPrivateKeyLength()) {
            if (!KeyLengthEnum.contain(issueApply.getPrivateKeyLength())) {
                return Result.failure(ErrorEnum.KEY_LENGTH_IS_NOT_SUPPORT);
            }
        }
        if (null != issueApply.getSignAlg()) {
            if (!SignAlgEnum.contain(issueApply.getSignAlg())) {
                return Result.failure(ErrorEnum.SIGN_ALG_IS_NOT_SUPPORT);
            }
        }
        return checkParameter(issueApply.getUserType(), issueApply.getLicenseType(), issueApply.getLicenseNumber());
    }

    /**
     * 校验更新申请参数
     *
     * @param updateApply
     * @return
     */
    public static Result checkParam(UpdateApply updateApply) {
        if (!CheckUtils.checkParamsNotNull(updateApply)) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        return Result.success(null);
    }

    /**
     * 校验撤销申请参数
     *
     * @param revokeApply
     * @return
     */
    public static Result checkParam(RevokeApply revokeApply) {
        if (!CheckUtils.checkParamsNotNull(revokeApply)) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        if (!RevokeReasonEnum.contain(revokeApply.getRevokeReason())) {
            return Result.failure(ErrorEnum.REVOKE_REASON_IS_NOT_SUPPORT);
        }
        return Result.success(null);
    }

    /**
     * 校验更新申请参数
     *
     * @param recoveryApply RecoveryApply
     * @return CoreResult
     */
    public static Result checkParam(RecoveryApply recoveryApply) {
        if (!CheckUtils.checkParamsNotNull(recoveryApply)) {
            logger.warn("RecoveryApply missing req param [signSn]");
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        return Result.success(null);
    }

    /**
     * 校验冻结解冻申请参数
     *
     * @param freezeApply FreezeApply
     * @return CoreResult
     */
    public static Result checkParam(FreezeApply freezeApply) {
        if (!CheckUtils.checkParamsNotNull(freezeApply)) {
            logger.warn("FreezeApply missing req param [signSn&isFreeze]");
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        return Result.success(null);
    }

    /**
     * 校验审核申请参数 、 校验制证参数
     *
     * @param applyType
     * @param object
     * @return
     */
    public static Result checkParam(Integer applyType, Object object) {
        if (!CheckUtils.checkParamsNotNull(object)) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        return checkApplyType(applyType);
    }

    /**
     * 校验申请类型
     *
     * @param applyType
     * @return
     */
    public static Result checkApplyType(Integer applyType) {
        if (!ApplyTypeEnum.contain(applyType)) {
            logger.error("不存在的申请类型");
            return Result.failure(ErrorEnum.THE_APPLY_TYPE_IS_NOT_EXIT);
        }
        return Result.success(null);
    }

    /**
     * 校验申请更新参数
     *
     * @param editCertApplyInfo
     * @return
     */
    public static Result checkParam(Integer applyType, EditCertApplyInfo editCertApplyInfo) {
       // DO 80888 【第三方接口-修改申请】传入申请类型为5时，返回缺少必要请求参数，应返回非法请求参数，不支持5 WLY
      /*  CoreResult result = checkApplyType(applyType);
        if (!result.isSuccess()) {
            return result;
        }*/
        if (applyType == ApplyTypeEnum.ISSUE_APPLY.id) {
            EditIssueApplyInfo editIssueApplyInfo = editCertApplyInfo.getEditIssueApplyInfo();
            if (null == editIssueApplyInfo || !CheckUtils.checkParamsNotNull(editIssueApplyInfo)) {
                return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            }
        } else if (applyType == ApplyTypeEnum.UPDATE_APPLY.id) {
            EditUpdateApplyInfo editUpdateApplyInfo = editCertApplyInfo.getEditUpdateApplyInfo();
            if (null == editUpdateApplyInfo || !CheckUtils.checkParamsNotNull(editUpdateApplyInfo)) {
                return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            }
        } else  if (applyType == ApplyTypeEnum.REVOKE_APPLY.id){
            EditRevokeApplyInfo editRevokeApplyInfo = editCertApplyInfo.getEditRevokeApplyInfo();
            if (null == editRevokeApplyInfo || !CheckUtils.checkParamsNotNull(editRevokeApplyInfo)) {
                return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
            }
        }else {
            logger.error("不存在的申请类型");
            return Result.failure(ErrorEnum.THE_APPLY_TYPE_IS_NOT_EXIT);
        }
        return Result.success(null);
    }

    private static boolean checkEditApplyParam(boolean b, boolean b2) {
        if (b) {
            return true;
        }
        if (!b2) {
            return true;
        }
        return false;
    }

    /*****************************************************************************************************
     * 证书申请参数校验 结束
     */
    public static Object resultDeal(Result result, HttpServletResponse resp) {
        logger.debug("处理结果：[{}]", result);
        if (result.isSuccess()) {
            return result.getInfo();
        } else {
            ErrorMsg errorMsg = new ErrorMsg();
            if (null != result.getError()) {
                /**
                 * 由于部分错误码无枚举，result.getError()为空，放到了ErrorBean，添加此判断。
                 */
                errorMsg.setErrorCode(result.getError().code);
                errorMsg.setErrorMsg(result.getError().desc);
            } else {
                ErrorBean errorBean = result.getErrorBean();
                errorMsg.setErrorCode(errorBean.getErrCode());
                errorMsg.setErrorMsg(errorBean.getErrMsg());
            }
            resp.setStatus(400);
            return errorMsg;

        }
    }

    /**
     * 参数异常分类 true 缺少必要参数 false 请求参数非法
     *
     * @param result
     * @return
     */
    public static Object resultSort(Result result, HttpServletResponse resp) {
        ErrorEnum errorEnum = result.getError();
        if (errorEnum.code != ErrorEnum.MISSING_REQUIRED_PARAMETERS.code) {
            return resultDeal(Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER), resp);
        } else {
            return resultDeal(Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS), resp);
        }
    }

}
