package com.xdja.pki.ra.openapi.normal.handler;

import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.openapi.core.BaseCMPInfo;
import com.xdja.pki.ra.openapi.core.common.FreeText;
import com.xdja.pki.ra.openapi.core.common.PKIMessageException;
import com.xdja.pki.ra.openapi.core.handler.ICmpMessageHandler;
import com.xdja.pki.ra.openapi.core.helper.PKIMessageHelper;
import com.xdja.pki.ra.service.manager.cache.RedisCacheManagerService;
import com.xdja.pki.ra.service.manager.certapply.CertApplyService;
import com.xdja.pki.ra.service.manager.customer.CustomerSysService;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.GeneralName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.security.PublicKey;

/**
 * cmp-normal错误确认消息处理类
 *
 * @author wly
 */
@Component("cmpNormalErrorMsgHandler")
public class CmpNormalErrorMsgHandler implements ICmpMessageHandler {

    protected Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    CertApplyService certApplyService;
    @Autowired
    CustomerSysService customerSysService;
    @Autowired
    RedisCacheManagerService redisCacheManagerService;

    @Override
    public Result handleMessage(PKIMessage pkiMessage, boolean authenticated) throws PKIMessageException {

        logger.info("RA证书错误消息 ========== 【开始】");
        Result result = new Result();
        logger.info("RA证书错误消息 ========== 1. 获取PkiMessage消息结构");
        final PKIMessage errorMessage = PKIMessage.getInstance(pkiMessage);
        if (errorMessage == null) {
            String errMsg = "RA证书错误消息 ========== No pkiMessage response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }
        logger.info("RA证书错误消息 ========== 2. 获取PkiMessage消息头PKIHeader");
        final PKIHeader header = errorMessage.getHeader();
        if (header == null) {
            String errMsg = "RA证书错误消息 ========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }
        GeneralName sender = header.getSender();
        GeneralName recipient = header.getRecipient();
        byte[] normalRecipNonce = new byte[0];
        byte[] normalSenderNonce = new byte[0];
        String normalTranID = null;
        AlgorithmIdentifier protectionAlg = null;
        try {
            normalRecipNonce = header.getRecipNonce() == null ? null : header.getRecipNonce().getOctets();
            normalSenderNonce = header.getSenderNonce() == null ? null : header.getSenderNonce().getOctets();
            normalTranID = header.getTransactionID() == null ? null : new String(header.getTransactionID().getOctets());
            protectionAlg = header.getProtectionAlg();
        } catch (Exception e) {
            String errMsg = "RA证书错误消息 ========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg, e);
        }

        // 对必填项参数的校验
        if (normalRecipNonce == null || normalSenderNonce == null || protectionAlg == null || StringUtils.isBlank(normalTranID)) {
            String errMsg = "RA证书错误消息 ========== 错误消息接口中必填项有空值";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }

        // RA缓存设备对应信息
        String baseCMPInfoStr = redisCacheManagerService.removeRaSdkCmpInfo(normalTranID);
        BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(baseCMPInfoStr, BaseCMPInfo.class);
        if (baseCMPInfo == null) {
            logger.info("RA证书错误消息 ========== 不存在对应的事务ID tranId:{}",normalTranID);
            result.setError(ErrorEnum.CMP_TRAN_ID_IS_NOT_EXIST);
            return result;
        }
        FreeText freeText = null;
        PKIFreeText pkiFreeText = header.getFreeText();
        if (pkiFreeText != null) {
            String freeTestStr = pkiFreeText.getStringAt(0).getString();
            freeText = JsonUtils.json2Object(freeTestStr, FreeText.class);
            if (freeText == null) {
                logger.info("RA证书错误消息 ========== PKI消息体中不包含sn信息");
                RevRepContent revRepContent = PKIMessageHelper.genFailRevRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
                PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_ERROR, normalRecipNonce, normalSenderNonce, normalTranID, revRepContent);
                result.setInfo(errPkiMessage);
                return result;
            }
        }else {
            logger.info("RA证书错误消息 ========== PKI消息体中不包含sn信息");
            RevRepContent revRepContent = PKIMessageHelper.genFailRevRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_ERROR, normalRecipNonce, normalSenderNonce, normalTranID, revRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        String signSn = freeText.getSignSn();
        String normalSender = sender.getName().toString();
        // 通过第三方唯一标识 确认第三方系统的证书和公钥
        Result certResult = customerSysService.getSysCertBySysNumber(normalSender, signSn);
        PublicKey publicKey;
        if (certResult.isSuccess()) {
            publicKey = (PublicKey) certResult.getInfo();
        } else {
            logger.info("RA签发处理 ========== 通过第三方系统标识确认第三方的证书错误 原因：{}", JsonUtils.object2Json(certResult));
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(-1, certResult.getError().code, certResult.getError().desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_ERROR, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        // 验证cmp消息的正确性
        logger.info("RA证书错误消息 ========== 3. 验证cmp消息的header和签名的正确性");
        // 验证cmp消息的header和消息保护
        DERBitString protectData = errorMessage.getProtection();
        byte[] sourceData = PKIMessageHelper.getProtectedBytes(pkiMessage);
        // 验证cmp消息的正确性
        Result checkResult = PKIMessageHelper.checkCmpHeaderAndSign(publicKey, header, protectData.getBytes(), sourceData, protectionAlg, null);
        if (!checkResult.isSuccess()) {
            logger.info("RA证书错误消息 ========== 3.1 验证cmp消息的header和签名错误 原因：{}", JsonUtils.object2Json(checkResult));
            result.setError(checkResult.getError());
            return result;
        }

        PKIBody body = errorMessage.getBody();
        if (body == null) {
            logger.info("RA证书错误消息 ========== 没有对应的PKI消息体");
            result.setError(ErrorEnum.NO_PKI_BODY_FOR_RECEIVED);
            return result;
        }

        if (body.getType() != PKIBody.TYPE_ERROR) {
            logger.info("RA证书错误消息 ========== PKI消息体的类型不是23");
            result.setError(ErrorEnum.ERROR_MSG_PKI_BODY_TAG_NOT_23);
            return result;
        }

        String applyNo = redisCacheManagerService.removeNomalTransId(normalTranID);

        // RA生成-发送证书错误消息
        ErrorMsgContent errorMsgContent = (ErrorMsgContent) body.getContent();
        try {
            int errorCode = errorMsgContent.getErrorCode()==null?111111: errorMsgContent.getErrorCode().getValue().intValue();
            String errorMsg = errorMsgContent.getErrorDetails()==null?"RA自定义normal返回的错误消息":errorMsgContent.getErrorDetails().getStringAt(0).getString();
            logger.info("RA证书错误消息 ========== 错误码:[{}] 错误消息:[{}]", errorCode, errorMsg);
            //生成错误CMP消息
            Result errResult = certApplyService.genErrorMsgContent(applyNo, errorCode, errorMsg, true);
            if (!errResult.isSuccess()) {
                logger.info("RA证书错误消息 ==========  证书确认消息请求错误{}", JsonUtils.object2Json(errResult));
                result.setErrorBean(errResult.getErrorBean());
                return result;
            }
        } catch (Exception e) {
            logger.error("RA证书错误消息 ========== 解析normal的错误消息时异常",e);
            result.setError(ErrorEnum.NORMAI_ERROR_MES_REQ_PARAMS_ERROR);
            return result;
        }

        logger.info("RA证书错误消息 ========== 【结束】");
        redisCacheManagerService.removeCaTransId(applyNo);
        return result;
    }
}
