package com.xdja.pki.ra.openapi.normal.handler;

import com.xdja.ca.constant.SdkConstants;
import com.xdja.ca.vo.UserCertInfo;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.cert.PKICertHelper;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.manager.dao.CertApplyDao;
import com.xdja.pki.ra.openapi.core.BaseCMPInfo;
import com.xdja.pki.ra.openapi.core.common.FreeText;
import com.xdja.pki.ra.openapi.core.common.PKIMessageException;
import com.xdja.pki.ra.openapi.core.handler.ICmpMessageHandler;
import com.xdja.pki.ra.openapi.core.helper.PKIMessageHelper;
import com.xdja.pki.ra.service.manager.cache.RedisCacheManagerService;
import com.xdja.pki.ra.service.manager.certapply.CertApplyManagerService;
import com.xdja.pki.ra.service.manager.customer.CustomerSysService;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.crmf.CertReqMessages;
import org.bouncycastle.asn1.crmf.CertReqMsg;
import org.bouncycastle.asn1.crmf.CertRequest;
import org.bouncycastle.asn1.crmf.CertTemplate;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.IOException;
import java.security.PublicKey;

/**
 * CMP-normal证书恢复处理类
 *
 * @author yangmenghao
 * @date 2020-01-14 08:54:45
 */
@Component(value = "cmpNormalRecoveryCertReqHandler")
public class CmpNormalRecoveryCertReqHandler implements ICmpMessageHandler {
    private final static Logger logger = LoggerFactory.getLogger(CmpNormalRecoveryCertReqHandler.class);
    @Resource
    private CertApplyManagerService certApplyManagerService;
    @Resource
    private CustomerSysService customerSysService;
    @Resource
    private CertApplyDao certApplyDao;
    @Autowired
    private RedisCacheManagerService redisCacheManagerService;

    @Override
    public Result handleMessage(PKIMessage pkiMessage, boolean authenticated) throws PKIMessageException, IOException {
        logger.info("[CmpNormalRecoveryCertReqHandler#handleMessage] RA接收normal更新处理 ========== 【开始】");
        Result result = new Result();
        logger.info("RA恢复处理 ========== 1. 获取PkiMessage消息结构");
        // PKIMessage
        PKIMessage reqPkiMessage = getPkiMessage(pkiMessage);
        // PKIHeader
        PKIHeader header = getPkiHeader(reqPkiMessage);
        GeneralName sender = GeneralName.getInstance(header.getSender());
        GeneralName recipient = header.getRecipient();
        byte[] recipNonce;
        byte[] senderNonce;
        String tranID;
        AlgorithmIdentifier protectionAlg;
        try {
            recipNonce = header.getRecipNonce() == null ? null : header.getRecipNonce().getOctets();
            senderNonce = header.getSenderNonce() == null ? null : header.getSenderNonce().getOctets();
            tranID = header.getTransactionID() == null ? null : new String(header.getTransactionID().getOctets());
            protectionAlg = header.getProtectionAlg();
        } catch (Exception e) {
            String errMsg = "RA恢复处理 ========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg, e);
        }
        //获取freeText中存放的模板信息
        FreeText freeText = null;
        PKIFreeText pkiFreeText = header.getFreeText();
        if (pkiFreeText != null) {
            String freeTestStr = pkiFreeText.getStringAt(0).getString();
            freeText = JsonUtils.json2Object(freeTestStr, FreeText.class);
            if (freeText == null) {
                logger.info("RA恢复处理 ========== PKI消息体中不包含申请信息");
                KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
                PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
                result.setInfo(errPkiMessage);
                return result;
            }
        }else {
            logger.info("RA恢复处理 ========== PKI消息体中不包含申请信息");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        String applyNo = freeText.getApplyNo();
        /**
         * 秘钥格式
         */
        Integer keyFormat = freeText.getKeyFormat();
        String normalSender = sender.getName().toString();
        // 获取缓存请求头部信息
        String baseCMPInfoStr = redisCacheManagerService.removeRaSdkCmpInfo(tranID);
        BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(baseCMPInfoStr, BaseCMPInfo.class);
        if (baseCMPInfo == null) {
            logger.info("RA恢复处理 ========== 不存在对应的事务ID tranId:{}", tranID);
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(-1, ErrorEnum.CMP_TRAN_ID_IS_NOT_EXIST.code, ErrorEnum.CMP_TRAN_ID_IS_NOT_EXIST.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        long signRequestId = getSignRequestId(reqPkiMessage);
        baseCMPInfo.setRequestId(signRequestId);
        redisCacheManagerService.cacheRaSdkCmpInfo(tranID, JsonUtils.object2Json(baseCMPInfo));

        // 通过第三方唯一标识 确认第三方系统的证书和公钥
        Result certResult = customerSysService.getSysCertBySysNumber(normalSender, freeText.getSignSn());
        PublicKey publicKey;
        if (certResult.isSuccess()) {
            publicKey = (PublicKey) certResult.getInfo();
        } else {
            logger.info("RA恢复处理 ========== 通过第三方系统标识确认第三方的证书错误 原因：{}", JsonUtils.object2Json(certResult));
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(-1, ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_ERROR.code, ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_ERROR.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA恢复处理 ========== 3. 验证cmp消息的header和签名的正确性");
        // 验证cmp消息的header和消息保护
        DERBitString protectData = reqPkiMessage.getProtection();// 被保护的内容的签名值
        byte[] sourceData = PKIMessageHelper.getProtectedBytes(pkiMessage); // 被保护内容

        // 验证cmp消息的正确性
        Result checkResult = PKIMessageHelper.checkCmpHeaderAndSign(publicKey, header, protectData.getBytes(), sourceData, protectionAlg, null);
        if (!checkResult.isSuccess()) {
            logger.info("RA恢复处理 ========== 验证cmp消息的header和签名错误 原因：{}", JsonUtils.object2Json(checkResult));
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(-1, checkResult.getError().code, checkResult.getError().desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }

        // 获取PKIMessage的Body体
        PKIBody body = pkiMessage.getBody();
        if (body == null) {
            logger.info("RA恢复处理 ========== 没有对应的PKI消息体");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.NO_PKI_BODY_FOR_RECEIVED.code, ErrorEnum.NO_PKI_BODY_FOR_RECEIVED.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }

        if (body.getType() != PKIBody.TYPE_KEY_RECOVERY_REQ) {
            logger.info("RA恢复处理 ========== PKI消息体的类型不是9");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.ISSUE_CERT_PKI_BODY_TAG_NOT_0_OR_2.code, ErrorEnum.ISSUE_CERT_PKI_BODY_TAG_NOT_0_OR_2.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }

        // 如果是普通的用户双证证书证书申请 当前CertReqMsg只会有一个 获取直接获取第0号 并拿到requestId
        ASN1Encodable content = body.getContent();
        CertReqMessages myCertReqMessages = CertReqMessages.getInstance(content);
        CertReqMsg[] certReqMsgs = myCertReqMessages.toCertReqMsgArray();
        CertReqMsg certReqMsg = certReqMsgs[0];
        CertRequest certReq = certReqMsg.getCertReq();
        CertTemplate certTemplate = certReq.getCertTemplate();

        // 用户输入-P10中的公钥
        PublicKey normalPublicKey = null;
        byte[] publickey;
        try {
            SubjectPublicKeyInfo subPublicKey = certTemplate.getPublicKey();
            if (subPublicKey == null) {
                publickey = null;
                logger.info("RA恢复处理 ========== PKI消息体中公钥信息获取失败");
            } else {
                normalPublicKey = PKICertHelper.getPublicKeyFromSubjectPublicKey(subPublicKey, "BC");
                publickey = normalPublicKey.getEncoded();
            }
        } catch (Exception e) {
            logger.info("RA恢复处理 ========== PKI消息体中公钥信息获取异常");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.GET_PKI_MESSAGE_PUBLIC_KEY_EXCEPTION.code, ErrorEnum.GET_PKI_MESSAGE_PUBLIC_KEY_EXCEPTION.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA恢复处理 ========== 更新申请的applyNo：{}  第三方系统-唯一标识:{} ", applyNo, normalSender);

        redisCacheManagerService.cacheNomalTransId(tranID, applyNo);

        //向CA发起更新证书
        logger.info("RA恢复处理 ========== 4. 发起恢复用户双证书请求");
        Result recoveryResult = certApplyManagerService.certApplyCarry(normalSender, Constants.CERT_APPLY_TYPE_RECOVERY_4, applyNo, freeText.getDoubleCode(), publickey, keyFormat);
        if (!recoveryResult.isSuccess()) {
            logger.info("RA恢复处理 ========== 4.1.恢复用户双证书请求错误:{} ", JsonUtils.object2Json(recoveryResult));
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, recoveryResult.getErrorBean().getErrCode(), recoveryResult.getErrorBean().getErrMsg());
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA恢复处理 ========== CA返回的证书恢复请求的响应结果 >>>>>>>:{} ", JsonUtils.object2Json(recoveryResult));
        if (recoveryResult.getInfo() == null) {
            logger.info("RA恢复处理 ========== 4.2. 恢复用户证书暂无返回证书信息");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.ISSUE_USER_CERT_NO_CERT_INFO.code, ErrorEnum.ISSUE_USER_CERT_NO_CERT_INFO.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }

        UserCertInfo userCertInfo = (UserCertInfo) recoveryResult.getInfo();
        // 是否有必要
        Integer certPatterm = null;
        try {
            certPatterm = certApplyDao.getCertPatterm(applyNo);
        } catch (Exception e) {
            logger.info("获取签发证书申请基本信息为空");
            result.setError(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
            return result;
        }

        if (null == userCertInfo.getSignCert()) {
            logger.info("RA恢复处理 ========== 4.3. 用户证书或加密证书为空");
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.code, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        if (SdkConstants.CERT_TYPE_SINGLE_1 != certPatterm) {
            if (null == userCertInfo.getEncCert()) {
                logger.info("RA恢复处理 ========== 4.3. 用户证书或加密证书为空");
                KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.code, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.desc);
                PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
                result.setInfo(errPkiMessage);
                return result;
            }

        }
        logger.info("密钥恢复申请处理 ========== 5. 将密钥恢复对象证书KeyRecRepContent结构体");
        // 封装KeyRecRepContent结构体
        KeyRecRepContent keyRecRepContent = null;
        try {
            keyRecRepContent = PKIMessageHelper.genKeyRecRepContent(userCertInfo);
        } catch (Exception e) {
            logger.error("封装密钥恢复返回对象KeyRecRepContent异常", e);
            KeyRecRepContent failKeyRecRepContent = PKIMessageHelper.genFailKeyRecRepContent(signRequestId, ErrorEnum.GEN_KEY_RECOVERY_REP_EXCEPTION.code, ErrorEnum.GEN_KEY_RECOVERY_REP_EXCEPTION.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, failKeyRecRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA恢复处理 ========== 6. 封装PKIMessage结构体");
        PKIMessage repPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_RECOVERY_REP, recipNonce, senderNonce, tranID, keyRecRepContent,null, userCertInfo.getExtraCertsP7b());
        result.setInfo(repPkiMessage);

        logger.info("RA恢复处理 ========== 【结束】");
        return result;

    }


    private PKIMessage getPkiMessage(PKIMessage pkiMessage) throws PKIMessageException {
        PKIMessage reqPkiMessage = PKIMessage.getInstance(pkiMessage);
        if (reqPkiMessage == null) {
            String errMsg = "RA恢复处理========== No pkiMessage response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }
        return reqPkiMessage;
    }

    private PKIHeader getPkiHeader(PKIMessage reqPkiMessage) throws PKIMessageException {
        PKIHeader header = null;
        logger.info("RA恢复处理 ========== 2. 获取PkiMessage消息头PKIHeader");
        try {
            header = reqPkiMessage.getHeader();
        } catch (Exception e) {
            logger.info("RA恢复处理 opani========", e);
        }
        if (header == null) {
            String errMsg = "RA恢复处理========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }
        return header;
    }

    private long getSignRequestId(PKIMessage reqPkiMessage) {
        CertReqMessages ir = (CertReqMessages) reqPkiMessage.getBody().getContent();
        return ir.toCertReqMsgArray()[0].getCertReq().getCertReqId().getValue().longValue();
    }
}
