package com.xdja.pki.ra.openapi.normal.handler;

import com.xdja.ca.constant.SdkConstants;
import com.xdja.ca.vo.UserCertInfo;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.cert.PKICertHelper;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.manager.dao.CertApplyDao;
import com.xdja.pki.ra.openapi.core.BaseCMPInfo;
import com.xdja.pki.ra.openapi.core.common.CmpRespCertType;
import com.xdja.pki.ra.openapi.core.common.FreeText;
import com.xdja.pki.ra.openapi.core.common.PKIMessageException;
import com.xdja.pki.ra.openapi.core.handler.ICmpMessageHandler;
import com.xdja.pki.ra.openapi.core.helper.PKIMessageHelper;
import com.xdja.pki.ra.service.manager.cache.RedisCacheManagerService;
import com.xdja.pki.ra.service.manager.certapply.CertApplyManagerService;
import com.xdja.pki.ra.service.manager.certapply.bean.DoubleCode;
import com.xdja.pki.ra.service.manager.customer.CustomerSysService;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.crmf.CertReqMessages;
import org.bouncycastle.asn1.crmf.CertReqMsg;
import org.bouncycastle.asn1.crmf.CertRequest;
import org.bouncycastle.asn1.crmf.CertTemplate;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.security.PublicKey;

/**
 * @author wly
 */
@Component("cmpNormalUpdateCertReqHandler")
public class CmpNormalUpdateCertReqHandler implements ICmpMessageHandler {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    CustomerSysService customerSysService;
    @Autowired
    CertApplyManagerService certApplyManagerService;
    @Autowired
    CertApplyDao certApplyDao;
    @Autowired
    RedisCacheManagerService redisCacheManagerService;

    @Override
    public Result handleMessage(PKIMessage pkiMessage, boolean authenticated) throws PKIMessageException {
        logger.info("RA接收normal更新处理 ========== 【开始】");
        Result result = new Result();

        logger.info("RA更新处理 ========== 1. 获取PkiMessage消息结构");
        PKIMessage reqPkiMessage = PKIMessage.getInstance(pkiMessage);
        if (reqPkiMessage == null) {
            String errMsg = "RA更新处理 ========== No pkiMessage response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }

        logger.info("RA更新处理 ========== 2. 获取PkiMessage消息头PKIHeader");
        final PKIHeader header = reqPkiMessage.getHeader();
        if (header == null) {
            String errMsg = "RA更新处理 ========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }
        GeneralName sender = GeneralName.getInstance(header.getSender());
        GeneralName recipient = header.getRecipient();
        byte[] normalRecipNonce = new byte[0];
        byte[] normalSenderNonce = new byte[0];
        String normalTranID = null;
        AlgorithmIdentifier protectionAlg = null;
        try {
            normalRecipNonce = header.getRecipNonce() == null ? null : header.getRecipNonce().getOctets();
            normalSenderNonce = header.getSenderNonce() == null ? null : header.getSenderNonce().getOctets();
            normalTranID = header.getTransactionID() == null ? null : new String(header.getTransactionID().getOctets());
            protectionAlg = header.getProtectionAlg();
        } catch (Exception e) {
            String errMsg = "RA更新处理 ========== No header in response message.";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg, e);
        }

        // 对必填项参数的校验
        if (normalRecipNonce == null || normalSenderNonce == null || protectionAlg == null || StringUtils.isBlank(normalTranID)) {
            String errMsg = "RA更新处理 ========== 更新接口中必填项有空值";
            logger.info(errMsg);
            throw new PKIMessageException(errMsg);
        }

        // 获取缓存请求头部信息
        String baseCMPInfoStr = redisCacheManagerService.removeRaSdkCmpInfo(normalTranID);
        BaseCMPInfo baseCMPInfo = JsonUtils.json2Object(baseCMPInfoStr, BaseCMPInfo.class);
        if (baseCMPInfo == null) {
            logger.info("更新处理 ========== 不存在对应的事务ID tranId:{}", normalTranID);
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(-1, ErrorEnum.CMP_TRAN_ID_IS_NOT_EXIST.code, ErrorEnum.CMP_TRAN_ID_IS_NOT_EXIST.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        CertReqMessages ir = (CertReqMessages) reqPkiMessage.getBody().getContent();
        long signRequestId = ir.toCertReqMsgArray()[0].getCertReq().getCertReqId().getValue().longValue();
        baseCMPInfo.setRequestId(signRequestId);
        redisCacheManagerService.cacheRaSdkCmpInfo(normalTranID, JsonUtils.object2Json(baseCMPInfo));

        //获取申请编号-applyNo
        FreeText freeText = null;
        PKIFreeText pkiFreeText = header.getFreeText();
        if (pkiFreeText != null) {
            String freeTestStr = pkiFreeText.getStringAt(0).getString();
            freeText = JsonUtils.json2Object(freeTestStr, FreeText.class);
            if (freeText == null) {
                logger.info("更新申请处理 ========== PKI消息体中不包含申请信息");
                RevRepContent revRepContent = PKIMessageHelper.genFailRevRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
                PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, revRepContent);
                result.setInfo(errPkiMessage);
                return result;
            }
        }else {
            logger.info("更新申请处理 ========== PKI消息体中不包含申请信息");
            RevRepContent revRepContent = PKIMessageHelper.genFailRevRepContent(-1, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.code, ErrorEnum.PKI_MESSAGE_NOT_CONTENT_REVOKE_INFO.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, revRepContent);
            result.setInfo(errPkiMessage);
            return result;
        }
        String applyNo = freeText.getApplyNo();
        DoubleCode doubleCode = freeText.getDoubleCode();
        String signSn = freeText.getSignSn();
        /**
         * 秘钥格式
         */
        Integer keyFormat = freeText.getKeyFormat();
        String normalSender = sender.getName().toString();
        // 通过第三方唯一标识 确认第三方系统的证书和公钥
        Result certResult = customerSysService.getSysCertBySysNumber(normalSender,signSn);
        PublicKey publicKey;
        if (certResult.isSuccess()) {
            publicKey = (PublicKey) certResult.getInfo();
        } else {
            logger.info("RA签发处理 ========== 通过第三方系统标识确认第三方的证书错误 原因：{}", JsonUtils.object2Json(certResult));
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(-1, ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_ERROR.code, ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_ERROR.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA更新处理 ========== 3. 验证cmp消息的header和签名的正确性");
        // 验证cmp消息的header和消息保护
        DERBitString protectData = reqPkiMessage.getProtection();// 被保护的内容的签名值
        byte[] sourceData = PKIMessageHelper.getProtectedBytes(pkiMessage); // 被保护内容

        // 验证cmp消息的正确性
        Result checkResult = PKIMessageHelper.checkCmpHeaderAndSign(publicKey, header, protectData.getBytes(), sourceData, protectionAlg, null);
        if (!checkResult.isSuccess()) {
            logger.info("RA更新处理 ========== 验证cmp消息的header和签名错误 原因：{}", JsonUtils.object2Json(checkResult));
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(-1, checkResult.getError().code, checkResult.getError().desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        // 获取PKIMessage的Body体
        PKIBody body = pkiMessage.getBody();
        if (body == null) {
            logger.info("RA更新处理 ========== 没有对应的PKI消息体");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.NO_PKI_BODY_FOR_RECEIVED.code, ErrorEnum.NO_PKI_BODY_FOR_RECEIVED.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        if (body.getType() != PKIBody.TYPE_KEY_UPDATE_REQ) {
            logger.info("RA更新处理 ========== PKI消息体的类型不是7");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.ISSUE_CERT_PKI_BODY_TAG_NOT_0_OR_2.code, ErrorEnum.ISSUE_CERT_PKI_BODY_TAG_NOT_0_OR_2.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        // 当前CertReqMsg只会有一个 获取直接获取第0号 并拿到requestId
        ASN1Encodable content = body.getContent();
        CertReqMessages myCertReqMessages = CertReqMessages.getInstance(content);
        CertReqMsg[] certReqMsgs = myCertReqMessages.toCertReqMsgArray();
        CertReqMsg certReqMsg = certReqMsgs[0];
        CertRequest certReq = certReqMsg.getCertReq();
        CertTemplate certTemplate = certReq.getCertTemplate();

        // 用户输入-P10中的公钥
        PublicKey normalPublicKey = null;
        byte[] publickey ;
        try {
            SubjectPublicKeyInfo subPublicKey = certTemplate.getPublicKey();
            if (subPublicKey == null) {
                publickey = null;
            }else {
                normalPublicKey = PKICertHelper.getPublicKeyFromSubjectPublicKey(subPublicKey, "BC");
                publickey = normalPublicKey.getEncoded();
            }
        } catch (Exception e) {
            logger.info("RA签发申请处理 ========== PKI消息体中公钥信息获取异常");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.GET_PKI_MESSAGE_PUBLIC_KEY_EXCEPTION.code, ErrorEnum.GET_PKI_MESSAGE_PUBLIC_KEY_EXCEPTION.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA更新处理 ========== 更新申请的applyNo：[{}] 第三方系统-唯一标识: [{}]", applyNo, normalSender);

        redisCacheManagerService.cacheNomalTransId(normalTranID, applyNo);

        //向CA发起更新证书
        logger.info("RA更新申请处理 ========== 4. 发起更新用户双证书请求");
        Result issueResult = certApplyManagerService.certApplyCarry(normalSender,Constants.CERT_APPLY_TYPE_UPDATE_2, applyNo, doubleCode, publickey,keyFormat);
        if (!issueResult.isSuccess()) {
            logger.info("RA更新申请处理 ========== 4.1. 更新用户双证书请求错误:{}", JsonUtils.object2Json(issueResult));
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, issueResult.getErrorBean().getErrCode(),issueResult.getErrorBean().getErrMsg());
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        logger.info("RA更新申请处理 ========== CA返回的证书更新请求的响应结果 >>>>>>>：{} ", JsonUtils.object2Json(issueResult));
        if (issueResult.getInfo() == null) {
            logger.info("RA更新申请处理 ========== 4.2. 更新用户证书暂无返回证书信息");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.ISSUE_USER_CERT_NO_CERT_INFO.code, ErrorEnum.ISSUE_USER_CERT_NO_CERT_INFO.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        UserCertInfo userCertInfo = (UserCertInfo) issueResult.getInfo();

        Integer certPatterm = null;
        try {
            certPatterm = certApplyDao.getCertPatterm(applyNo);
        } catch (Exception e) {
            logger.info("获取签发证书申请基本信息为空");
            result.setError(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
            return result;
        }

        if (null == userCertInfo.getSignCert()) {
            logger.info("RA更新申请处理 ========== 4.3. 用户证书或加密证书为空");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.code, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        if(SdkConstants.CERT_TYPE_SINGLE_1 != certPatterm ){
            if (null == userCertInfo.getEncCert()) {
                logger.info("RA更新申请处理 ========== 4.3. 用户证书或加密证书为空");
                CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.code, ErrorEnum.SIGN_CERT_OR_ENC_CERT_IS_EMPTY.desc);
                PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
                result.setInfo(errPkiMessage);
                return result;
            }

        }

        // 封装多个CertResponse
        CertResponse signCertResponse = null;
        CertResponse encCertResponse = null;
        try {
            logger.info("RA更新申请处理 ========== 5. 将证书封装签名CertResponse结构体");
            signCertResponse = PKIMessageHelper.genCertResponse(signRequestId, userCertInfo, CmpRespCertType.GEN_CERT_RESPONSE_SIGN_CERT_1.value,keyFormat);
            if (SdkConstants.CERT_TYPE_SINGLE_1 != certPatterm) {
                if (StringUtils.isNotBlank(userCertInfo.getEncPriKey())) {
                    logger.info("RA更新申请处理 ========== 6. 将证书封装加密(有加密私钥信封)CertResponse结构体");
                    encCertResponse = PKIMessageHelper.genCertResponse(signRequestId, userCertInfo, CmpRespCertType.GEN_CERT_RESPONSE_ENC_CERT_AND_ENC_PRI_KEY_2.value,keyFormat);
                } else {
                    logger.info("更新申请处理 ========== 6. 将证书封装加密CertResponse结构体");
                    encCertResponse = PKIMessageHelper.genCertResponse(signRequestId, userCertInfo, CmpRespCertType.GEN_CERT_RESPONSE_ENC_CERT_3.value,keyFormat);
                }
            }

        } catch (Exception e) {
            logger.error("RA更新申请处理 ========== 封装CertResponse结构体异常", e);
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.MAKE_CERT_RESPONSE_ERROR.code, ErrorEnum.MAKE_CERT_RESPONSE_ERROR.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }
        CertResponse[] certRespos ;
        if (signCertResponse != null && encCertResponse == null) {
            certRespos = new CertResponse[1];
            certRespos[0] = signCertResponse;
        } else if (signCertResponse != null && encCertResponse != null) {
            certRespos = new CertResponse[2];
            certRespos[0] = signCertResponse;
            certRespos[1] = encCertResponse;
        } else {
            logger.info("更新申请处理 ========== 6.1. 封装CertResponse失败");
            CertRepMessage certRepMessage = PKIMessageHelper.genFailCertResponse(signRequestId, ErrorEnum.MAKE_CERT_RESPONSE_ERROR.code, ErrorEnum.MAKE_CERT_RESPONSE_ERROR.desc);
            PKIMessage errPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage);
            result.setInfo(errPkiMessage);
            return result;
        }

        logger.info("RA更新申请处理 ========== 7. 封装CertRepMessage结构体");
        CertRepMessage certRepMessage = new CertRepMessage(null, certRespos);

        // 生成CertRepMessage结构体
        logger.info("RA更新申请处理 ========== 8. 封装PKIMessage结构体");
        PKIMessage repPkiMessage = PKIMessageHelper.generatePKIMessage(sender, recipient, PKIBody.TYPE_KEY_UPDATE_REP, normalRecipNonce, normalSenderNonce, normalTranID, certRepMessage, null, userCertInfo.getExtraCertsP7b());
        result.setInfo(repPkiMessage);

        logger.info("RA更新申请处理 ========== 【结束】");
        return result;
    }
}
