package com.xdja.ra.sdk;

import com.alibaba.fastjson.JSON;
import com.xdja.ra.bean.*;
import com.xdja.ra.constant.SdkConstants;
import com.xdja.ra.ennum.ReqMethodEnum;
import com.xdja.ra.ennum.SignatureAlgorithmEnum;
import com.xdja.ra.error.ErrorEnum;
import com.xdja.ra.utils.*;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URLDecoder;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.net.InetSocketAddress;
import java.net.Socket;

/**
 * @author: ggp
 * @Date: 2019/11/1 13:54
 * @Description:
 */
public class SDKService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static SDKService sdkService;

    public static SDKService getInstance(){
        return sdkService;
    }
    public static Config config;

    public SDKService() {
    }

    public SDKService(Config config) {
        config = config;
    }

    public void init() {
        try {
            config = ConfigUtils.getConfig();
            String baseUrl = config.getServerIp() + ":" + config.getServerPort() + "/ra-web";
            config.setRaBaseUrl(baseUrl);
            X509Certificate cert = (X509Certificate) KeyStoreUtils.getPublicKeyFromP12(null, config.getPrikeyPath(), config.getPriKeyPwd());
            config.setSystemCert(cert);
            config.setUserCertSn(cert.getSerialNumber().toString(16));
            config.setCertSignAlgOid(cert.getSigAlgOID());
            if(null == cert.getSigAlgName()) {
                config.setSignName(SignatureAlgorithmEnum.getName(config.getCertSignAlgOid()));
            }else{
                config.setSignName(cert.getSigAlgName());
            }
            X509Certificate[] caCerts;
            String trustCertP7b = FileUtils.read(config.getTrustCertPath());
            byte[] base64Bytes = Base64.decode(trustCertP7b);
            List<X509Certificate> certs = SignedDataUtils.resolveCertChain(base64Bytes);
            caCerts = new X509Certificate[certs.size()];
            for (int i = 0; i < certs.size(); i++) {
                caCerts[i] = certs.get(i);
            }
            config.setTrustCert(caCerts);
            PrivateKey privateKey = SignUtils.getPrivateKeyFromP12(config.getPrikeyPath(), config.getPriKeyPwd().toCharArray());
            config.setPrivateKey(privateKey);
            sdkService = new SDKService(config);
        } catch (Exception e) {
            logger.error("初始化SDKService异常",e);
        }
    }

    /**
     * 初始化类
     *
     * @param serverIp
     * @param serverPort
     * @param prikeyPath
     * @param prikeyPwd
     * @param trustCertPath
     * @param systemFlag
     * @return
     */
    public SDKService(String serverIp, String serverPort, String prikeyPath, String prikeyPwd, String trustCertPath, String systemFlag) {
        try {
            ConfigUtils.setServerIp(serverIp);
            ConfigUtils.setServerPort(serverPort);
            ConfigUtils.setPriKeyPath(prikeyPath);
            ConfigUtils.setPriKeyPwd(prikeyPwd);
            ConfigUtils.setTrustCertPath(trustCertPath);
            ConfigUtils.setSystemFlag(systemFlag);
            init();
            SignUtils.sign(SDKService.config.getSignName(), SDKService.config.getPrivateKey(),"init");
            JSON.toJSONString(new UserInfo());
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            throw new RuntimeException("初始化SDK失败", e);
        }
    }

    /**
     * 用户注册
     *
     * @param userType
     * @param userInfo
     * @return
     */
    public Result userRegist(Integer userType, UserInfo userInfo) {
        Result result;
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType;
            String source = reqUrl + JSON.toJSONString(userInfo);
            result = buildData(reqUrl, source, ReqMethodEnum.POST.name, JSON.toJSONString(userInfo));
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }


    /**
     * 用户信息查询
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result userQuery(Integer userType, Integer identType, String identNumber) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                UserInfo userInfo = JSON.parseObject(new String((byte[]) result.getInfo()), UserInfo.class);
                result.setInfo(userInfo);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 用户信息更新
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result userUpdate(Integer userType, Integer identType, String identNumber, UserInfo userInfo) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber;
            String source = reqUrl + JSON.toJSONString(userInfo);
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(userInfo));
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 修改用户状态
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @param userStatus
     * @return
     */
    public Result userStatusUpdate(Integer userType, Integer identType, String identNumber, Integer userStatus) {
        try {
            String reqUrl = "/v1/ra-openapi/user/" + userType + "/" + identType + "/" + identNumber + "/" + userStatus;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, null);
            result.setInfo(null);
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 获取RA系统BaseDn
     *
     * @return
     */
    public Result getRASystemBaseDn() {
        try {
            String reqUrl = "/v1/ra-openapi/apply/baseDn";
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                result.setInfo(new String((byte[]) result.getInfo()));
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 获取用户模板信息
     *
     * @param systemFlag
     * @return
     */
    public Result getTemplateInfoList(String systemFlag) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/template/" + systemFlag;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                List<TemplateInfo> templateInfoList = JSON.parseArray(new String((byte[]) result.getInfo()), TemplateInfo.class);
                result.setInfo(templateInfoList);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }

    }

    /**
     * 签发申请-发起
     *
     * @param issueApply
     * @return
     */
    public Result startIssueUserCertApply(IssueApply issueApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/issue";
            return getApplyResult(JSON.toJSONString(issueApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }


    /**
     * 更新申请-发起
     *
     * @param updateApply
     * @return
     */
    public Result startUpdateUserCertApply(UpdateApply updateApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/update";
            return getApplyResult(JSON.toJSONString(updateApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 撤销申请-发起
     *
     * @param revokeApply
     * @returne
     */
    public Result startRevokeUserCertApply(RevokeApply revokeApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/revoke";
            return getApplyResult(JSON.toJSONString(revokeApply), reqUrl);
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 申请审核
     *
     * @param applyType
     * @param applyNo
     * @param checkApply
     * @return
     */
    public Result checkUserCertApply(Integer applyType, String applyNo, CheckApply checkApply) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/check/" + applyType + "/" + applyNo;
            String source = reqUrl + JSON.toJSONString(checkApply);
            Result result = buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(checkApply));
            if (result.isSuccess()) {
                DoubleCode doubleCode = JSON.parseObject(new String((byte[]) result.getInfo()), DoubleCode.class);
                result.setInfo(doubleCode);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 申请制证执行
     *
     * @param applyType
     * @param applyNo
     * @param userCertReq
     * @return
     */
    public Result generateUserCert (Integer applyType, String applyNo, UserCertReq userCertReq){
        Result result = new Result();
        if (applyType == null || userCertReq == null || StringUtils.isAnyBlank(applyNo, userCertReq.getAuthCode(), userCertReq.getRefCode())) {
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        if (applyType != SdkConstants.CERT_APPLY_TYPE_ISSUE_1 && applyType != SdkConstants.CERT_APPLY_TYPE_UPDATE_2) {
            return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }
        //事务控制 ID
        String transId = UUID.randomUUID().toString().replace("-", "");
        RaCmpApi raCmpApi = new RaCmpApi();
        Result sdkResult  ;
        try {
            sdkResult = raCmpApi.sendCertReqMessage(applyType, transId, config.getSystemFlag(), config.getCertSignAlgOid(), applyNo, userCertReq);
        } catch (Exception e) {
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        logger.debug("SDKService.generateUserCert.result:" + SdkJsonUtils.object2Json(sdkResult.getInfo()));
        return getCmpResult(result,sdkResult);
    }

    /**
     * 证书签发确认消息
     *
     * @param applyNo
     * @return
     */
    public Result certApplyConfirmMsg (String applyNo){
        Result result = new Result();
        RaCmpApi raCmpApi = new RaCmpApi();
        Result sdkResult;
        try {
            sdkResult = raCmpApi.sendConfirmMessage(config.getCertSignAlgOid(), config.getSystemFlag(), applyNo);
        } catch (Exception e) {
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        return getCmpResult(result,sdkResult);
    }

    /**
     * 证书签发错误消息
     *
     * @param applyNo
     * @param errorMsg
     * @return
     */
    public Result certApplyErrorMsg (String applyNo, ErrorMsg errorMsg){
        Result result = new Result();
        RaCmpApi raCmpApi = new RaCmpApi();
        Result sdkResult;
        try {
            sdkResult = raCmpApi.sendErrorMessage(applyNo, config.getCertSignAlgOid(), config.getSystemFlag(), errorMsg.getErrorMsg(), errorMsg.getErrorCode());
        } catch (Exception e) {
            return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
        }
        return getCmpResult(result,sdkResult);
    }

    private Result getCmpResult (Result result, Result sdkResult){
        if (!sdkResult.isSuccess()) {
            if (SdkConstants.ERROR_CODE_110.equals(String.valueOf(sdkResult.getErrorMsg().getErrorCode()).substring(0, 3))) {
                return Result.failure(ErrorEnum.SDK_INNER_EXCEPTION);
            } else {
                result.setErrorMsg(sdkResult.getErrorMsg());
                return result;
            }
        }
        return Result.success(sdkResult.getInfo());
    }


    /**
     * 查询申请
     *
     * @param applyType
     * @param applyNo
     * @return
     */
    public Result certApplyQuery(Integer applyType, String applyNo) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/" + applyType + "/" + applyNo;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                CertApplyInfo certApplyInfo = JSON.parseObject(new String((byte[]) result.getInfo()), CertApplyInfo.class);
                result.setInfo(certApplyInfo);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 更新证书申请信息
     *
     * @param applyType
     * @param applyNo
     * @param editCertApplyInfo
     * @return
     */
    public Result certApplyUpdate(Integer applyType, String applyNo, EditCertApplyInfo editCertApplyInfo) {
        try {
            String reqUrl = "/v1/ra-openapi/apply/" + applyType + "/" + applyNo;
            String source = reqUrl + JSON.toJSONString(editCertApplyInfo);
            return buildData(reqUrl, source, ReqMethodEnum.PUT.name, JSON.toJSONString(editCertApplyInfo));
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 查询用户证书
     *
     * @param userType
     * @param identType
     * @param identNumber
     * @return
     */
    public Result userCertQuery(Integer userType, Integer identType, String identNumber) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/" + userType + "/" + identType + "/" + identNumber;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                List<UserCertInfo> userCertInfoList = JSON.parseArray(new String((byte[]) result.getInfo()), UserCertInfo.class);
                result.setInfo(userCertInfoList);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 查询证书详情
     *
     * @param certSn
     * @return
     */
    public Result userCertDetailQuery(String certSn) {
        try {
            String reqUrl = "/v1/ra-openapi/cert/" + certSn;
            String source = reqUrl;
            Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
            if (result.isSuccess()) {
                CertBaseInfo certBaseInfo = JSON.parseObject(new String((byte[]) result.getInfo()), CertBaseInfo.class);
                result.setInfo(certBaseInfo);
            } else {
                result.setInfo(null);
            }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }
    }

    /**
     * 证书下载
     *
     * @param certSn
     * @return
     */
    public Result userCertDownload(String certSn) {
        try{

        String reqUrl = "/v1/ra-openapi/cert/file/" + certSn;
        String source = reqUrl;
        Result result = buildData(reqUrl, source, ReqMethodEnum.GET.name, null);
        if (!result.isSuccess()) {
            result.setInfo(null);
        }
            return result;
        } catch (Exception e) {
            logger.error("读写配置文件失败", e);
            return Result.failure(ErrorEnum.WRITE_OR_READ_CONFIG_FILE_FAIL);
        }

    }

    /**
     * 测试端口连通性
     */
    private boolean isHostConnectivity(String host, int port) {
        Socket socket = new Socket();
        try {
            socket.connect(new InetSocketAddress(host, port));
        } catch (Exception e) {
            logger.debug("isHostConnectivity socket connect {}:{} error", host, port, e);
            return false;
        } finally {
            try {
                socket.close();
            } catch (Exception e) {
                logger.debug("isHostConnectivity socket close {}:{} error", host, port, e);
            }
        }
        return true;
    }

    /**
     * 组装数据  发送请求
     *
     * @param reqUrl
     * @param source
     * @param methodName
     * @return
     * @throws Exception
     */
    private Result buildData(String reqUrl, String source, String methodName, String reqBody) throws Exception {
        String url = config.getRaBaseUrl() + reqUrl;
        String sourceData = Base64.toBase64String(URLDecoder.decode(source, "UTF-8").getBytes("utf-8"));
        String sign = SignUtils.sign(config.getSignName(), config.getPrivateKey(),sourceData );
        Map<String, String> header = new HashMap<>();
        /**
         * 组装header
         */
        header.put("Content-Type", "application/json; charset=UTF-8");
        header.put("systemFlag", config.getSystemFlag());
        header.put("timestamp", String.valueOf(System.currentTimeMillis()));
        header.put("signAlg", config.getSignName());
        header.put("signValue", sign);
        header.put("signSn", config.getUserCertSn());
        logger.debug("请求参数sign[{}]", sign);
        logger.debug("source:[{}]", sourceData);
        if (null == reqBody) {
            reqBody = "";
        }
        Result result = GMSSLHttpReqUtils.sendGMSSLHttpReq(methodName, url, true, config.getTrustCert(), header, reqBody.getBytes(), null);
        return result;
    }

    /**
     * 发起申请数据组装
     *
     * @param applyReq
     * @param reqUrl
     * @return
     * @throws Exception
     */
    private Result getApplyResult(String applyReq, String reqUrl) throws Exception {
        String source = reqUrl + applyReq;
        Result result = buildData(reqUrl, source, ReqMethodEnum.POST.name, applyReq);
        if (result.isSuccess()) {
            ApplyRep applyRep = JSON.parseObject(new String((byte[]) result.getInfo()), ApplyRep.class);
            result.setInfo(applyRep);
        } else {
            result.setInfo(null);
        }
        return result;
    }

}
