package com.xdja.ra.utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;

/**
 * 文件上传下载工具类
 * syg
 */
public class FileUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(FileUtils.class.getClass());

    /**
     * 读取文件内容
     *
     * @param path 完整文件路径
     * @return 文件内容字符串
     * @throws Exception
     */
    public static String read(String path) throws Exception {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        try {
            File file = new File(path);
        } catch (Exception e) {
           logger.error("创建文件", e);
        }
        try {
            reader = new BufferedReader(new FileReader(new File(path)));

            String temp;

            while (null != (temp = reader.readLine())) {
                content.append(temp);
            }
            return content.toString();
        } catch (Exception e) {
            logger.error("读取文件异常", e);
            throw e;
        } finally {
            if (null != reader) {
                try {
                    reader.close();
                } catch (IOException e) {
                    logger.error("读取文件时流关闭异常", e);
                }
            }
        }
    }

    /**
     * 以字节为单位读取文件，通常用于读取二进制文件，如图片
     *
     * @param path
     * @return
     */
    public static String readByBytes(String path) {
        String content = null;
        InputStream inputStream = null;
        try {
            inputStream = new FileInputStream(path);
            StringBuffer sb = new StringBuffer();
            int c = 0;
            byte[] bytes = new byte[1024];
            while ((c = inputStream.read(bytes)) != -1) {
                sb.append(new String(bytes, 0, c, "utf-8"));
            }
            content = sb.toString();
        } catch (Exception e) {
            logger.error("以字节为单位读取文件", e);
            throw new RuntimeException("以字节为单位读取文件异常", e);
        } finally {
            try {
                if (null != inputStream) {
                    inputStream.close();
                }
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
        return content;
    }

    /**
     * 以行为单位读取文件，常用于读取面向行的格式化文件
     *
     * @param path
     * @return
     */
    public static String readByLines(String path) {
        String content = null;
        BufferedReader bufferedReader = null;
        try {
            File file = new File(path);
            if (!file.exists()) {
                if (!file.createNewFile()) {
                    logger.error("创建文件异常", new Exception("创建文件异常"));
                }
            }
            bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(path), "utf-8"));
            StringBuffer sb = new StringBuffer();
            String temp = null;

            while ((temp = bufferedReader.readLine()) != null) {
                sb.append(temp);
            }
            content = sb.toString();
        } catch (Exception e) {
            logger.error("以行为单位读取文件", e);
            throw new RuntimeException("以行为单位读取文件异常", e);
        } finally {
            try {
                if (null != bufferedReader) {
                    bufferedReader.close();
                }
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
        return content;
    }

    /**
     * 以字符为单位读取文件，常用于读取文本文件
     *
     * @param path
     * @return
     */
    public static String readByChars(String path) {
        String content = null;
        Reader reader = null;
        try {
            reader = new InputStreamReader(new FileInputStream(path), "utf-8");
            StringBuffer sb = new StringBuffer();
            char[] tempchars = new char[1024];
            while (reader.read(tempchars) != -1) {
                sb.append(tempchars);
            }
            content = sb.toString();
        } catch (Exception e) {
            logger.error("以字符为单位读取文件", e);
            throw new RuntimeException(" 以字符为单位读取文件异常", e);
        } finally {
            try {
                if (null != reader) {
                    reader.close();
                }
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
        return content;
    }

    /**
     * 把内容content写的path文件中
     *
     * @param content
     * @param path
     * @return
     */
    public static boolean saveAs(String content, String path) {
        FileWriter fw = null;
        try {
            synchronized (FileUtils.class) {
                File file = new File(path);
                if (file.exists()) {
                    if (!file.delete()) {
                        logger.error("删除文件异常", new Exception("删除文件异常"));
                    }
                }

                if (!file.createNewFile()) {
                    logger.error("创建文件异常", new Exception("创建文件异常"));
                }
                fw = new FileWriter(file, false);
                if (content != null) {
                    fw.write(content);
                }
                fw.flush();
            }
            return true;
        } catch (IOException e) {
            logger.error("写入配置文件异常", e);
            throw new RuntimeException("写入配置文件异常", e);
        } finally {
            try {
                if (null != fw) {
                    fw.close();
                }
            } catch (IOException e) {
                logger.error("写入配置文件异常", e);
            }
        }
    }

    /**
     * 向文件保存内容 （字符串）
     *
     * @param content 待保存内容
     * @param path    文件全路径
     * @throws IOException
     */
    public static void saveFile(String content, String path) {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdirs();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        FileWriter fw = null;
        try {
            synchronized (FileUtils.class) {
                fw = new FileWriter(new File(path), false);
                if (content != null) {
                    fw.write(content);
                }
                fw.flush();
            }
        } catch (IOException e) {
            logger.error("写入配置文件异常", e);
        } finally {
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e) {
                    logger.error("关闭文件写入流异常", e);
                }
            }
        }
    }

    /**
     * 向文件保存内容 (字节)
     *
     * @param content 待保存内容
     * @param path    文件全路径
     * @throws IOException
     */
    public static void saveFile(byte[] content, String path) {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdirs();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        OutputStream out = null;
        InputStream is = null;
        try {
            synchronized (FileUtils.class) {
                out = new FileOutputStream(new File(path), false);
                is = new ByteArrayInputStream(content);
                byte[] buff = new byte[1024];
                int len = 0;
                while ((len = is.read(buff)) != -1) {
                    out.write(buff, 0, len);
                }
            }
        } catch (IOException e) {
            logger.error("写入配置文件异常", e);
        } finally {
            try {
                if (null != is) {
                    is.close();
                }
                if (null != out) {
                    out.close();
                }
            } catch (IOException e) {
                logger.error("关闭文件写入流异常", e);
            }
        }
    }

    /**
     * 读取文件二进制内容
     *
     * @param path
     * @return
     * @throws IOException
     */
    public static byte[] readByBinary(String path) {
        InputStream inputStream = null;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            inputStream = new FileInputStream(path);
            int c = 0;
            byte[] bytes = new byte[1024];
            while ((c = inputStream.read(bytes)) != -1) {
                baos.write(bytes, 0, c);
            }
            return baos.toByteArray();
        } catch (Exception e) {
            logger.error("读取文件二进制内容异常", e);
            throw new RuntimeException("读取文件二进制内容异常", e);
        } finally {
            try {
                if (null != inputStream) {
                    inputStream.close();
                }
                baos.close();
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
    }

    /**
     * 向文件写入二进制数据
     *
     * @param path
     * @param datas
     * @throws IOException
     */
    public static void writeByBinary(String path, byte[] datas) throws IOException {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdir();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        if (!file.exists()) {
            if (!file.createNewFile()) {
                logger.error("创建文件异常", new Exception("创建文件异常"));
            }
        }
        OutputStream os = null;
        try {
            os = new FileOutputStream(path);
            os.write(datas);
        } catch (Exception e) {
            logger.error("写入文件二进制内容异常", e);
            throw new RuntimeException("写入文件二进制内容异常", e);
        } finally {
            try {
                if (null != os) {
                    os.close();
                }
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
    }

    /**
     * 将指定文件写入到输出流中
     *
     * @param out  输出流
     * @param file 待写入输出流文件
     * @throws Exception
     */
    public final static void write(OutputStream out, File file) throws Exception {
        InputStream in = null;

        try {
            in = new FileInputStream(file);

            byte[] data = new byte[1024];
            int size = 0;
            while ((size = in.read(data)) != -1) {
                out.write(data, 0, size);
            }
        } finally {
            try {
                if (null != in) {
                    in.close();
                }
            } catch (Exception e) {
                logger.error("关闭文件流异常", e);
            }
        }
    }
}


