package com.xdja.platform.service.http;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import com.xdja.platform.core.ServiceException;
import com.xdja.platform.service.bean.RequestBean;
import com.xdja.platform.service.bean.ResponseBean;
import com.xdja.platform.service.client.BaseClient;
import com.xdja.platform.util.json.JSONException;
import com.xdja.platform.util.json.JSONUtil;

/**
 * 
 * @ProjectName：platform-service
 * @ClassName：HttpClient
 * @Description：调用http服务的客户端
 * @author: 任瑞修
 * @date: 2013-11-6 上午11:45:54
 *
 */
public class HttpClient extends BaseClient {

	/**
	 * http服务的访问地址
	 */
	private String url = null;
	
	/**
	 * http服务响应处理实例
	 */
	private ResponseHandler<ResponseBean> handler = new ResponseHandler<ResponseBean>() {

	    @Override
	    public ResponseBean handleResponse(
	            final HttpResponse response) throws IOException {
	        StatusLine statusLine = response.getStatusLine();
	        HttpEntity entity = response.getEntity();
	        
	        if (statusLine.getStatusCode() >= HttpServletResponse.SC_MULTIPLE_CHOICES) {
	        	if (statusLine.getStatusCode() == HttpServletResponse.SC_NOT_FOUND) {
					throw new RuntimeException("请求地址不存在");
				}
	        	throw new HttpResponseException(statusLine.getStatusCode(), statusLine.getReasonPhrase());
	        }
	        if (entity == null) {
	            throw new RuntimeException("响应中不包含任何内容");
	        }
	        
	        Reader reader = new InputStreamReader(entity.getContent(), ContentType.getOrDefault(entity).getCharset());
	        try {
				return JSONUtil.toObjecFromReader(reader, ResponseBean.class);
			} catch (JSONException e) {
				throw new RuntimeException(e.getMessage(), e);
			}
	    }
	};
	
	/**
	 * @Description：构造函数
	 * @param url http服务的访问地址
	 * @throws ServiceException 异常
	 */
	public HttpClient(String url) throws ServiceException {
		if (StringUtils.isBlank(url)) {
			throw ServiceException.create(0x00, "http服务地址为空", null);
		}
		this.url = url;
	}
	
	/**
	 * 
	 * @Description：发起http请求，并得到响应结果
	 * @author: 任瑞修
	 * @date: 2013-11-6 下午1:01:20
	 * @param requestBean 请求
	 * @return ResponseBean
	 * @throws ServiceException 异常
	 */
	public ResponseBean request(RequestBean requestBean) throws ServiceException {
		return request(requestBean, null);
	}
	
	public ResponseBean request(RequestBean requestBean, String authorization) throws ServiceException {
		CloseableHttpClient client = HttpClients.createDefault();
		HttpPost post = new HttpPost(url);
		try {
			if(StringUtils.isNotBlank(authorization)) {
				post.addHeader("Authorization", authorization);
			}
			
			RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(10000).build();
			post.setConfig(requestConfig);
			
			post.setEntity(new StringEntity(JSONUtil.toJSONString(requestBean), 
					ContentType.create("application/json", "UTF-8")));
			return client.execute(post, handler);
		} catch (ClientProtocolException e) {
			logger.error(e.getMessage(), e);
			throw ServiceException.create(0x00, e.getMessage(), e);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
			throw ServiceException.create(0x00, e.getMessage(), e);
		} catch (RuntimeException e) {
			logger.error(e.getMessage(), e);
			throw ServiceException.create(0x00, e.getMessage(), e);
		} catch (JSONException e) {
			logger.error(e.getMessage(), e);
			throw ServiceException.create(0x00, e.getMessage(), e);
		} finally {
			try {
				client.close();
			} catch (IOException e) {
				logger.error("http client关闭失败", e);
			}
		}
		
	}
}
