package com.xdja.platform.service.http;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xdja.platform.core.ServiceException;
import com.xdja.platform.service.bean.ErrorBean;
import com.xdja.platform.service.bean.RequestBean;
import com.xdja.platform.service.bean.ResponseBean;
import com.xdja.platform.service.operator.OperatorFactory;
import com.xdja.platform.util.json.JSONException;
import com.xdja.platform.util.json.JsonMapper;

/**
 * 
 * @ProjectName：platform-service
 * @ClassName：HttpServerImpl
 * @Description：Http服务实现
 * @author: 任瑞修
 * @date: 2013-11-5 上午10:41:16
 * 
 */
public class HttpServerImpl extends HttpServlet {

	private static final long serialVersionUID = 5360012772865207427L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	private static final String CONTENT_TYPE = "application/json; charset=UTF-8";

	private JsonMapper jsonMapper = JsonMapper.nonEmptyMapper();
	
	@Override
	public void init() throws ServletException {
		String jsonMapperStr = getServletConfig().getInitParameter("jsonMapper");
		if (StringUtils.equals("NON_NULL", jsonMapperStr)) {
			jsonMapper = JsonMapper.nonNullMapper();
		} else if (StringUtils.equals("NON_DEFAULT", jsonMapperStr)) {
			jsonMapper = JsonMapper.nonDefaultMapper();
		} else if (StringUtils.equals("ALWAYS", jsonMapperStr)) {
			jsonMapper = JsonMapper.alwaysMapper();
		}
	}

	@Override
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException,
			IOException {
		String id = null;
		
		String requestDataString = null;
		try {
			requestDataString = getRequestDataString(request);
			
			logger.debug("{}的原始请求是：\n{}", request.getRemoteAddr(), requestDataString);
		} catch (ServiceException e) {
			logger.error("获取请求信息出错。", e);
			try {
				renderJson(request, response, jsonMapper.toJson(ResponseBean.create(id, ErrorBean.create(e))));
			} catch (ServiceException e1) {
				logger.error("向请求客户端发送错误信息失败", e1);
			} catch (JSONException e2) {
				logger.error(e2.getMessage(), e2);
			}
		}
		
		if (StringUtils.isNotBlank(requestDataString)) {
			try {
				RequestBean requestBean = jsonMapper.fromJson(requestDataString, RequestBean.class);
				requestBean.setServletRequest(request);
				id = requestBean.getId();
				String result = jsonMapper.toJson(OperatorFactory.getOperator(requestBean).operate(requestBean));
				
				logger.debug("系统给{}的响应：\n{}", request.getRemoteAddr(), result);
				
				renderJson(request, response, result);
			} catch (ServiceException e) {
				logger.error("业务操作失败。", e);
				try {
					renderJson(request, response, jsonMapper.toJson(ResponseBean.create(id, ErrorBean.create(e))));
				} catch (ServiceException e1) {
					logger.error("向请求客户端发送错误信息失败", e1);
				} catch (JSONException e2) {
					logger.error(e2.getMessage(), e2);
				}
			} catch (JSONException e) {
				logger.error("业务操作失败。", e);
				try {
					renderJson(request, response, jsonMapper.toJson(ResponseBean.create(id, ErrorBean.create(ServiceException.create(-32700, "服务器收到不合法的JSON格式.", e)))));
				} catch (ServiceException e1) {
					logger.error("向请求客户端发送错误信息失败", e1);
				} catch (JSONException e2) {
					logger.error(e2.getMessage(), e2);
				}
			}
		} else {
			renderJson(request, response, "客户端发送的请求信息为空");
		}
	}
	
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException,
			IOException {
		doPost(request, response);
	}
	
	private String getRequestDataString(HttpServletRequest request) throws ServiceException {
		byte[] requestDataBytes = null;
		if (request.getContentLength() > 0) {
			try {
				requestDataBytes = new byte[request.getContentLength()];
			} catch (Throwable t) {
				logger.error("获取原始请求出错。", t);
				throw ServiceException.create(0x00, "请求格式非法。", null);
			}
			ServletInputStream is = null;
			try {
				is = request.getInputStream();
				int off = 0, lastReadSum = 1;
				while (true) {
					lastReadSum = is.read(requestDataBytes, off, 2048);
					off += lastReadSum;
					if (lastReadSum < 0) break;
				}
			} catch (IOException e) {
				logger.error("获取请求原始数据出错。", e);
				throw ServiceException.create(0x00, "获取请求原始数据出错。", null);
			} finally {
				if (null != is) {
					try {
						is.close();
					} catch (IOException e) {
						logger.error("关闭从request中获取到的输入流失败", e);
					}
				}
			}
		} else {
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			ServletInputStream is = null;
			try {
				is = request.getInputStream();
				int c;
				while ((c = is.read()) != -1) {
					os.write(c);
				}
				requestDataBytes = os.toByteArray();
			} catch (IOException e) {
				logger.error("获取请求原始数据出错。", e);
				throw ServiceException.create(0x00, "获取请求原始数据出错。", null);
			} finally {
				if (null != is) {
					try {
						is.close();
					} catch (IOException e) {
						logger.error("关闭从request中获取到的输入流失败", e);
					}
				}
			}
		}
		
		return getRequestString(request, requestDataBytes);
	}
	
	private String getRequestString(HttpServletRequest request, byte[] requestDataBytes) throws ServiceException {
		String result = null;
		try {
			result = new String(requestDataBytes, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			logger.error("请求编码不是UTF-8。", e);
			throw ServiceException.create(0x00, "请求编码不是UTF-8。", null);
		}
		
		return result;
	}
	
	/**
	 * 
	 * @Title: renderJson
	 * @Description: 向请求客户端使用UTF-8编码返回json结果
	 * @param request
	 * @param response
	 * @param text 
	 */
	private void renderJson(HttpServletRequest request, HttpServletResponse response, String text) {
		render(request, response, CONTENT_TYPE, text);
	}
	
	/**
	 * 
	 * @Title: render
	 * @Description: 向请求客户端返回指定的结果
	 * @param request
	 * @param response
	 * @param contentType
	 * @param responseStr
	 */
	private void render(HttpServletRequest request, HttpServletResponse response, String contentType, String responseStr) {
		response.setContentType(contentType);
		response.setContentLength(responseStr.getBytes(Charset.forName("UTF-8")).length);
		response.setCharacterEncoding("UTF-8");
		try {
			response.getWriter().write(responseStr);
			response.getWriter().flush();
			response.getWriter().close();
		} catch (IOException e) {
			logger.error("写入response失败，不再写入任何字符串。", e);
		} catch (ServiceException e) {
			logger.error(e.getMessage(), e);
		}
	}
}
