package com.xdja.platform.remoting.protocol;

import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicInteger;

import com.xdja.platform.remoting.serialize.Serialization;
import com.xdja.platform.remoting.serialize.SerializationFacade;
import com.xdja.platform.remoting.serialize.exception.SerializeException;

/**
 * 传输协议数据定义
 * +--------+--------+--------+---------------+-------+
 * | length | oneway | opaque | serialization |  body |
 * | 4个字节      |  1个字节   |  4个字节   |     1个字节               | 序列化后  |
 * +--------+--------+--------+---------------+-------+
 * @date 2014-12-18
 * @author 任瑞修
 * 
 */
public class ProtocolData {
	
	/**
	 * 报文头所占字节数oneway+opaque+serialization
	 *              1       4         1
	 */
	private static final int HEAD_LENGTH = 6;
	
	private static AtomicInteger requestId = new AtomicInteger(1);
	
	/**
	 * 
	 * 创建请求
	 * @param serialization 序列化方式
	 * @param bodyData
	 */
	public static ProtocolData createRequest(Serialization serialization, Object bodyData) throws SerializeException {
		ProtocolData request = new ProtocolData();
		
		request.opaque = requestId.getAndIncrement();
		request.serialization = serialization;
		request.body = SerializationFacade.serialize(serialization, bodyData);
		
		return request;
	}

	/**
	 * 
	 * 创建响应
	 * @param request    请求对象
	 * @param bodyData    业务内容
	 */
	public static ProtocolData createResponse(ProtocolData request, Object bodyData) throws SerializeException {
		ProtocolData response = new ProtocolData();
		
		response.opaque = request.opaque;
		response.serialization = request.serialization;
		response.body = SerializationFacade.serialize(request.serialization, bodyData);
		
		return response;
	}
	
	/**
	 * 
	 * 解码
	 * @param byteBuffer
	 * @return
	 */
	public static ProtocolData decode(ByteBuffer byteBuffer){
		ProtocolData data = new ProtocolData();
		
		int length = byteBuffer.getInt();
		
		data.oneway = byteBuffer.get() == 1;

		data.opaque = byteBuffer.getInt();
		
		data.serialization = Serialization.valueOf(byteBuffer.get());
		
        int contentLength = length - HEAD_LENGTH;
        
        byte[] content = null;
        if (contentLength > 0) {
        	content = new byte[contentLength];
            byteBuffer.get(content);
        }

        data.body = content;

        return data;
	}

	/**
	 * 
	 * 解码
	 * @param dataArray
	 * @return
	 */
	public static ProtocolData decode(byte[] dataArray){
		 ByteBuffer byteBuffer = ByteBuffer.wrap(dataArray);
		 return decode(byteBuffer);
	}
	
	/**
	 * 用于匹配请求和响应，由通信框架生成和维护
	 */
	private int opaque;
	/**
	 * true:单程请求，不需要服务端回复；false:需要服务端回复
	 */
	private boolean oneway = false;
	/**
	 * 序列化方式
	 */
	private Serialization serialization;
	/**
	 * 业务内容
	 */
	private byte[] body;

	private ProtocolData(){}

	/**
	 * @return 返回{@linkplain #opaque}
	 */
	public int getOpaque() {
		return opaque;
	}

	/**
	 * @param opaque 要设置的 opaque
	 */
	public void setOpaque(int opaque) {
		this.opaque = opaque;
	}

	/**
	 * @return 返回{@linkplain #oneway}
	 */
	public boolean isOneway() {
		return oneway;
	}

	/**
	 * 
	 * 获取反序列化后的内容
	 * @param clazz
	 * @return
	 * @throws SerializeException
	 */
	public <T> T getBody(Class<T> clazz) throws SerializeException {
		return SerializationFacade.deserialize(serialization, body, clazz);
	}

	/**
	 * 
	 * 设置请求为单程请求（不需要服务端返回响应）
	 */
	public void makeOneway(){
		this.oneway = true;
	}

	/**
	 * 
	 * 编码
	 * @return
	 */
	public ByteBuffer encode(){
		int length = HEAD_LENGTH;

        if (null != body) {
        	length += body.length;
        }

        ByteBuffer result = ByteBuffer.allocate(4 + length);

        result.putInt(length);
        
        Integer onewayValue = isOneway() ? 1 : 0;
        result.put(onewayValue.byteValue());
        
        result.putInt(opaque);
        
        result.put((byte)serialization.ordinal());

        if (null != body) {
        	result.put(body);
        }

        result.flip();

        return result;
	}

}
