package com.xdja.platform.service.rmi.core;

import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.util.HashMap;

/**
 * RMI客户端
 * @author 武宗品
 * @时间 2013年10月23日
 */
public class RmiClient<RMIInterface extends IRmiCommonRemote> {

	/**
	 * 已有的客户端
	 */
	@SuppressWarnings("rawtypes")
	private static HashMap<String, RmiClient> RMI_CLIENT = new HashMap<String, RmiClient>();
	
	/**
	 * RMI服务的地址，如果是本机，则为null
	 */
	private final String ip;
	
	/**
	 * RMI服务的端口
	 */
	private final int port;
	
	/**
	 * RMI服务的名称
	 */
	private final String name;
	
	/**
	 * 绑定的远程方法
	 */
	private final RMIInterface remote;
	
	/**
	 * 远程接口的类名
	 */
	private final String remoteClazz;

	private RmiClient(String ip, int port, String name, RMIInterface remote, String remoteClazz) {
		super();
		this.ip = ip;
		this.port = port;
		this.name = name;
		this.remote = remote;
		this.remoteClazz = remoteClazz;
	}
	
	/**
	 * 根据RMI的服务地址，返回RMI客户端。如果出错，则返回null。
	 * @param ip RMI服务的地址，如果是本机，传入null即可。
	 * @param port RMI服务的端口。
	 * @param name RMI服务的名称。
	 * @param isUseNewClient 是否使用新的客户端。
	 */
	@SuppressWarnings("unchecked")
	public static <RMIInterface extends IRmiCommonRemote> RmiClient<RMIInterface> getRMIClient(String ip, int port, 
			String name, Class<RMIInterface> clazz, boolean isUseNewClient) {
		String rmiClientKey = ip + port + name;
		RmiClient<RMIInterface> rmiClient = RMI_CLIENT.get(rmiClientKey);
		if(rmiClient == null || isUseNewClient) {
			RMIInterface remote = null;
			try {
				Registry registry = null;
				if(ip == null) {
					registry = LocateRegistry.getRegistry(port);
				} else {
					registry = LocateRegistry.getRegistry(ip, port);
				}
				remote = (RMIInterface)registry.lookup(name);
				if(remote.getParentInterfaceClazz().indexOf(clazz.getName()) < 0) {
					return null;
				}
			} catch (RemoteException e) {
				return null;
			} catch (NotBoundException e) {
				return null;
			} catch (ClassCastException e) {
				return null;
			}
			rmiClient = new RmiClient<RMIInterface>(ip, port, name, remote, clazz.getName());
			RMI_CLIENT.put(rmiClientKey, rmiClient);
		} else {
			if(!rmiClient.getRemoteClazz().equals(clazz.getName())) {
				return null;
			}
		}
		return rmiClient;
	}

	public RMIInterface getRemote() {
		return remote;
	}

	public String getIp() {
		return ip;
	}

	public int getPort() {
		return port;
	}

	public String getName() {
		return name;
	}

	public String getRemoteClazz() {
		return remoteClazz;
	}
}
